/***************************************************************************
                         pov31file.cpp  -  description
                            -------------------
   begin                : Fri May 17 2002
   copyright            : (C) 2000 by Gran Gustafsson
   email                : gorgus@algonet.se
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "pov31file.h"
#include <vector>
#include <iostream.h>

extern "C"
{
	FilePlugin * i3d_createInstance()
	{
		return new POV31File();
	}
}


/************************************************************************\
*
* Init plugin object.
*
\************************************************************************/
POV31File::POV31File(void) : FilePlugin("POV-Ray 3.1", "pov")
{
	setImportable(false);
	setExportable(true);
}

/************************************************************************\
*
* Deinit plugin object.
*
\************************************************************************/
POV31File::~POV31File(void)
{
}

/************************************************************************\
*
* Import mesh data. (Not implemented for this plugin)
*
\************************************************************************/
void POV31File::importData(ifstream &in)
{
	// No implementation
}

/************************************************************************\
*
* Export all mesh data to povray file.
*
\************************************************************************/
void POV31File::exportData(ofstream &out)
{
	vector<Selectable *> *list;
	vector<int> *ilist;
	FaceList *flist;
	Face *f;
	Mesh *obj;
	int n, m, size;
	bool dotris;

	// Write all materials as texture declarations.
	writeMat31(out);
	
	// Get a list of all meshes to export.
	list = getMeshes();
	size = (int)list->size();

	// Loop through all meshes and export them.
	for(n = 0; n < size; n++)
	{
		obj = (Mesh *)(*list)[n];
		// Make sure the mesh is triangulated.
		// TODO: Work on a copy of the mesh or at least make it undoable.
		obj->triangulate();
		// Write the mesh to povray file.
		writePOV31((Mesh *)(*list)[n], out);
	}
}

/************************************************************************\
*
* Write all materials to the export file.
*
\************************************************************************/
void POV31File::writeMat31(ofstream &out)
{
	vector<TextureMaterial *> *texlist;
	TextureMaterial *tex;
	Color col;
	double alpha;
	int n;

	// Get list of materials from the material editor.
	texlist = getMaterials();
	// Loop through each material and export it.
	for(n = 0; n < (int)texlist->size(); n++)
	{
		tex = (TextureMaterial *)(*texlist)[n];
		out << "#declare " << tex->getName() << " =" << endl;
		out << "\ttexture {" << endl;
		out << "\t\t" << " pigment {" << endl;
		// Use ambient color as color of povray mesh.
		// TODO: Figure out how to convert I3D materials to povray.
		col = tex->cAmbient;
		// Add alpha as transmit component of color in povray.
		if(tex->alpha > 255)
			alpha = 255.0;
		else
			alpha = (double)tex->alpha;
		alpha = 1.0 - alpha / 255.0;
		out << "\t\t\tcolor rgbt<" << (col.r / 255.0) << "," << (col.g / 255.0) << "," << (col.b / 255.0) << "," << alpha << ">" << endl;
		out << "\t\t}" << endl;
		out << "\t}" << endl;
	}
}

/************************************************************************\
*
* Write a single mesh to the export file.
*
\************************************************************************/
void POV31File::writePOV31(Object *o, ofstream &out)
{
	Vector4 vec;
	TextureMaterial *tex;
	vector<int> *ilist;
	VertexList *vlist;
	FaceList *flist;
	Vertex *v;
	Face *f;
	const char *notes;
	int n;
	bool smooth;
	
	// Get notes for special smooth flag.
	notes = o->getNotes().c_str();
	// Get material of mesh.
	tex = o->getTextureMaterial();

	// Check for special smooth flag in notes.
	if(strstr(notes, "pov:smooth"))
		smooth = true;
	else
		smooth = false;
		
	out << "// " << o->getName() << endl;
	out << "mesh {" << endl;

	// Get mesh vertices and faces.
	vlist = o->getVerts();
	flist = o->getFaces();
	// Loop through all faces and export them.
	for(n = 0; n < (int)flist->size(); n++)
	{
		f = (Face *)(*flist)[n];
		// Get vertex index list for a face.
		ilist = f->getVerts();
		// Check if face is smooth or flat.
		if(smooth)
			out << "\tsmooth_triangle {" << endl;
		else
			out << "\ttriangle {" << endl;
		// Export first vertex of face.
		v = (Vertex *)(*vlist)[(*ilist)[0]];
		vec = v->getPosition();
		out << "\t\t<" << vec.x << ", " << vec.y << ", " << -vec.z << ">,";
		if(smooth)
		{
			vec = v->getNormal();
			out << " <" << vec.x << ", " << vec.y << ", " << -vec.z << ">,";
		}
		out << endl;
		// Export second vertex of face.
		v = (Vertex *)(*vlist)[(*ilist)[1]];
		vec = v->getPosition();
		out << "\t\t<" << vec.x << ", " << vec.y << ", " << -vec.z << ">,";
		if(smooth)
		{
			vec = v->getNormal();
			out << " <" << vec.x << ", " << vec.y << ", " << -vec.z << ">,";
		}
		out << endl;
		// Export third vertex of face.
		v = (Vertex *)(*vlist)[(*ilist)[2]];
		vec = v->getPosition();
		out << "\t\t<" << vec.x << ", " << vec.y << ", " << -vec.z << ">";
		if(smooth)
		{
			vec = v->getNormal();
			out << ", <" << vec.x << ", " << vec.y << ", " << -vec.z << ">";
		}
		out << endl;

		// Reference a declared material if one is selected for this mesh.
		if(tex)
			out << "\t\ttexture { " << tex->getName() << " }" << endl;
				
		out << "\t}" << endl;
	}

	// Get transformation matrix for the mesh.
	Matrix44 pos;
	o->getCompleteMatrix(&pos);
	// Write transformation matrix as a povray matrix instruction.
	out << "\tmatrix <"
			<< pos.m[0][0] << ","
			<< pos.m[0][1] << "," 
			<< pos.m[0][2] << "," 
			<< endl << "\t        "
			<< pos.m[1][0] << ","
			<< pos.m[1][1] << "," 
			<< pos.m[1][2] << "," 
			<< endl << "\t        "
			<< pos.m[2][0] << ","
			<< pos.m[2][1] << "," 
			<< pos.m[2][2] << "," 
			<< endl << "\t        "
			<< pos.m[3][0] << ","
			<< pos.m[3][1] << "," 
			<< -pos.m[3][2] << ">" 
			<< endl;
	
	out << "}" << endl;
}
