/***************************************************************************
                         pov35file.cpp  -  description
                            -------------------
   begin                : Sun May 12 2002
   copyright            : (C) 2000 by Gran Gustafsson
   email                : gorgus@algonet.se
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "pov35file.h"
#include <vector>
#include <iostream.h>

extern "C"
{
	FilePlugin * i3d_createInstance()
	{
		return new POV35File();
	}
}


/************************************************************************\
*
* Init plugin object.
*
\************************************************************************/
POV35File::POV35File(void) : FilePlugin("POV-Ray 3.5", "pov")
{
	setImportable(false);
	setExportable(true);
}

/************************************************************************\
*
* Deinit plugin object.
*
\************************************************************************/
POV35File::~POV35File(void)
{
}

/************************************************************************\
*
* Import mesh data. (Not implemented for this plugin)
*
\************************************************************************/
void POV35File::importData(ifstream &in)
{
	// No implementation
}

/************************************************************************\
*
* Export all mesh data to povray file.
*
\************************************************************************/
void POV35File::exportData(ofstream &out)
{
	vector<Selectable *> *list;
	vector<int> *ilist;
	FaceList *flist;
	Face *f;
	Mesh *obj;
	int n, m, size;
	bool dotris;
	
	// Write material declarations.
	writeMat35(out);
	
	// Get a list of all meshes to export.
	list = getMeshes();
	size = (int)list->size();

	// Loop through all meshes and export them.
	for(n = 0; n < size; n++)
	{
		obj = (Mesh *)(*list)[n];
		// Make sure the mesh is triangulated.
		// TODO: Work on a copy of the mesh or at least make it undoable.
		obj->triangulate();
		// Write the mesh to povray file.
		writePOV35((Mesh *)(*list)[n], out);
	}
}

/************************************************************************\
*
* Write all materials to the export file.
*
\************************************************************************/
void POV35File::writeMat35(ofstream &out)
{
	vector<TextureMaterial *> *texlist;
	TextureMaterial *tex;
	Color col;
	double alpha;
	int n;

	// Get list of materials from the material editor.
	texlist = getMaterials();
	// Loop through each material and export it.
	for(n = 0; n < (int)texlist->size(); n++)
	{
		tex = (TextureMaterial *)(*texlist)[n];
		out << "#declare " << tex->getName() << " =" << endl;
		out << "\ttexture {" << endl;
		out << "\t\t" << (tex->texture ? "uv_mapping" : "") << " pigment {" << endl;
		// Check for texture image in material.
		if(tex->texture)
		{
			// Use texture image as povray texture.
			out << "\t\t\timage_map { png \"" << *(tex->texture->getFilename()) << "\" }" << endl;
		}
		else
		{
			// Use ambient color as color of povray mesh.
			// TODO: Figure out how to convert I3D materials to povray.
			col = tex->cAmbient;
			// Add alpha as transmit component of color in povray.
			if(tex->alpha > 255)
				alpha = 255.0;
			else
				alpha = (double)tex->alpha;
			alpha = 1.0 - alpha / 255.0;
			out << "\t\t\tcolor rgbt<" << (col.r / 255.0) << "," << (col.g / 255.0) << "," << (col.b / 255.0) << "," << alpha << ">" << endl;
		}
		out << "\t\t}" << endl;
		out << "\t}" << endl;
	}
}

/************************************************************************\
*
* Write a single mesh to the export file.
*
\************************************************************************/
void POV35File::writePOV35(Object *o, ofstream &out)
{
	Vector4 vec;
	vector<int> *ilist;
	VertexList *vlist;
	FaceList *flist;
	UVList *uvlist;
	TextureMaterial *tex;
	Color col;
	Vertex *v;
	Face *f;
	double alpha;
	const char *notes;
	int n;
	
	// Get notes for special smooth flag.
	notes = o->getNotes().c_str();
	// Get material of mesh.
	tex = o->getTextureMaterial();

	out << "// " << o->getName() << endl;
	out << "mesh2 {" << endl;

	// Export all vertices first.
	out << "\tvertex_vectors {" << endl;
	vlist = o->getVerts();
	out << "\t\t" << vlist->size() << "," << endl;
	for(n = 0; n < (int)vlist->size(); n++)
	{
		v = (Vertex *)(*vlist)[n];
		vec = v->getPosition();
		out << "\t\t<" << vec.x << ", " << vec.y << ", " << -vec.z << ">";
		if(n != (int)vlist->size() - 1)
			out << ",";
		out << endl;
	}
	out << "\t}" << endl;

	// Check for special smooth flag in mesh notes.
	if(strstr(notes, "pov:smooth"))
	{
		// Export normals for all vertices of smooth mesh.
		out << "\tnormal_vectors {" << endl;
		vlist = o->getVerts();
		out << "\t\t" << vlist->size() << "," << endl;
		for(n = 0; n < (int)vlist->size(); n++)
		{
			v = (Vertex *)(*vlist)[n];
			vec = v->getNormal();
			out << "\t\t<" << vec.x << ", " << vec.y << ", " << -vec.z << ">";
			if(n != (int)vlist->size() - 1)
				out << ",";
			out << endl;
		}
		out << "\t}" << endl;
	}
	
	// Check for valid material.
	if(tex)
	{
		// Check if material uses a texture image.
		if(tex->texture)
		{
			// Export UV coordinates for all vertices if a texture image is used.
			out << "\tuv_vectors {" << endl;
			uvlist = o->getUVs();
			out << "\t\t" << uvlist->size() << "," << endl;
			for(n = 0; n < (int)uvlist->size(); n++)
			{
				v = (Vertex *)(*uvlist)[n];
				vec = v->getPosition();
				out << "\t\t<" << vec.x << ", " << vec.y << ">";
				if(n != (int)uvlist->size() - 1)
					out << ",";
				out << endl;
			}
			out << "\t}" << endl;
		}
		
		// Export material references used in mesh.
		out << "\ttexture_list {" << endl;
		out << "\t\t1," << endl;
		out << "\t\ttexture { " << tex->getName() << " } " << endl;
		#if 0
		out << "\t\t\t" << (tex->texture ? "uv_mapping" : "") << " pigment {" << endl;
		if(tex->texture)
		{
			out << "\t\t\t\timage_map { png \"" << *(tex->texture->getFilename()) << "\" }" << endl;
		}
		else
		{
			col = tex->cAmbient;
			if(tex->alpha > 255)
				alpha = 255.0;
			else
				alpha = (double)tex->alpha;
			alpha = 1.0 - alpha / 255.0;
			out << "\t\t\t\tcolor rgbt<" << (col.r / 255.0) << "," << (col.g / 255.0) << "," << (col.b / 255.0) << "," << alpha << ">" << endl;
		}
		out << "\t\t\t}" << endl;
		out << "\t\t}" << endl;
		#endif
		out << "\t}" << endl;
	}

	// Export all faces of mesh as vertex references.
	out << "\tface_indices {" << endl;
	flist = o->getFaces();
	out << "\t\t" << flist->size() << "," << endl;
	for(n = 0; n < (int)flist->size(); n++)
	{
		f = (Face *)(*flist)[n];
		ilist = f->getVerts();
		out << "\t\t<"
				<< (*ilist)[0] << ", "
				<< (*ilist)[1] << ", "
				<< (*ilist)[2] << ">";
		if(tex)
			out << ", 0";
		if(n != (int)flist->size() - 1)
			out << ",";
		out << endl;
	}
	out << "\t}" << endl;
	
	// Check for valid material and texture image.
	if(tex && tex->texture)
	{
		// Export UV references for all faces.
		out << "\tuv_indices {" << endl;
		flist = o->getFaces();
		out << "\t\t" << flist->size() << "," << endl;
		for(n = 0; n < (int)flist->size(); n++)
		{
			f = (Face *)(*flist)[n];
			ilist = f->getUVs();
			out << "\t\t<"
					<< (*ilist)[0] << ", "
					<< (*ilist)[1] << ", "
					<< (*ilist)[2] << ">";
			if(n != (int)flist->size() - 1)
				out << ",";
			out << endl;
		}
		out << "\t}" << endl;
	}
		
	// Get transformation matrix for the mesh.
	Matrix44 pos;
	o->getCompleteMatrix(&pos);
	// Write transformation matrix as a povray matrix instruction.
	out << "\tmatrix <"
			<< pos.m[0][0] << ","
			<< pos.m[0][1] << "," 
			<< pos.m[0][2] << "," 
			<< endl << "\t        "
			<< pos.m[1][0] << ","
			<< pos.m[1][1] << "," 
			<< pos.m[1][2] << "," 
			<< endl << "\t        "
			<< pos.m[2][0] << ","
			<< pos.m[2][1] << "," 
			<< pos.m[2][2] << "," 
			<< endl << "\t        "
			<< pos.m[3][0] << ","
			<< pos.m[3][1] << "," 
			<< -pos.m[3][2] << ">" 
			<< endl;
	
	out << "}" << endl;
}
