/**
* Innovation3D Quake MDL Plugin
*   (c) 2000 Jon Anderson <janderson@onelink.com>
*   (c) 2001 Filip Van Raemdonck <mechanix@digibel.org>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License as
* published by the Free Software Foundation; either version 2 of
* the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
* 
* See the GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to: 
*    Free Software Foundation, Inc.
*    59 Temple Place - Suite 330
*    Boston, MA  02111-1307, USA.
*
*   $Id: qmdlfile.cc,v 1.2 2001/08/16 00:39:22 janders Exp $
**/

#include "qmdlfile.h"
#include <vector>
#include <iostream.h>
#include <basecontroller.h>

extern "C"
{
   FilePlugin * i3d_createInstance ()
   {
      return new QMDLFile ();
   }
}

QMDLFile::QMDLFile () : FilePlugin ( "QMDL", "mdl" )
{
   setImportable ( true );
   //setExportable (true);
   setBinary ( true );
}

QMDLFile::~QMDLFile () {}

void QMDLFile::importData ( ifstream &in )
{
   uvs.clear ();
   verts.clear ();
   triangles.clear ();

   readHeader ( in );
   /* Quake skins start right after the header ... */
   readSkins ( sizeof ( mdl_t ), in );
   /* ... are followed by the texture coordinates ... */
   readTexCoords ( offsetTexCoords, in );
   /* ... the triangle information ... */
   readTriangles ( offsetTriangles, in );
   /* ... and finally the frame animations. */
   readFrames ( offsetFrames, in );

   createObject ();
}

void QMDLFile::createObject ()
{
   int i, j;
   Controller* c;
   Face* f;
   Mesh* m;
   Vector4 pos;
   Vertex* v;

   m = new Mesh();
   /* create the vertices for the first frame */

   for ( i = 0; i < header.numverts; i++ )
   {
      pos = verts[ 0 ][ i ];
      v = m->createVertex ( pos.x, pos.y, pos.z );
      /* save a keyframe every 2 frames for each MDL frame */
      v->setAnimatable ( true );
      c = v->getController ();
      c->saveTime ( 1 );

      for ( j = 1; j < header.numframes; j++ )
      {
         pos = verts[ j ][ i ];
         v->setPosition ( pos.x, pos.y, pos.z );
         c->saveTime ( 1 + j );
      }
   }

   //BaseController::getInstance()->setMaxTime( ( header.numframes ) );
   /* create the triangles for the mesh */
   for ( i = 0; i < header.numtris; i++ )
   {
      f = m->createFace ( triangles[ i ].vertindex[ 0 ], triangles[ i ].vertindex[ 1 ], triangles[ i ].vertindex[ 2 ] );
      /*f->setUVCoord (0, uvs[triangles[i].textureIndices[0]] );
      f->setUVCoord (1, uvs[triangles[i].textureIndices[1]] );
      f->setUVCoord (2, uvs[triangles[i].textureIndices[2]] );*/
   }

   m->normalize ();
   addEntity ( m );
}

void QMDLFile::readHeader ( ifstream& in )
{
   in.read ( &header, sizeof ( mdl_t ) );
   assert ( header.ident == IDPOLYHEADER );
}

void QMDLFile::readFrames ( int offset, ifstream& in )
{
   int i, j;
   daliasframetype_t frametype;
   daliasgroup_t framegroup;
   daliasinterval_t frametime;

   in.seekg ( offset );

   for ( i = 0; i < header.numframes; i++ )
   {
      in.read ( &frametype, sizeof ( daliasframetype_t ) );

      if ( frametype.type == ALIAS_SINGLE )
      {
         readSimpleFrame ( in );
      }

      else
      {
         in.read ( &framegroup, sizeof ( daliasgroup_t ) );

         for ( j = 0; j < framegroup.numframes; j++ )
         {
            in.read ( &frametime, sizeof ( daliasinterval_t ) );
         }

         for ( j = 0; j < framegroup.numframes; j++ )
         {
            readSimpleFrame ( in );
         }
      }
   }
}

void QMDLFile::readSimpleFrame ( ifstream& in )
{
   int i, x, y, z;
   daliasframe_t frame;
   trivertx_t* fverts;
   vector< Vector4 > frame_verts ( header.numverts );
   Vector4 pos;

   in.read ( &frame, sizeof ( daliasframe_t ) );
   fverts = new trivertx_t[ header.numverts ];
   in.read ( fverts, sizeof ( trivertx_t ) * header.numverts );

   for ( i = 0; i < header.numverts; i++ )
   {
      x = fverts[ i ].v[ 0 ];
      y = fverts[ i ].v[ 1 ];
      z = fverts[ i ].v[ 2 ];
      //swap the y and z axis.
      pos.x = ( ( float ) x ) * header.scale[ 0 ] + header.scale_origin[ 0 ];
      pos.z = ( ( float ) y ) * header.scale[ 1 ] + header.scale_origin[ 1 ];
      pos.y = ( ( float ) z ) * header.scale[ 2 ] + header.scale_origin[ 2 ];
      frame_verts[ i ] = pos;
   }

   verts.push_back( frame_verts );
}

void QMDLFile::readTexCoords ( int offset, ifstream& in )
{
   int i;
   stvert_t uv;
   Vector4 pos;

   offsetTriangles = offset;
   in.seekg ( offset );

   for ( i = 0; i < header.numverts; i++ )
   {
      in.read ( &uv, sizeof ( uv ) );
      pos.x = ( ( float ) uv.s ) / header.skinwidth;
      pos.y = 1 - ( ( float ) uv.t ) / header.skinheight;
      uvs.push_back ( pos );
   }

   offsetTriangles += sizeof ( stvert_t ) * header.numverts;
}

void QMDLFile::readTriangles ( int offset, ifstream& in )
{
   int i;
   dtriangle_t triangle;

   offsetFrames = offset;
   in.seekg ( offset );

   for ( i = 0; i < header.numtris; i++ )
   {
      in.read ( &triangle, sizeof ( dtriangle_t ) );
      triangles.push_back ( triangle );
   }

   offsetFrames += sizeof ( dtriangle_t ) * header.numtris;
}

void QMDLFile::readSkins ( int offset, ifstream& in )
{
   int skinsize, i;
   daliasskintype_t skintype;
   daliasskingroup_t skingroup;
   /* daliasskininterval_t* skininterval; */

   skinsize = header.skinwidth * header.skinheight;
   /* The offsets for the different sections are not known beforehand.
      Each section must be gone through to know where the next one starts. */
   offsetTexCoords = offset;
   in.seekg ( offset );
   /* Iterate through all skins in the mdl (usually 1). */

   for ( i = 0; i < header.numskins; i++ )
   {
      /* Each skin may be a singular skin, or a skingroup. */
      in.read ( &skintype, sizeof ( daliasskintype_t ) );
      offsetTexCoords += sizeof ( daliasskintype_t );

      if ( skintype.type == ALIAS_SKIN_SINGLE )
      {
         /* FIXME: read the skin here */
         offsetTexCoords += skinsize;
      }

      else
      {
         in.read ( &skingroup, sizeof ( daliasskingroup_t ) );
         offsetTexCoords += sizeof ( daliasskingroup_t );
         offsetTexCoords += sizeof ( daliasskininterval_t ) * skingroup.numskins;
         /* FIXME: read the skin here */
         offsetTexCoords += skinsize * skingroup.numskins;
      }

      /* Make sure we're at the correct location if another skin must be read. */
      in.seekg ( offsetTexCoords );
   }
}

void QMDLFile::exportData ( ofstream& ) {}

void QMDLFile::writeHeader ( ofstream& ) {}
