/*=========================================================================

  Program:   Insight Segmentation & Registration Toolkit
  Module:    $RCSfile: itkPowellOptimizerTest.cxx,v $
  Language:  C++
  Date:      $Date: 2008-04-30 03:30:43 $
  Version:   $Revision: 1.4 $

  Copyright (c) Insight Software Consortium. All rights reserved.
  See ITKCopyright.txt or http://www.itk.org/HTML/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even 
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
     PURPOSE.  See the above copyright notices for more information.

=========================================================================*/
#if defined(_MSC_VER)
#pragma warning ( disable : 4786 )
#endif

#include <itkPowellOptimizer.h>
#include <vnl/vnl_math.h>

int POWELL_CALLS_TO_GET_VALUE = 0;

/** 
 *  The objectif function is the quadratic form:
 *
 *  1/2 x^T A x - b^T x
 *
 *  Where A is a matrix and b is a vector
 *  The system in this example is:
 *
 *     | 3  2 ||x|   | 2|   |0|
 *     | 2  6 ||y| + |-8| = |0|
 *
 *
 *   the solution is the vector | 2 -2 |
 *
 */ 
class PowellBoundedCostFunction : public itk::SingleValuedCostFunction 
{
public:

  typedef PowellBoundedCostFunction                  Self;
  typedef itk::SingleValuedCostFunction   Superclass;
  typedef itk::SmartPointer<Self>         Pointer;
  typedef itk::SmartPointer<const Self>   ConstPointer;
  itkNewMacro( Self );
  itkTypeMacro( PowellBoundedCostFunction, SingleValuedCostFunction );

  enum { SpaceDimension=2 };
  
  typedef Superclass::ParametersType      ParametersType;
  typedef Superclass::DerivativeType      DerivativeType;
  typedef Superclass::MeasureType         MeasureType ;

  PowellBoundedCostFunction()
  {
  }


  void GetDerivative( const ParametersType & ,
                      DerivativeType &  ) const
  {
  }

  MeasureType  GetValue( const ParametersType & parameters ) const
  { 
    ++POWELL_CALLS_TO_GET_VALUE;

    double x = parameters[0];
    double y = parameters[1];

    std::cout << "      GetValue( " ;
    std::cout << x << " ";
    std::cout << y << ") = ";

    MeasureType measure = 0.5*(3*x*x+4*x*y+6*y*y) - 2*x + 8*y;

    std::cout << measure << std::endl; 

    return measure;

  }

  unsigned int GetNumberOfParameters(void) const
    {
    return SpaceDimension;
    }


private:


};



int itkPowellOptimizerTest(int, char* [] ) 
{
  std::cout << "Powell Optimizer Test ";
  std::cout << std::endl << std::endl;

  typedef  itk::PowellOptimizer  OptimizerType;

  typedef OptimizerType::ScalesType        ScalesType;
    
  // Declaration of a itkOptimizer
  OptimizerType::Pointer  itkOptimizer = OptimizerType::New();


  // Declaration of the CostFunction 
  PowellBoundedCostFunction::Pointer costFunction = PowellBoundedCostFunction::New();


  itkOptimizer->SetCostFunction( costFunction.GetPointer() );

  
  typedef PowellBoundedCostFunction::ParametersType    ParametersType;

  const unsigned int spaceDimension = 
                      costFunction->GetNumberOfParameters();

  // We start not so far from  | 2 -2 |
  ParametersType  initialPosition( spaceDimension );

  initialPosition[0] =  100;
  initialPosition[1] = -100;

  itkOptimizer->SetMaximize(false);
  itkOptimizer->SetStepLength( 10 );
  itkOptimizer->SetStepTolerance( 0.01 );
  itkOptimizer->SetValueTolerance( 0.1 );
  itkOptimizer->SetMaximumIteration( 100 );

  itkOptimizer->SetInitialPosition( initialPosition );

  try 
    {
    itkOptimizer->StartOptimization();
    }
  catch( itk::ExceptionObject & e )
    {
    std::cout << "Exception thrown ! " << std::endl;
    std::cout << "An error ocurred during Optimization" << std::endl;
    std::cout << "Location    = " << e.GetLocation()    << std::endl;
    std::cout << "Description = " << e.GetDescription() << std::endl;
    return EXIT_FAILURE;
    }

  ParametersType finalPosition = itkOptimizer->GetCurrentPosition();
  std::cout << "Solution        = (";
  std::cout << finalPosition[0] << "," ;
  std::cout << finalPosition[1] << ")" << std::endl;  

  //
  // check results to see if it is within range
  //
  bool pass = true;
  double trueParameters[2] = { 2, -2 };
  for( unsigned int j = 0; j < 2; j++ )
    {
    if( vnl_math_abs( finalPosition[j] - trueParameters[j] ) > 0.01 )
      pass = false;
    }

  // Exercise various member functions.
  std::cout << "Maximize: " << itkOptimizer->GetMaximize() << std::endl;
  std::cout << "StepLength: " << itkOptimizer->GetStepLength();
  std::cout << std::endl;
  std::cout << "CurrentIteration: " << itkOptimizer->GetCurrentIteration();
  std::cout << std::endl;

  itkOptimizer->Print( std::cout );

  std::cout << "Calls to GetValue = " << POWELL_CALLS_TO_GET_VALUE << std::endl;

  if( !pass )
    {
    std::cout << "Test failed." << std::endl;
    return EXIT_FAILURE;
    }

  std::cout << "Test passed." << std::endl;
  return EXIT_SUCCESS;


}



