/*
 * ion/de/init.c
 *
 * Copyright (c) Tuomo Valkonen 1999-2004. 
 *
 * Ion is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 */

#include <string.h>

#include <ioncore/common.h>
#include <ioncore/global.h>
#include <ioncore/rootwin.h>
#include <ioncore/extl.h>
#include <ioncore/extlconv.h>
#include <ioncore/ioncore.h>
#include <ioncore/readconfig.h>

#include "brush.h"
#include "font.h"
#include "colour.h"
#include "misc.h"


/*{{{ Borders */


#define CF_BORDER_VAL_SANITY_CHECK 16

static void get_border_val(uint *val, ExtlTab tab, const char *what)
{
    int g;
    
    if(extl_table_gets_i(tab, what, &g)){
        if(g>CF_BORDER_VAL_SANITY_CHECK || g<0)
            warn("Border attribute %s sanity check failed.", what);
        else
            *val=g;
    }
}


static void get_border_style(uint *ret, ExtlTab tab)
{
    char *style=NULL;
    
    if(!extl_table_gets_s(tab, "border_style", &style))
        return;
    
    if(strcmp(style, "inlaid")==0)
        *ret=DEBORDER_INLAID;
    else if(strcmp(style, "elevated")==0)
        *ret=DEBORDER_ELEVATED;
    else if(strcmp(style, "groove")==0)
        *ret=DEBORDER_GROOVE;
    else if(strcmp(style, "ridge")==0)
        *ret=DEBORDER_RIDGE;
    else
        warn("Unknown border style \"%s\".", style);
    
    free(style);
}


static void get_border(DEBorder *border, ExtlTab tab)
{
    get_border_val(&(border->sh), tab, "shadow_pixels");
    get_border_val(&(border->hl), tab, "highlight_pixels");
    get_border_val(&(border->pad), tab, "padding_pixels");
    get_border_style(&(border->style), tab);
}


/*}}}*/


/*{{{ Colours */


static bool get_colour(WRootWin *rootwin, DEColour *ret, 
                       ExtlTab tab, DEStyle *based_on,
                       const char *what, DEColour substitute)
{
    char *name=NULL;
    bool ok=FALSE;
    
    if(extl_table_gets_s(tab, what, &name)){
        ok=de_alloc_colour(rootwin, ret, name);
    
        if(!ok)
            warn("Unable to allocate colour \"%s\".", name);

        free(name);
    }
    
    if(!ok && based_on!=NULL){
        return get_colour(rootwin, ret, based_on->data_table,
                          based_on->based_on, what, substitute);
    }else if(!ok){
        return de_duplicate_colour(rootwin, substitute, ret);
    }
    
    return ok;
}


static void get_colour_group(WRootWin *rootwin, DEColourGroup *cg, 
                             ExtlTab tab, DEStyle *based_on)
{
    get_colour(rootwin, &(cg->hl), tab, based_on, "highlight_colour",
               DE_WHITE(rootwin));
    get_colour(rootwin, &(cg->sh), tab, based_on, "shadow_colour",
               DE_WHITE(rootwin));
    get_colour(rootwin, &(cg->bg), tab, based_on, "background_colour",
               DE_BLACK(rootwin));
    get_colour(rootwin, &(cg->fg), tab, based_on, "foreground_colour",
               DE_WHITE(rootwin));
    get_colour(rootwin, &(cg->pad), tab, based_on, "padding_colour", cg->bg);
}


static void get_extra_cgrps(WRootWin *rootwin, DEStyle *style, ExtlTab tab)
{
    
    uint i=0, nfailed=0, n=extl_table_get_n(tab);
    char *name;
    ExtlTab sub;
    
    if(n==0)
        return;
    
    style->extra_cgrps=ALLOC_N(DEColourGroup, n);
    
    if(style->extra_cgrps==NULL){
        warn_err();
        return;
    }

    for(i=0; i<n-nfailed; i++){
        if(!extl_table_geti_t(tab, i+1, &sub))
            goto err;
        if(!extl_table_gets_s(sub, "substyle_pattern", &name)){
            extl_unref_table(sub);
            goto err;
        }
        
        /*de_init_colour_group(rootwin, style->extra_cgrps+i-nfailed);*/
        style->extra_cgrps[i-nfailed].spec=name;
        get_colour_group(rootwin, style->extra_cgrps+i-nfailed, sub, style);
        
        extl_unref_table(sub);
        continue;
        
    err:
        warn("Corrupt substyle table %d.", i);
        nfailed++;
    }
    
    if(n-nfailed==0){
        free(style->extra_cgrps);
        style->extra_cgrps=NULL;
    }
    
    style->n_extra_cgrps=n-nfailed;
}


/*}}}*/


/*{{{ Misc. */


static void get_text_align(int *alignret, ExtlTab tab)
{
    char *align=NULL;
    
    if(!extl_table_gets_s(tab, "text_align", &align))
        return;
    
    if(strcmp(align, "left")==0)
        *alignret=DEALIGN_LEFT;
    else if(strcmp(align, "right")==0)
        *alignret=DEALIGN_RIGHT;
    else if(strcmp(align, "center")==0)
        *alignret=DEALIGN_CENTER;
    else
        warn("Unknown text alignment \"%s\".", align);
    
    free(align);
}


static void get_transparent_background(uint *mode, ExtlTab tab)
{
    if(extl_table_is_bool_set(tab, "transparent_background"))
        *mode=GR_TRANSPARENCY_YES;
    else
        *mode=GR_TRANSPARENCY_NO;
}


/*}}}*/


/*{{{ de_defstyle */


/*EXTL_DOC
 * Define a style for the root window \var{rootwin}. 
 */
EXTL_EXPORT
bool de_defstyle_rootwin(WRootWin *rootwin, const char *name, ExtlTab tab)
{
    DEStyle *style;
    char *fnt;
    uint n;
    DEStyle *based_on=NULL;
    char *based_on_name=NULL;

    if(name==NULL)
        return FALSE;
    
    style=de_create_style(rootwin, name);

    if(style==NULL)
        return FALSE;

    style->data_table=extl_ref_table(tab);

    if(extl_table_gets_s(tab, "based_on", &based_on_name)){
        based_on=de_get_style(rootwin, based_on_name);
        if(based_on==style){
            warn("'based_on' for %s points back to the style itself.", name);
        }else if(based_on==NULL){
            warn("Unknown base style \"%s\".", based_on);
        }else{
            style->based_on=based_on;
            based_on->usecount++;
            /* Copy simple parameters */
            style->border=based_on->border;
            style->transparency_mode=based_on->transparency_mode;
            style->textalign=based_on->textalign;
            style->spacing=based_on->spacing;
        }
        free(based_on_name);
    }
    
    get_border(&(style->border), tab);
    get_border_val(&(style->spacing), tab, "spacing");

    get_text_align(&(style->textalign), tab);

    get_transparent_background(&(style->transparency_mode), tab);
    
    if(extl_table_gets_s(tab, "font", &fnt)){
        de_load_font_for_style(style, fnt);
        free(fnt);
    }else{
        de_set_font_for_style(style, based_on->font);
    }
    
    if(style->font==NULL)
        de_load_font_for_style(style, CF_FALLBACK_FONT_NAME);
    
    style->cgrp_alloced=TRUE;
    get_colour_group(rootwin, &(style->cgrp), tab, based_on);
    get_extra_cgrps(rootwin, style, tab);
    
    return TRUE;
}


/*EXTL_DOC
 * Define a style.
 */
EXTL_EXPORT
bool de_defstyle(const char *name, ExtlTab tab)
{
    bool ok=TRUE;
    WRootWin *rw;
    
    FOR_ALL_ROOTWINS(rw){
        if(!de_defstyle_rootwin(rw, name, tab))
            ok=FALSE;
    }
    
    return ok;
}


/*EXTL_DOC
 * Define a substyle.
 */
EXTL_EXPORT
ExtlTab de_substyle(const char *pattern, ExtlTab tab)
{
    extl_table_sets_s(tab, "substyle_pattern", pattern);
    return extl_ref_table(tab);
}


/*}}}*/


/*{{{ Module initialisation */


#include "../version.h"

char de_ion_api_version[]=ION_API_VERSION;


extern bool de_register_exports();
extern void de_unregister_exports();


bool de_init()
{
    WRootWin *rootwin;
    DEStyle *style;
    
    if(!de_register_exports())
        return FALSE;
    
    if(!gr_register_engine("de", (GrGetBrushFn*)&de_get_brush)){
        warn("DE module", "Failed to register the drawing engine");
        goto fail;
    }
    
    /* Create fallback brushes */
    FOR_ALL_ROOTWINS(rootwin){
        style=de_create_style(rootwin, "*");
        if(style==NULL){
            warn_obj("DE module", "Could not initialise fallback style for "
                     "root window %d.\n", rootwin->xscr);
        }else{
            style->is_fallback=TRUE;
            de_load_font_for_style(style, CF_FALLBACK_FONT_NAME);
        }
    }
    
    return TRUE;
    
fail:
    de_unregister_exports();
    return FALSE;
}


void de_deinit()
{
    gr_unregister_engine("de");
    de_unregister_exports();
    de_deinit_styles();
}


/*}}}*/

