/*
 * ion/ioncore/clientwin.c
 *
 * Copyright (c) Tuomo Valkonen 1999-2004. 
 *
 * Ion is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 */

#include <string.h>
#include <limits.h>

#include <libtu/objp.h>
#include <libtu/minmax.h>
#include "common.h"
#include "global.h"
#include "property.h"
#include "focus.h"
#include "sizehint.h"
#include "hooks.h"
#include "event.h"
#include "clientwin.h"
#include "colormap.h"
#include "resize.h"
#include "attach.h"
#include "regbind.h"
#include "names.h"
#include "stacking.h"
#include "saveload.h"
#include "manage.h"
#include "extl.h"
#include "extlconv.h"
#include "fullscreen.h"
#include "event.h"
#include "rootwin.h"
#include "activity.h"
#include "netwm.h"
#include "region-iter.h"
#include "xwindow.h"


static void set_clientwin_state(WClientWin *cwin, int state);
static bool send_clientmsg(Window win, Atom a, Time stmp);


WHooklist *clientwin_do_manage_alt=NULL;


#define LATEST_TRANSIENT(CWIN) REGION_LAST_MANAGED((CWIN)->transient_list)


/*{{{ Get properties */


void clientwin_get_protocols(WClientWin *cwin)
{
    Atom *protocols=NULL, *p;
    int n;
    
    cwin->flags&=~(CLIENTWIN_P_WM_DELETE|CLIENTWIN_P_WM_TAKE_FOCUS);
    
    if(!XGetWMProtocols(ioncore_g.dpy, cwin->win, &protocols, &n))
        return;
    
    for(p=protocols; n; n--, p++){
        if(*p==ioncore_g.atom_wm_delete)
            cwin->flags|=CLIENTWIN_P_WM_DELETE;
        else if(*p==ioncore_g.atom_wm_take_focus)
            cwin->flags|=CLIENTWIN_P_WM_TAKE_FOCUS;
    }
    
    if(protocols!=NULL)
        XFree((char*)protocols);
}


static bool get_winprop_fn_set=FALSE;
static ExtlFn get_winprop_fn;

/*EXTL_DOC
 * Set function used to look up winprops.
 */
EXTL_EXPORT
void ioncore_set_get_winprop_fn(ExtlFn fn)
{
    if(get_winprop_fn_set)
        extl_unref_fn(get_winprop_fn);
    get_winprop_fn=extl_ref_fn(fn);
    get_winprop_fn_set=TRUE;
}


static void clientwin_get_winprops(WClientWin *cwin)
{
    ExtlTab tab, tab2;
    int i1, i2;

    if(!get_winprop_fn_set)
        return;
    
    if(!extl_call(get_winprop_fn, "o", "t", cwin, &tab))
        return;
    
    cwin->proptab=tab;
    
    if(tab==extl_table_none())
        return;

    if(extl_table_is_bool_set(tab, "transparent"))
        cwin->flags|=CLIENTWIN_PROP_TRANSPARENT;

    if(extl_table_is_bool_set(tab, "acrobatic"))
        cwin->flags|=CLIENTWIN_PROP_ACROBATIC;
    
    if(extl_table_gets_t(tab, "max_size", &tab2)){
        if(extl_table_gets_i(tab2, "w", &i1) &&
           extl_table_gets_i(tab2, "h", &i2)){
            cwin->size_hints.max_width=i1;
            cwin->size_hints.max_height=i2;
            cwin->size_hints.flags|=PMaxSize;
            cwin->flags|=CLIENTWIN_PROP_MAXSIZE;
        }
        extl_unref_table(tab2);
    }

    if(extl_table_gets_t(tab, "min_size", &tab2)){
        if(extl_table_gets_i(tab2, "w", &i1) &&
           extl_table_gets_i(tab2, "h", &i2)){
            cwin->size_hints.min_width=i1;
            cwin->size_hints.min_height=i2;
            cwin->size_hints.flags|=PMinSize;
            cwin->flags|=CLIENTWIN_PROP_MINSIZE;
        }
        extl_unref_table(tab2);
    }

    if(extl_table_gets_t(tab, "aspect", &tab2)){
        if(extl_table_gets_i(tab2, "w", &i1) &&
           extl_table_gets_i(tab2, "h", &i2)){
            cwin->size_hints.min_aspect.x=i1;
            cwin->size_hints.max_aspect.x=i1;
            cwin->size_hints.min_aspect.y=i2;
            cwin->size_hints.max_aspect.y=i2;
            cwin->size_hints.flags|=PAspect;
            cwin->flags|=CLIENTWIN_PROP_ASPECT;
        }
        extl_unref_table(tab2);
    }
    
    if(extl_table_is_bool_set(tab, "ignore_resizeinc"))
        cwin->flags|=CLIENTWIN_PROP_IGNORE_RSZINC;

    if(extl_table_is_bool_set(tab, "ignore_cfgrq"))
        cwin->flags|=CLIENTWIN_PROP_IGNORE_CFGRQ;

    if(extl_table_is_bool_set(tab, "transients_at_top"))
        cwin->flags|=CLIENTWIN_TRANSIENTS_AT_TOP;
}


void clientwin_get_size_hints(WClientWin *cwin)
{
    XSizeHints tmp=cwin->size_hints;
    
    xwindow_get_sizehints(cwin->win, &(cwin->size_hints));
    
    if(cwin->flags&CLIENTWIN_PROP_MAXSIZE){
        cwin->size_hints.max_width=tmp.max_width;
        cwin->size_hints.max_height=tmp.max_height;
        cwin->size_hints.flags|=PMaxSize;
    }

    if(cwin->flags&CLIENTWIN_PROP_MINSIZE){
        cwin->size_hints.min_width=tmp.min_width;
        cwin->size_hints.min_height=tmp.min_height;
        cwin->size_hints.flags|=PMinSize;
    }
    
    if(cwin->flags&CLIENTWIN_PROP_ASPECT){
        cwin->size_hints.min_aspect=tmp.min_aspect;
        cwin->size_hints.max_aspect=tmp.max_aspect;
        cwin->size_hints.flags|=PAspect;
    }
    
    if(cwin->flags&CLIENTWIN_PROP_IGNORE_RSZINC)
        cwin->size_hints.flags&=~PResizeInc;
}


void clientwin_get_set_name(WClientWin *cwin)
{
    char **list=NULL;
    int n=0;
    
    if(ioncore_g.use_mb && ioncore_g.atom_net_wm_name!=0)
        list=xwindow_get_text_property(cwin->win, ioncore_g.atom_net_wm_name, NULL);

    if(list==NULL){
        list=xwindow_get_text_property(cwin->win, XA_WM_NAME, &n);
    }else{
        cwin->flags|=CLIENTWIN_USE_NET_WM_NAME;
    }

    if(list==NULL){
        /* Special condition kludge: property exists, but couldn't
         * be converted to a string list.
         */
        clientwin_set_name(cwin, (n==-1 ? "???" : NULL));
    }else{
        clientwin_set_name(cwin, *list);
        XFreeStringList(list);
    }
}


/* Some standard winprops */


bool clientwin_get_switchto(WClientWin *cwin)
{
    bool b;
    
    if(ioncore_g.opmode==IONCORE_OPMODE_INIT)
        return FALSE;
    
    if(extl_table_gets_b(cwin->proptab, "switchto", &b))
        return b;
    
    return ioncore_g.switchto_new;
}


int clientwin_get_transient_mode(WClientWin *cwin)
{
    char *s;
    int mode=TRANSIENT_MODE_NORMAL;
    
    if(extl_table_gets_s(cwin->proptab, "transient_mode", &s)){
        if(strcmp(s, "current")==0)
            mode=TRANSIENT_MODE_CURRENT;
        else if(strcmp(s, "off")==0)
            mode=TRANSIENT_MODE_OFF;
        free(s);
    }
    return mode;
}


/*}}}*/


/*{{{ Manage/create */


static void configure_cwin_bw(Window win, int bw)
{
    XWindowChanges wc;
    ulong wcmask=CWBorderWidth;
    
    wc.border_width=bw;
    XConfigureWindow(ioncore_g.dpy, win, wcmask, &wc);
}


static bool clientwin_init(WClientWin *cwin, WWindow *par, Window win,
                           XWindowAttributes *attr)
{
    WRectangle geom;
    char *name;

    cwin->flags=0;
    cwin->win=win;
    cwin->state=WithdrawnState;
    
    geom.x=attr->x;
    geom.y=attr->y;
    geom.w=attr->width;
    geom.h=attr->height;
    
    /* The idiot who invented special server-supported window borders that
     * are not accounted for in the window size should be "taken behind a
     * sauna".
     */
    cwin->orig_bw=attr->border_width;
    configure_cwin_bw(cwin->win, 0);
    if(cwin->orig_bw!=0 && cwin->size_hints.flags&PWinGravity){
        geom.x+=xgravity_deltax(cwin->size_hints.win_gravity, 
                               -cwin->orig_bw, -cwin->orig_bw);
        geom.y+=xgravity_deltay(cwin->size_hints.win_gravity, 
                               -cwin->orig_bw, -cwin->orig_bw);
    }

    cwin->last_fp.g=geom;
    cwin->last_fp.mode=REGION_FIT_EXACT;
    cwin->last_h_rq=geom.h;

    cwin->transient_for=None;
    cwin->transient_list=NULL;
    
    cwin->n_cmapwins=0;
    cwin->cmap=attr->colormap;
    cwin->cmaps=NULL;
    cwin->cmapwins=NULL;
    cwin->n_cmapwins=0;
    cwin->event_mask=IONCORE_EVENTMASK_CLIENTWIN;

    watch_init(&(cwin->fsinfo.last_mgr_watch));
    
    region_init(&(cwin->region), par, &(cwin->last_fp));
    
    clientwin_get_colormaps(cwin);
    clientwin_get_protocols(cwin);
    clientwin_get_set_name(cwin);
    clientwin_get_winprops(cwin);
    clientwin_get_size_hints(cwin);
    
    XSelectInput(ioncore_g.dpy, win, cwin->event_mask);

    XSaveContext(ioncore_g.dpy, win, ioncore_g.win_context, (XPointer)cwin);
    XAddToSaveSet(ioncore_g.dpy, win);

    return TRUE;
}


static WClientWin *create_clientwin(WWindow *par, Window win,
                                    XWindowAttributes *attr)
{
    CREATEOBJ_IMPL(WClientWin, clientwin, (p, par, win, attr));
}


static bool handle_target_props(WClientWin *cwin, const WManageParams *param)
{
    WRegion *r=NULL;
    char *target_name=NULL;
    
    if(extl_table_gets_s(cwin->proptab, "target", &target_name)){
        r=ioncore_lookup_region(target_name, NULL);
        
        free(target_name);
    
        if(r!=NULL){
            if(region_manage_clientwin(r, cwin, param, 
                                       MANAGE_REDIR_PREFER_NO))
                return TRUE;
        }
    }
    
    if(!extl_table_is_bool_set(cwin->proptab, "fullscreen"))
            return FALSE;
    
    return clientwin_enter_fullscreen(cwin, param->switchto);
}


WClientWin *clientwin_get_transient_for(WClientWin *cwin)
{
    Window tforwin;
    WClientWin *tfor=NULL;
    
    if(clientwin_get_transient_mode(cwin)!=TRANSIENT_MODE_NORMAL)
        return NULL;

    if(!XGetTransientForHint(ioncore_g.dpy, cwin->win, &tforwin))
        return NULL;
    
    if(tforwin==None)
        return NULL;
    
    tfor=XWINDOW_REGION_OF_T(tforwin, WClientWin);
    
    if(tfor==cwin){
        warn("The transient_for hint for \"%s\" points to itself.",
             region_name((WRegion*)cwin));
    }else if(tfor==NULL){
        if(xwindow_region_of(tforwin)!=NULL){
            warn("Client window \"%s\" has broken transient_for hint. "
                 "(\"Extended WM hints\" multi-parent brain damage?)",
                 region_name((WRegion*)cwin));
        }
    }else if(!region_same_rootwin((WRegion*)cwin, (WRegion*)tfor)){
        warn("The transient_for window for \"%s\" is not on the same "
             "screen.", region_name((WRegion*)cwin));
    }else{
        return tfor;
    }
    
    return NULL;
}


static bool postmanage_check(WClientWin *cwin, XWindowAttributes *attr)
{
    /* Check that the window exists. The previous check and selectinput
     * do not seem to catch all cases of window destroyal.
     */
    XSync(ioncore_g.dpy, False);
    
    if(XGetWindowAttributes(ioncore_g.dpy, cwin->win, attr))
        return TRUE;
    
    warn("Window %#x disappeared", cwin->win);
    
    return FALSE;
}


/* This is called when a window is mapped on the root window.
 * We want to check if we should manage the window and how and
 * act appropriately.
 */
WClientWin* ioncore_manage_clientwin(Window win, bool maprq)
{
    WRootWin *rootwin;
    WClientWin *cwin=NULL;
    XWindowAttributes attr;
    XWMHints *hints;
    int init_state=NormalState;
    WManageParams param=MANAGEPARAMS_INIT;

    param.dockapp=FALSE;
    
again:
    /* Is the window already being managed? */
    cwin=XWINDOW_REGION_OF_T(win, WClientWin);
    if(cwin!=NULL)
        return cwin;
    
    /* Select for UnmapNotify and DestroyNotify as the
     * window might get destroyed or unmapped in the meanwhile.
     */
    XSelectInput(ioncore_g.dpy, win, StructureNotifyMask);

    
    /* Is it a dockapp?
     */
    hints=XGetWMHints(ioncore_g.dpy, win);

    if(hints!=NULL && hints->flags&StateHint)
        init_state=hints->initial_state;
    
    if(!param.dockapp && init_state==WithdrawnState && 
       hints->flags&IconWindowHint && hints->icon_window!=None){
        /* The dockapp might be displaying its "main" window if no
         * wm that understands dockapps has been managing it.
         */
        if(!maprq)
            XUnmapWindow(ioncore_g.dpy, win);
        
        XSelectInput(ioncore_g.dpy, win, 0);
        
        win=hints->icon_window;
        
        /* It is a dockapp, do everything again from the beginning, now
         * with the icon window.
         */
        param.dockapp=TRUE;
        goto again;
    }
    
    if(hints!=NULL)
        XFree((void*)hints);

    if(!XGetWindowAttributes(ioncore_g.dpy, win, &attr)){
        if(maprq)
            warn("Window %#x disappeared", win);
        goto fail2;
    }
    
    attr.width=maxof(attr.width, 1);
    attr.height=maxof(attr.height, 1);

    /* Do we really want to manage it? */
    if(!param.dockapp && (attr.override_redirect || 
        (!maprq && attr.map_state!=IsViewable))){
        goto fail2;
    }

    /* Find root window */
    FOR_ALL_ROOTWINS(rootwin){
        if(WROOTWIN_ROOT(rootwin)==attr.root)
            break;
    }

    if(rootwin==NULL){
        warn("Unable to find a matching root window!");
        goto fail2;
    }

    /* Allocate and initialize */
    cwin=create_clientwin((WWindow*)rootwin, win, &attr);
    
    if(cwin==NULL){
        warn("Unable to create a client window structure!");
        goto fail2;
    }

    param.geom=REGION_GEOM(cwin);
    param.maprq=maprq;
    param.userpos=(cwin->size_hints.flags&USPosition);
    param.switchto=(init_state!=IconicState && clientwin_get_switchto(cwin));
    param.jumpto=extl_table_is_bool_set(cwin->proptab, "jumpto");
    param.gravity=(cwin->size_hints.flags&PWinGravity
                   ? cwin->size_hints.win_gravity
                   : ForgetGravity);
    
    param.tfor=clientwin_get_transient_for(cwin);

    if(!handle_target_props(cwin, &param)){
        bool managed=FALSE;
        
        CALL_ALT_B(managed, clientwin_do_manage_alt, (cwin, &param));

        if(!managed){
            warn("Unable to manage client window %d\n", win);
            goto failure;
        }
    }
    
    if(!region_is_fully_mapped((WRegion*)cwin) && 
       ioncore_g.opmode==IONCORE_OPMODE_NORMAL){
        region_notify_activity((WRegion*)cwin);
    }
    
    
    if(postmanage_check(cwin, &attr)){
        if(param.jumpto && ioncore_g.focus_next==NULL)
            region_goto((WRegion*)cwin);
        extl_call_named("call_hook", "so", NULL, "clientwin_added", cwin);
        return cwin;
    }

failure:
    clientwin_destroyed(cwin);
    return NULL;

fail2:
    XSelectInput(ioncore_g.dpy, win, 0);
    return NULL;
}


void clientwin_tfor_changed(WClientWin *cwin)
{
#if 0
    WManageParams param=MANAGEPARAMS_INIT;
    bool succeeded=FALSE;
    param.tfor=clientwin_get_transient_for(cwin);
    if(param.tfor==NULL)
        return;
    
    region_rootpos((WRegion*)cwin, &(param.geom.x), &(param.geom.y));
    param.geom.w=REGION_GEOM(cwin).w;
    param.geom.h=REGION_GEOM(cwin).h;
    param.maprq=FALSE;
    param.userpos=FALSE;
    param.switchto=region_may_control_focus((WRegion*)cwin);
    param.jumpto=extl_table_is_bool_set(cwin->proptab, "jumpto");
    param.gravity=ForgetGravity;
    
    CALL_ALT_B(succeeded, clientwin_do_manage_alt, (cwin, &param));
    warn("WM_TRANSIENT_FOR changed for \"%s\".", region_name((WRegion*)cwin));
#else
    warn("Changes is WM_TRANSIENT_FOR property are unsupported.");
#endif        
}


/*}}}*/


/*{{{ Add/remove managed */

typedef struct{
    WClientWin *cwin;
    WRegion *transient;
} TransRepar;

static TransRepar *transient_reparents=NULL;

static WRegion *clientwin_do_attach_transient(WClientWin *cwin, 
                                              WRegionAttachHandler *fn,
                                              void *fnparams,
                                              WRegion *thereg)
{
    WWindow *par=REGION_PARENT_CHK(cwin, WWindow);
    WRegion *reg;
    TransRepar tp, *tpold;
    WFitParams fp;

    if(par==NULL)
        return NULL;

    tp.cwin=cwin;
    tp.transient=thereg;
    tpold=transient_reparents;
    transient_reparents=&tp;
    
    fp.mode=REGION_FIT_BOUNDS;
    fp.g=cwin->last_fp.g;
    
    reg=fn(par, &fp, fnparams);
    
    transient_reparents=tpold;
    
    if(reg==NULL)
        return NULL;
    
    region_set_manager(reg, (WRegion*)cwin, &(cwin->transient_list));
    region_stack_above(reg, (WRegion*)cwin);
    
    if(REGION_IS_MAPPED((WRegion*)cwin))
        region_map(reg);
    else
        region_unmap(reg);
    
    if(region_may_control_focus((WRegion*)cwin))
        region_set_focus(reg);
    
    return reg;
}


bool clientwin_attach_transient(WClientWin *cwin, WRegion *transient)
{
    return (region__attach_reparent((WRegion*)cwin, transient,
                                    ((WRegionDoAttachFn*)
                                     clientwin_do_attach_transient), 
                                    transient)!=NULL);
}


static void clientwin_managed_remove(WClientWin *cwin, WRegion *transient)
{
    bool mcf=region_may_control_focus((WRegion*)cwin);
    
    region_unset_manager(transient, (WRegion*)cwin, &(cwin->transient_list));
    region_reset_stacking(transient);
    
    if(mcf){
        WRegion *reg=LATEST_TRANSIENT(cwin);
        if(reg==NULL)
            reg=&cwin->region;
        /*else
            region_raise(reg);*/
        
        region_set_focus(reg);
    }
}


static bool clientwin_rescue_clientwins(WClientWin *cwin)
{
    bool ret1, ret2;
    
    ret1=region_rescue_managed_clientwins((WRegion*)cwin,
                                          cwin->transient_list);
    ret2=region_rescue_child_clientwins((WRegion*)cwin);
    
    return (ret1 && ret2);
}


bool clientwin_manage_clientwin(WClientWin *cwin, WClientWin *cwin2,
                                const WManageParams *param, int redir)
{
    if(redir==MANAGE_REDIR_STRICT_YES)
        return FALSE;
    
    /* Only catch windows with transient mode set to current here. */
    if(clientwin_get_transient_mode(cwin2)!=TRANSIENT_MODE_CURRENT)
        return FALSE;
    
    return clientwin_attach_transient(cwin, (WRegion*)cwin2);
}


/*}}}*/


/*{{{ Unmanage/destroy */


static void reparent_root(WClientWin *cwin)
{
    XWindowAttributes attr;
    WWindow *par;
    Window dummy;
    int x=0, y=0;
    
    XGetWindowAttributes(ioncore_g.dpy, cwin->win, &attr);
    
    par=REGION_PARENT_CHK(cwin, WWindow);
    
    if(par==NULL){
        x=REGION_GEOM(cwin).x;
        y=REGION_GEOM(cwin).y;
    }else{
        int dr=REGION_GEOM(par).w-REGION_GEOM(cwin).w-REGION_GEOM(cwin).x;
        int db=REGION_GEOM(par).h-REGION_GEOM(cwin).h-REGION_GEOM(cwin).y;
        dr=maxof(dr, 0);
        db=maxof(db, 0);
        
        XTranslateCoordinates(ioncore_g.dpy, par->win, attr.root, 0, 0, 
                              &x, &y, &dummy);

        x-=xgravity_deltax(cwin->size_hints.win_gravity, 
                           maxof(0, REGION_GEOM(cwin).x), dr);
        y-=xgravity_deltay(cwin->size_hints.win_gravity, 
                           maxof(0, REGION_GEOM(cwin).y), db);
    }
    
    XReparentWindow(ioncore_g.dpy, cwin->win, attr.root, x, y);
}


void clientwin_deinit(WClientWin *cwin)
{
    WRegion *reg;
    
    while(cwin->transient_list!=NULL)
        destroy_obj((Obj*)(cwin->transient_list));
    
    if(cwin->win!=None){
        XSelectInput(ioncore_g.dpy, cwin->win, 0);
        
        reparent_root(cwin);
        
        if(cwin->orig_bw!=0)
            configure_cwin_bw(cwin->win, cwin->orig_bw);
        
        XRemoveFromSaveSet(ioncore_g.dpy, cwin->win);
        XDeleteContext(ioncore_g.dpy, cwin->win, ioncore_g.win_context);
        
        if(ioncore_g.opmode==IONCORE_OPMODE_DEINIT){
            XMapWindow(ioncore_g.dpy, cwin->win);
            /* Make sure the topmost window has focus; it doesn't really
             * matter which one has as long as some has.
             */
            xwindow_do_set_focus(cwin->win);
        }else{
            set_clientwin_state(cwin, WithdrawnState);
            XDeleteProperty(ioncore_g.dpy, cwin->win, 
                            ioncore_g.atom_net_wm_state);
        }
    }
    clientwin_clear_colormaps(cwin);
    
    watch_reset(&(cwin->fsinfo.last_mgr_watch));
    region_deinit((WRegion*)cwin);
}


/* Used when the window was unmapped */
void clientwin_unmapped(WClientWin *cwin)
{
    bool cf=region_may_control_focus((WRegion*)cwin);
    region_rescue_clientwins((WRegion*)cwin);
    if(cf && cwin->fsinfo.last_mgr_watch.obj!=NULL)
        region_goto((WRegion*)(cwin->fsinfo.last_mgr_watch.obj));
    destroy_obj((Obj*)cwin);
}


/* Used when the window was deastroyed */
void clientwin_destroyed(WClientWin *cwin)
{
    XDeleteContext(ioncore_g.dpy, cwin->win, ioncore_g.win_context);
    cwin->win=None;
    clientwin_unmapped(cwin);
}


/*}}}*/


/*{{{ Kill/close */


static bool send_clientmsg(Window win, Atom a, Time stmp)
{
    XClientMessageEvent ev;
    
    ev.type=ClientMessage;
    ev.window=win;
    ev.message_type=ioncore_g.atom_wm_protocols;
    ev.format=32;
    ev.data.l[0]=a;
    ev.data.l[1]=stmp;
    
    return (XSendEvent(ioncore_g.dpy, win, False, 0L, (XEvent*)&ev)!=0);
}


/*EXTL_DOC
 * Attempt to kill (with XKillWindow) the client that owns the X
 * window correspoding to \var{cwin}.
 */
EXTL_EXPORT_MEMBER
void clientwin_kill(WClientWin *cwin)
{
    XKillClient(ioncore_g.dpy, cwin->win);
}


bool clientwin_rqclose(WClientWin *cwin)
{
    if(cwin->flags&CLIENTWIN_P_WM_DELETE){
        send_clientmsg(cwin->win, ioncore_g.atom_wm_delete, 
                       ioncore_get_timestamp());
        return TRUE;
    }else{
        warn("Client does not support WM_DELETE.");
        return FALSE;
    }
}


/*}}}*/


/*{{{ State (hide/show) */


static void set_clientwin_state(WClientWin *cwin, int state)
{
    if(cwin->state!=state){
        cwin->state=state;
        xwindow_set_state_property(cwin->win, state);
    }
}


static void hide_clientwin(WClientWin *cwin)
{
    if(cwin->flags&CLIENTWIN_PROP_ACROBATIC){
        XMoveWindow(ioncore_g.dpy, cwin->win,
                    -2*cwin->last_fp.g.w, -2*cwin->last_fp.g.h);
        return;
    }
    
    set_clientwin_state(cwin, IconicState);
    XSelectInput(ioncore_g.dpy, cwin->win,
                 cwin->event_mask&~(StructureNotifyMask|EnterWindowMask));
    XUnmapWindow(ioncore_g.dpy, cwin->win);
    XSelectInput(ioncore_g.dpy, cwin->win, cwin->event_mask);
}


static void show_clientwin(WClientWin *cwin)
{
    if(cwin->flags&CLIENTWIN_PROP_ACROBATIC){
        XMoveWindow(ioncore_g.dpy, cwin->win,
                    REGION_GEOM(cwin).x, REGION_GEOM(cwin).y);
        if(cwin->state==NormalState)
            return;
    }
    
    XSelectInput(ioncore_g.dpy, cwin->win,
                 cwin->event_mask&~(StructureNotifyMask|EnterWindowMask));
    XMapWindow(ioncore_g.dpy, cwin->win);
    XSelectInput(ioncore_g.dpy, cwin->win, cwin->event_mask);
    set_clientwin_state(cwin, NormalState);
}


/*}}}*/


/*{{{ Resize/reparent/reconf helpers */


void clientwin_notify_rootpos(WClientWin *cwin, int rootx, int rooty)
{
    XEvent ce;
    Window win;
    
    if(cwin==NULL)
        return;
    
    win=cwin->win;
    
    ce.xconfigure.type=ConfigureNotify;
    ce.xconfigure.event=win;
    ce.xconfigure.window=win;
    ce.xconfigure.x=rootx-cwin->orig_bw;
    ce.xconfigure.y=rooty-cwin->orig_bw;
    ce.xconfigure.width=REGION_GEOM(cwin).w;
    ce.xconfigure.height=REGION_GEOM(cwin).h;
    ce.xconfigure.border_width=cwin->orig_bw;
    ce.xconfigure.above=None;
    ce.xconfigure.override_redirect=False;
    
    XSelectInput(ioncore_g.dpy, win, cwin->event_mask&~StructureNotifyMask);
    XSendEvent(ioncore_g.dpy, win, False, StructureNotifyMask, &ce);
    XSelectInput(ioncore_g.dpy, win, cwin->event_mask);
}


static void sendconfig_clientwin(WClientWin *cwin)
{
    int rootx, rooty;
    
    region_rootpos(&cwin->region, &rootx, &rooty);
    clientwin_notify_rootpos(cwin, rootx, rooty);
}


static void do_reparent_clientwin(WClientWin *cwin, Window win, int x, int y)
{
    XSelectInput(ioncore_g.dpy, cwin->win,
                 cwin->event_mask&~StructureNotifyMask);
    XReparentWindow(ioncore_g.dpy, cwin->win, win, x, y);
    XSelectInput(ioncore_g.dpy, cwin->win, cwin->event_mask);
}


static void convert_geom(WClientWin *cwin, const WFitParams *fp,
                         WRectangle *geom)
{
    const WRectangle *max_geom=&(fp->g);
    bool bottom=FALSE;
    bool top=FALSE;
    int htry=max_geom->h;
    WClientWin *mgr;
    
    if(transient_reparents!=NULL && 
       transient_reparents->transient==(WRegion*)cwin){
        mgr=transient_reparents->cwin;
    }else{
        mgr=REGION_MANAGER_CHK(cwin, WClientWin);
    }
    
    if(mgr!=NULL && fp->mode==REGION_FIT_BOUNDS){
        if(mgr->flags&CLIENTWIN_TRANSIENTS_AT_TOP)
            top=TRUE;
        else
            bottom=TRUE;
        if(cwin->last_h_rq<htry)
            htry=cwin->last_h_rq;
    }
    
    geom->w=max_geom->w;
    geom->h=htry;
    
    xsizehints_correct(&(cwin->size_hints), &(geom->w), &(geom->h), FALSE);

    geom->x=max_geom->x+max_geom->w/2-geom->w/2;
    
    if(top)
        geom->y=max_geom->y;
    else if(bottom)
        geom->y=max_geom->y+max_geom->h-geom->h;
    else
        geom->y=max_geom->y+max_geom->h/2-geom->h/2;
    
    if(geom->h<=1)
        geom->h=1;
    if(geom->w<=1)
        geom->w=1;
}


/*}}}*/


/*{{{ Region dynfuns */


static void clientwin_managed_rqgeom(WClientWin *cwin, WRegion *sub,
                                     int flags, const WRectangle *geom, 
                                     WRectangle *geomret)
{
    WRectangle g=*geom;

    rectangle_constrain(&g, &(cwin->last_fp.g));
    
    if(geomret!=NULL)
        *geomret=g;
    
    if(!(flags&REGION_RQGEOM_TRYONLY))
        region_fit(sub, &g, REGION_FIT_EXACT);
}



static bool clientwin_fitrep(WClientWin *cwin, WWindow *np, WFitParams *fp)
{
    WRegion *transient, *next;
    WRectangle geom;
    int diff;
    bool changes;
    int w, h;

    if(np!=NULL && !region_same_rootwin((WRegion*)cwin, (WRegion*)np))
        return FALSE;

    convert_geom(cwin, fp, &geom);

    changes=(REGION_GEOM(cwin).x!=geom.x ||
             REGION_GEOM(cwin).y!=geom.y ||
             REGION_GEOM(cwin).w!=geom.w ||
             REGION_GEOM(cwin).h!=geom.h);
    
    cwin->last_fp=*fp;
    REGION_GEOM(cwin)=geom;
    
    if(np==NULL && !changes)
        return TRUE;
    
    if(np!=NULL){
        region_detach_parent((WRegion*)cwin);
        do_reparent_clientwin(cwin, np->win, geom.x, geom.y);
        region_attach_parent((WRegion*)cwin, (WRegion*)np);
        sendconfig_clientwin(cwin);

        if(!CLIENTWIN_IS_FULLSCREEN(cwin) && 
           cwin->fsinfo.last_mgr_watch.obj!=NULL){
            watch_reset(&(cwin->fsinfo.last_mgr_watch));
        }
        netwm_update_state(cwin);
    }
    
    w=maxof(1, geom.w);
    h=maxof(1, geom.h);
    
    if(cwin->flags&CLIENTWIN_PROP_ACROBATIC && !REGION_IS_MAPPED(cwin)){
        XMoveResizeWindow(ioncore_g.dpy, cwin->win,
                          -2*cwin->last_fp.g.w, -2*cwin->last_fp.g.h, w, h);
    }else{
        XMoveResizeWindow(ioncore_g.dpy, cwin->win, geom.x, geom.y, w, h);
    }
    
    cwin->flags&=~CLIENTWIN_NEED_CFGNTFY;
    
    
    FOR_ALL_MANAGED_ON_LIST_W_NEXT(cwin->transient_list, transient, next){
        WFitParams fp2;
        fp2.g=fp->g;
        fp2.mode=REGION_FIT_BOUNDS;
        
        if(!region_fitrep(transient, np, &fp2) && np!=NULL){
           warn("Problem: can't reparent a %s managed by a WClientWin"
                "being reparented. Detaching from this object.",
                OBJ_TYPESTR(transient));
                region_detach_manager(transient);
        }
        if(np!=NULL)
            region_stack_above(transient, (WRegion*)cwin);
    }
    
    return TRUE;
}


static void clientwin_map(WClientWin *cwin)
{
    WRegion *sub;
    
    show_clientwin(cwin);
    REGION_MARK_MAPPED(cwin);
    
    FOR_ALL_MANAGED_ON_LIST(cwin->transient_list, sub){
        region_map(sub);
    }
}


static void clientwin_unmap(WClientWin *cwin)
{
    WRegion *sub;
    
    hide_clientwin(cwin);
    REGION_MARK_UNMAPPED(cwin);
    
    FOR_ALL_MANAGED_ON_LIST(cwin->transient_list, sub){
        region_unmap(sub);
    }
}


static void clientwin_do_set_focus(WClientWin *cwin, bool warp)
{
    WRegion *reg=LATEST_TRANSIENT(cwin);
    
    if(warp)
        region_do_warp((WRegion*)cwin);
    
    if(reg!=NULL){
        region_do_set_focus(reg, FALSE);
        return;
    }

    if(cwin->flags&CLIENTWIN_P_WM_TAKE_FOCUS){
        Time stmp=ioncore_get_timestamp();
        send_clientmsg(cwin->win, ioncore_g.atom_wm_take_focus, stmp);
    }

    xwindow_do_set_focus(cwin->win);
    
    XSync(ioncore_g.dpy, 0);
}


static bool clientwin_managed_display(WClientWin *cwin, WRegion *sub)
{
    if(!REGION_IS_MAPPED(cwin))
        return FALSE;
    region_map(sub);
    region_raise(sub);
    return TRUE;
}


static Window clientwin_restack(WClientWin *cwin, Window other, int mode)
{
    xwindow_restack(cwin->win, other, mode);
    return cwin->win;
}


static Window clientwin_x_window(WClientWin *cwin)
{
    return cwin->win;
}


static void clientwin_activated(WClientWin *cwin)
{
    clientwin_install_colormap(cwin);
}


static void clientwin_resize_hints(WClientWin *cwin, XSizeHints *hints_ret,
                                   uint *relw_ret, uint *relh_ret)
{
    if(relw_ret!=NULL)
        *relw_ret=REGION_GEOM(cwin).w;
    if(relh_ret!=NULL)
        *relh_ret=REGION_GEOM(cwin).h;
    *hints_ret=cwin->size_hints;
    
    xsizehints_adjust_for(hints_ret, cwin->transient_list);
}


static WRegion *clientwin_managed_focus(WClientWin *cwin, WRegion *reg)
{
    return LATEST_TRANSIENT(cwin);
}


/*}}}*/


/*{{{ Identity & lookup */


/*EXTL_DOC
 * Returns a table containing the properties \code{WM_CLASS} (table entries
 * \var{instance} and \var{class}),  \code{WM_WINDOW_ROLE} (and \var{role})
 * and \code{_ION_KLUDGES} (\var{kludges}) properties for \var{cwin}.
 * If a property is not set, the corresponding field(s) are unset in the 
 * table.
 */
EXTL_EXPORT_MEMBER
ExtlTab clientwin_get_ident(WClientWin *cwin)
{
    char **p=NULL, *wrole=NULL, *kludges=NULL;
    int n=0, n2=0, n3=0, tmp=0;
    ExtlTab tab;
    
    p=xwindow_get_text_property(cwin->win, XA_WM_CLASS, &n);
    wrole=xwindow_get_string_property(cwin->win, ioncore_g.atom_wm_window_role, &n2);
    kludges=xwindow_get_string_property(cwin->win, ioncore_g.atom_kludges, &n3);
    
    tab=extl_create_table();
    if(n>=2 && p[1]!=NULL)
        extl_table_sets_s(tab, "class", p[1]);
    if(n>=1 && p[0]!=NULL)
        extl_table_sets_s(tab, "instance", p[0]);
    if(wrole!=NULL)
        extl_table_sets_s(tab, "role", wrole);
    if(kludges!=NULL)
        extl_table_sets_s(tab, "kludges", kludges);
    
    if(p!=NULL)
        XFreeStringList(p);
    if(wrole!=NULL)
        free(wrole);
    if(kludges!=NULL)
        free(kludges);
    
    return tab;
}


/*}}}*/


/*{{{ ConfigureRequest */


#undef MAX
#define MAX(X, Y) ((X) > (Y) ? (X) : (Y))


void clientwin_handle_configure_request(WClientWin *cwin,
                                        XConfigureRequestEvent *ev)
{
    if(ev->value_mask&CWBorderWidth)
        cwin->orig_bw=ev->border_width;
    
    if(cwin->flags&CLIENTWIN_PROP_IGNORE_CFGRQ){
        sendconfig_clientwin(cwin);
        return;
    }

    /* check full screen request */
    if((ev->value_mask&(CWWidth|CWHeight))==(CWWidth|CWHeight)){
        bool sw=region_may_control_focus((WRegion*)cwin);
        if(clientwin_check_fullscreen_request(cwin, ev->width, ev->height, sw))
            return;
    }

    cwin->flags|=CLIENTWIN_NEED_CFGNTFY;

    if(ev->value_mask&(CWX|CWY|CWWidth|CWHeight)){
        WRectangle geom;
        WRegion *mgr;
        int rqflags=REGION_RQGEOM_WEAK_ALL;
        int gdx=0, gdy=0;

        /* Do I need to insert another disparaging comment on the person who
         * invented special server-supported window borders that are not 
         * accounted for in the window size? Keep it simple, stupid!
         */
        if(cwin->size_hints.flags&PWinGravity){
            gdx=xgravity_deltax(cwin->size_hints.win_gravity, 
                               -cwin->orig_bw, -cwin->orig_bw);
            gdy=xgravity_deltay(cwin->size_hints.win_gravity, 
                               -cwin->orig_bw, -cwin->orig_bw);
        }
        
        /* Rootpos is usually wrong here, but managers (frames) that respect
         * position at all, should define region_rqgeom_clientwin to
         * handle this how they see fit.
         */
        region_rootpos((WRegion*)cwin, &(geom.x), &(geom.y));
        geom.w=REGION_GEOM(cwin).w;
        geom.h=REGION_GEOM(cwin).h;
        
        if(ev->value_mask&CWWidth){
            /* If x was not changed, keep reference point where it was */
            if(cwin->size_hints.flags&PWinGravity){
                geom.x+=xgravity_deltax(cwin->size_hints.win_gravity, 0,
                                       ev->width-geom.w);
            }
            geom.w=MAX(ev->width, 1);
            rqflags&=~REGION_RQGEOM_WEAK_W;
        }
        if(ev->value_mask&CWHeight){
            /* If y was not changed, keep reference point where it was */
            if(cwin->size_hints.flags&PWinGravity){
                geom.y+=xgravity_deltay(cwin->size_hints.win_gravity, 0,
                                       ev->height-geom.h);
            }
            geom.h=MAX(ev->height, 1);
            cwin->last_h_rq=geom.h;
            rqflags&=~REGION_RQGEOM_WEAK_H;
        }
        if(ev->value_mask&CWX){
            geom.x=ev->x+gdx;
            rqflags&=~REGION_RQGEOM_WEAK_X;
        }
        if(ev->value_mask&CWY){
            geom.y=ev->y+gdy;
            rqflags&=~REGION_RQGEOM_WEAK_Y;
        }
        
        mgr=region_manager((WRegion*)cwin);
        if(mgr!=NULL && HAS_DYN(mgr, region_rqgeom_clientwin)){
            /* Manager gets to decide how to handle position request. */
            region_rqgeom_clientwin(mgr, cwin, rqflags, &geom);
        }else{
            /*region_convert_root_geom(region_parent((WRegion*)cwin),
                                     &geom);
            region_rqgeom((WRegion*)cwin, rqflags, &geom, NULL);*/
            /* Just use any known available space wanted or give up some */
            region_fitrep((WRegion*)cwin, NULL, &(cwin->last_fp));
        }
    }

    if(cwin->flags&CLIENTWIN_NEED_CFGNTFY){
        sendconfig_clientwin(cwin);
        cwin->flags&=~CLIENTWIN_NEED_CFGNTFY;
    }
}


/*}}}*/


/*{{{ Kludges */


/*EXTL_DOC
 * Attempts to fix window size problems with non-ICCCM compliant
 * programs.
 */
EXTL_EXPORT_MEMBER
void clientwin_nudge(WClientWin *cwin)
{
    XResizeWindow(ioncore_g.dpy, cwin->win, 2*cwin->last_fp.g.w,
                  2*cwin->last_fp.g.h);
    XFlush(ioncore_g.dpy);
    XResizeWindow(ioncore_g.dpy, cwin->win, REGION_GEOM(cwin).w,
                  REGION_GEOM(cwin).h);
}


/*}}}*/


/*{{{ Misc. */


/*EXTL_DOC
 * Returns a list of regions managed by the clientwin (transients, mostly).
 */
EXTL_EXPORT_MEMBER
ExtlTab clientwin_managed_list(WClientWin *cwin)
{
    return managed_list_to_table(cwin->transient_list, NULL);
}


/*EXTL_DOC
 * Toggle transients managed by \var{cwin} between top/bottom
 * of the window.
 */
EXTL_EXPORT_MEMBER
void clientwin_toggle_transients_pos(WClientWin *cwin)
{
    WRegion *transient;
    WFitParams fp;
    
    cwin->flags^=CLIENTWIN_TRANSIENTS_AT_TOP;

    fp.mode=REGION_FIT_BOUNDS;
    fp.g=cwin->last_fp.g;
    FOR_ALL_MANAGED_ON_LIST(cwin->transient_list, transient){
        region_fitrep(transient, NULL, &fp);
    }
}


static WRegion *clientwin_current(WClientWin *cwin)
{
    return LATEST_TRANSIENT(cwin);
}

/*}}}*/


/*{{{Save/load */


static int last_checkcode=1;


static ExtlTab clientwin_get_configuration(WClientWin *cwin)
{
    int chkc=0;
    ExtlTab tab;
    
    tab=region_get_base_configuration((WRegion*)cwin);

    extl_table_sets_d(tab, "windowid", (double)(cwin->win));
    
    if(last_checkcode!=0){
        chkc=last_checkcode++;
        xwindow_set_integer_property(cwin->win, ioncore_g.atom_checkcode, 
                                     chkc);
        extl_table_sets_i(tab, "checkcode", chkc);
    }
    
    return tab;
}


WRegion *clientwin_load(WWindow *par, const WFitParams *fp, ExtlTab tab)
{
    double wind=0;
    Window win=None;
    int chkc=0, real_chkc=0;
    WClientWin *cwin=NULL;
    XWindowAttributes attr;
    WRectangle rg;

    if(!extl_table_gets_d(tab, "windowid", &wind) ||
       !extl_table_gets_i(tab, "checkcode", &chkc)){
        return NULL;
    }
    
    win=(Window)wind;

    if(!xwindow_get_integer_property(win, ioncore_g.atom_checkcode, &real_chkc))
        return NULL;

    if(real_chkc!=chkc){
        /*warn("Client window check code mismatch.");*/
        return NULL;
    }

    /* Found it! */
    
    if(!XGetWindowAttributes(ioncore_g.dpy, win, &attr)){
        warn("Window %#x disappeared", win);
        return NULL;
    }
    
    if(attr.override_redirect || 
       (ioncore_g.opmode==IONCORE_OPMODE_INIT && attr.map_state!=IsViewable)){
        warn("Saved client window does not want to be managed");
        return NULL;
    }

    attr.x=fp->g.x;
    attr.y=fp->g.y;
    attr.width=fp->g.w;
    attr.height=fp->g.h;

    cwin=create_clientwin(par, win, &attr);
    
    if(cwin==NULL)
        return FALSE;
    
    /* Reparent and resize taking limits set by size hints into account */
    convert_geom(cwin, fp, &rg);
    REGION_GEOM(cwin)=rg;
    do_reparent_clientwin(cwin, par->win, rg.x, rg.y);
    XResizeWindow(ioncore_g.dpy, win, maxof(1, rg.w), maxof(1, rg.h));
    
    if(!postmanage_check(cwin, &attr)){
        clientwin_destroyed(cwin);
        return NULL;
    }
    
    return (WRegion*)cwin;
}


/*}}}*/


/*{{{ Dynfuntab and class info */


static DynFunTab clientwin_dynfuntab[]={
    {(DynFun*)region_fitrep,
     (DynFun*)clientwin_fitrep},

    {(DynFun*)region_current,
     (DynFun*)clientwin_current},
    
    {region_map,
     clientwin_map},
    
    {region_unmap,
     clientwin_unmap},
    
    {region_do_set_focus, 
     clientwin_do_set_focus},
    
    {(DynFun*)region_managed_display,
     (DynFun*)clientwin_managed_display},
    
    {region_notify_rootpos, 
     clientwin_notify_rootpos},
    
    {(DynFun*)region_restack, 
     (DynFun*)clientwin_restack},
    
    {(DynFun*)region_xwindow, 
     (DynFun*)clientwin_x_window},
    
    {region_activated, 
     clientwin_activated},
    
    {region_resize_hints, 
     clientwin_resize_hints},
    
    {(DynFun*)region_managed_control_focus,
     (DynFun*)clientwin_managed_focus},
    
    {region_managed_remove, 
     clientwin_managed_remove},
    
    {region_managed_rqgeom, 
     clientwin_managed_rqgeom},
    
    {(DynFun*)region_rqclose, 
     (DynFun*)clientwin_rqclose},
    
    {(DynFun*)region_get_configuration,
     (DynFun*)clientwin_get_configuration},
    
    {(DynFun*)region_rescue_clientwins,
     (DynFun*)clientwin_rescue_clientwins},
    
    {(DynFun*)region_manage_clientwin,
     (DynFun*)clientwin_manage_clientwin},

    END_DYNFUNTAB
};


IMPLCLASS(WClientWin, WRegion, clientwin_deinit, clientwin_dynfuntab);


/*}}}*/
