// -*- C++ -*-
// --------------------------------------------------------------------
// A page of a document.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEPAGE_H
#define IPEPAGE_H

#include "ipeobj.h"
#include "ipepath.h"

// --------------------------------------------------------------------

class IpeStyleSheet;
class IpeXmlDataSource;
class IpeRepository;

struct SSubPath;

class IPE_EXPORT IpeLayer {
public:
  explicit IpeLayer(IpeString name);
  explicit IpeLayer(const IpeXmlAttributes &attr);
  void SaveAsXml(IpeStream &stream) const;
  //! Return name.
  inline IpeString Name() const { return iName; }
  //! Set name.
  inline void SetName(const IpeString &name) { iName = name; }

  //! Is visible?
  inline bool IsVisible() const { return !(iFlags & EHidden); }
  //! Is dimmed?
  inline bool IsDimmed() const { return !!(iFlags & EDim); }
  //! Is locked?
  inline bool IsLocked() const { return !!(iFlags & ELocked); }
  //! Is snapping enabled?
  inline bool IsSnapping() const { return !(iFlags & ENoSnapping); }

  void SetVisible(bool flag);
  void SetDimmed(bool flag);
  void SetLocked(bool flag);
  void SetSnapping(bool flag);

public:
  enum {EHidden = 0x01, EDim = 0x02, ELocked = 0x04, ENoSnapping = 0x08 };
  IpeString iName;
  int iFlags;
};

//! Layers of a page.
typedef std::vector<IpeLayer> IpeLayerSeq;

// --------------------------------------------------------------------

class IPE_EXPORT IpeView {
public:
  //! The various fancy effects that Acrobat Reader will show.
  enum TEffect {
    ENormal, ESplitHI, ESplitHO, ESplitVI, ESplitVO,
    EBlindsH, EBlindsV, EBoxI, EBoxO,
    EWipeLR, EWipeBT, EWipeRL, EWipeTB,
    EDissolve, EGlitterLR, EGlitterTB, EGlitterD };

  IpeView();
  IpeView(const IpeXmlAttributes &attr);
  void SaveAsXml(IpeStream &stream) const;
  void PageDictionary(IpeStream &stream) const;
public:
  std::vector<IpeString> iLayers; //!< Names of layers being shown.
  IpeString iActive;         //!< The layer active in this view.
  TEffect iEffect;           //!< The fancy effect for showing the page.
  int iTransitionTime;       //!< Transition time for effect in showing page.
  int iDuration;             //!< Duration during which page is shown.
};

//! Presentation sequence of a page.
typedef std::vector<IpeView> IpeViewSeq;

// --------------------------------------------------------------------

class IPE_EXPORT IpePage : public IpePgObjectSeq {
public:
  explicit IpePage();

  void SaveAsXml(IpePainter &painter, IpeStream &stream) const;

  //! Return number of layers.
  inline int CountLayers() const { return iLayers.size(); }
  //! Return a layer.
  inline const IpeLayer &Layer(int index) const { return iLayers[index]; }
  int FindLayer(IpeString name) const;
  void SetLayer(int index, const IpeLayer &layer);
  int AddLayer(const IpeLayer &layer, int index = -1);
  int NewLayer(int index);
  void DeleteLayer(int index);
  bool IsLayerActiveInView(int index) const;

  //! Return presentation sequence of this page.
  inline const IpeViewSeq &Views() const {return iViews;}
  //! Return number of views.
  inline int CountViews() const { return iViews.size(); }
  //! Return a view.
  inline const IpeView &View(int index) const { return iViews[index]; }
  void SetViews(const IpeViewSeq &view);
  void SetView(int index, const IpeView &view);
  void AddView(const IpeView &view, int index = -1);
  void DeleteView(int index);

  IpeRect TextBox(const IpeRect &media, const IpeStyleSheet *sheet) const;

  void SetEdited(bool edited);
  //! Return true if page has been modified since last save.
  inline bool IsEdited() const { return iEdited; }

  void DeselectAll();
  void DeselectLayer(int layer);
  void DeselectNotInView(int view);
  bool HasSelection() const;
  iterator PrimarySelection();
  void EnsurePrimarySelection();
  void ExtractSelection(IpePgObjectSeq &seq);
  bool UpdateCloseSelection(const IpeVector &pos, double d, bool primaryOnly);

  void Copy(IpeStream &stream, const IpeStyleSheet *sheet) const;
  bool Paste(int layer, IpeXmlDataSource &source, IpeRepository *rep);
  void CopyPage(IpeStream &stream, const IpeStyleSheet *sheet) const;

  void MoveToLayer(int layer);
  void Delete();
  void SelectAll();
  void SelectAllInLayer(int layer);
  void Group(int layer);
  bool Ungroup(int layer);
  void Front();
  void Back();
  void Duplicate(int layer);
  void SetStroke(IpeAttribute color);
  void SetFill(IpeAttribute color);
  void SetLineWidth(IpeAttribute attr);
  void SetDashStyle(IpeAttribute attr);
  void SetArrows(bool forward, bool backward, IpeAttribute size);
  void SetArrowSize(IpeAttribute size);
  void SetMarkShape(int shape);
  void SetMarkSize(IpeAttribute size);
  void SetTextSize(IpeAttribute size);
  bool ComposePaths(int layer);
  bool DecomposePath(int layer);
  bool JoinPaths(int layer);

  //! Return grid size for this page (0 if not set).
  inline IpeScalar GridSize() const { return iGridSize; }
  //! Set grid size.
  inline void SetGridSize(IpeScalar gs) { iGridSize = gs; }

private:
  IpePath *DoJoinPaths(IpePath *prim, SSubPath *subs, int size);
private:
  bool iEdited;
  IpeViewSeq iViews;
  IpeLayerSeq iLayers;
  IpeScalar iGridSize;
};

//! Constructor takes ownership of object.
inline IpePgObject::IpePgObject(TSelect sel, int layer, IpeObject *obj)
  : iSelect(sel), iLayer(layer), iObject(obj)
{
  // nothing
}

// --------------------------------------------------------------------
#endif
