// --------------------------------------------------------------------
// The Ipe document.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipedoc.h"
#include "ipeiml.h"
#include "ipestyle.h"
#include "ipepainter.h"
#include "ipeutils.h"

/*! \class IpeDocument
  \ingroup doc
  \brief The model for an Ipe document.

  The IpeDocument class is the engine behind the Ipe program.  It
  represents the contents of an Ipe document, and all the methods
  necessary to load, save, and edit it.  It is independent of a
  specific user interface.

  IpeDocument's cannot be copied or assigned.

  Note that the IpeDocument owns the IpeRepository that defines the
  meaning of all object's attributes.  Therefore, the document is the
  largest extent where IpeObject's make sense.  You can have several
  documents at once, but you cannot move IpeObject's (or
  IpeStyleSheet's) from one to the other. If you need to do so, you'll
  have externalize them to XML and internalize them into the other
  document.
*/

//! Constructor clears the boolean flags.
IpeDocument::SProperties::SProperties()
{
  iFullScreen = false;
  iCropBox = false;
}

//! Construct an empty document for filling by a client.
/*! As constructed, it has no pages, A4 media, and
  only the standard style sheet. */
IpeDocument::IpeDocument()
{
  iEdited = false;
  iStyleSheet = IpeStyleSheet::Standard(&iRepository);
  iProperties.iMedia = IpeRect(IpeVector::Zero, IpeVector(595,842)); // A4
}

//! Destructor.
IpeDocument::~IpeDocument()
{
  for (const_iterator it = begin(); it != end(); ++it)
    delete (*it);
  delete iStyleSheet;
}

//! Copy constructor is disabled: it panics.
IpeDocument::IpeDocument(const IpeDocument &)
  : IpePageSeq()
{
  assert(false);
}

//! Assignment operator is disabled: it panics.
IpeDocument &IpeDocument::operator=(const IpeDocument &)
{
  assert(false);
  return *this;
}

//! Construct a document from a file.
/*! Returns 0 if file couldn't be read or parsed, and a reason
  explaining that in \a reason.  If \a reason is positive, it is a
  file (stream) offset where parsing failed.  If \a reason is
  negative, it is an error code. If it is smaller than -60000, then
  it is the negative value of the Ipe version that created that file
  (and which exceeds the version of this Ipe).

  Note that the file name \a fname is an 8-bit string, and passed
  through unchanged to \c fopen.  What encoding you have to use for \a
  fname depends on the operating system.  Under Windows, you probably
  cannot load arbitrary Unicode filenames---you need to roll your own
  function if you need to do that.
*/
IpeDocument *IpeDocument::New(const char *fname, int &reason)
{
  std::FILE *file = std::fopen(fname, "rb");
  if (!file) {
    reason = -1;
    return 0;
  }
  IpeDocument *self = new IpeDocument;
  IpeXmlFileSource source(file);
  IpeImlParser parser(source, self->Repository());
  int requires;
  if (!parser.ParseDocument(*self, requires) || requires > IPELIB_VERSION) {
    delete self;
    self = 0;
    if (requires > IPELIB_VERSION)
      reason = -requires;
    else
      reason = parser.ParsePosition();
  }
  std::fclose(file);
  return self;
}

//! Save in XML format into an IpeStream.
/*! You can set \a creator to set the \c creator attribute in the
  \c <ipe> tag. */
void IpeDocument::SaveAsXml(IpeStream &stream, IpeString creator,
			    bool usePdfBitmaps)
{
  stream << "<ipe version=\"" << IPELIB_VERSION << "\"";
  if (!creator.empty())
    stream << " creator=\"" << creator << "\"";
  stream << " media=\"" << iProperties.iMedia.Min() << " "
	 << iProperties.iMedia.Max() << "\"";
  stream << ">\n";
  IpeString info;
  IpeStringStream infoStr(info);
  infoStr << "<info";
  if (!iProperties.iCreated.empty())
    infoStr << " created=\"" << iProperties.iCreated << "\"";
  if (!iProperties.iModified.empty())
    infoStr << " modified=\"" << iProperties.iModified << "\"";
  if (!iProperties.iTitle.empty()) {
    infoStr << " title=\"";
    infoStr.PutXmlString(iProperties.iTitle);
    infoStr << "\"";
  }
  if (!iProperties.iAuthor.empty()) {
    infoStr << " author=\"";
    infoStr.PutXmlString(iProperties.iAuthor);
    infoStr << "\"";
  }
  if (!iProperties.iSubject.empty()) {
    infoStr << " subject=\"";
    infoStr.PutXmlString(iProperties.iSubject);
    infoStr << "\"";
  }
  if (!iProperties.iKeywords.empty()) {
    infoStr << " keywords=\"";
    infoStr.PutXmlString(iProperties.iKeywords);
    infoStr << "\"";
  }
  if (iProperties.iFullScreen) {
    infoStr << " pagemode=\"fullscreen\"";
  }
  if (iProperties.iCropBox) {
    infoStr << " bbox=\"cropbox\"";
  }
  infoStr << "/>\n";
  if (info.size() > 10)
    stream << info;

  if (!iProperties.iPreamble.empty()) {
    stream << "<preamble>";
    stream.PutXmlString(iProperties.iPreamble);
    stream << "</preamble>\n";
  }
  if (iStyleSheet)
    iStyleSheet->SaveCascadeAsXml(stream);
  // save bitmaps
  IpeBitmapFinder bm;
  for (const_iterator it = begin(); it != end(); ++it)
    bm.ScanPage(*it);
  if (!bm.iBitmaps.empty()) {
    int id = 1;
    IpeBitmap prev;
    std::sort(bm.iBitmaps.begin(), bm.iBitmaps.end());
    for (std::vector<IpeBitmap>::iterator it = bm.iBitmaps.begin();
	 it != bm.iBitmaps.end(); ++it) {
      if (!it->Equal(prev)) {
	if (usePdfBitmaps) {
	  it->SaveAsXml(stream, it->ObjNum(), it->ObjNum());
	} else {
	  it->SaveAsXml(stream, id);
	  it->SetObjNum(id);
	}
      } else
	it->SetObjNum(prev.ObjNum()); // noop if prev == it
      prev = *it;
      ++id;
    }
  }
  // save pages
  IpePainter painter(iStyleSheet);
  for (const_iterator it = begin(); it != end(); ++it)
    (*it)->SaveAsXml(painter, stream);
  stream << "</ipe>\n";
}

//! Save in a file.
/*! Returns true if sucessful.

  Does not reset IsEdited().

  Note that the file name \a fname is an 8-bit string, and passed
  through unchanged to \c fopen.  What encoding you have to use for \a
  fname depends on the operating system.  Under Windows, you probably
  cannot load arbitrary Unicode filenames---you need to roll your own
  function if you need to do that.
*/
bool IpeDocument::Save(const char *fname, IpeString creator)
{
  std::FILE *file = std::fopen(fname, "wb");
  if (!file)
    return false;
  IpeFileStream stream(file);
  SaveAsXml(stream, creator);
  std::fclose(file);
  return true;
}

//! Return true if document has been edited since last save.
bool IpeDocument::IsEdited() const
{
  if (iEdited)
    return true;
  for (const_iterator it = begin(); it != end(); ++it) {
    if ((*it)->IsEdited())
      return true;
  }
  return false;
}

//! Set whether document has been edited.
/*! Methods that can modify the document already set the flag, so you
  only need to call this when inserting or deleting pages, etc.  When
  modifying an IpePage, rather call IpePage::SetEdited(true).

  Clients need to manually reset the edited flag when they save the
  document, or after constructing it during loading.

  Note that calling this with \c edited == \c false will call
  IpePage::SetEdited(false) for all pages.
*/

void IpeDocument::SetEdited(bool edited)
{
  iEdited = edited;
  if (!edited) {
    for (iterator it = begin(); it != end(); ++it)
      (*it)->SetEdited(false);
  }
}

//! Set document properties.
void IpeDocument::SetProperties(const SProperties &props)
{
  iProperties = props;
  iEdited = true;
}

//! Replace the style sheet cascade.
/*! The previous contents is not deleted (because this function is
  often used to insert style sheets into the cascade).

  Sets the edited flag. */
void IpeDocument::SetStyleSheet(IpeStyleSheet *sheet)
{
  iStyleSheet = sheet;
  iEdited = true;
}

//! Check all symbolic attributes in the document.
/*!  This function verifies that all symbolic attributes in the
  document are defined in the style sheet. It appends to \a seq all
  symbolic attributes (in no particular order, but with not
  duplicates) that are NOT defined.

  Returns \c true if there are no undefined symbolic attributes in the
  document.
*/
bool IpeDocument::CheckStyle(IpeAttributeSeq &seq) const
{
  for (const_iterator it = begin(); it != end(); ++it) {
    const IpePage *page = *it;
    for (IpePage::const_iterator it1 = page->begin(); it1 != page->end();
	 ++it1) {
      it1->Object()->CheckStyle(StyleSheet(), seq);
    }
  }
  return (seq.size() == 0);
}

// --------------------------------------------------------------------
