// --------------------------------------------------------------------
// Wrapper for Freetype library
// Based heavily on code in Xpdf, copyright 2001-2002 Glyph & Cog, LLC
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2004  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipeftfont_p.h"

#include <qpixmap.h>
#include <qimage.h>
#include <qmessagebox.h>

#include <cmath>

// currently pdftex doesn't support CID or CFF, no need to support it
// (this depends on freetype internals, which is dangerous)
#if 0 //~ cff cid->gid map
#include "freetype/internal/cfftypes.h"
#include "freetype/internal/tttypes.h"
#endif

//------------------------------------------------------------------------

/*! \class FTFontEngine
  \brief Wraps handle to Freetype library.
*/

FTFontEngine::FTFontEngine(bool aa)
{
  iOk = false;
  if (FT_Init_FreeType(&iLib))
    return;
  iAa = aa;

#if FREETYPE_MINOR >= 1
  int major, minor, patch;
  FT_Library_Version( iLib, &major, &minor, &patch);
  qDebug("Using Freetype version %d.%d.%d", major, minor, patch);

  if (major != FREETYPE_MAJOR ||
      minor != FREETYPE_MINOR ||
      patch != FREETYPE_PATCH) {
    FT_Done_FreeType(iLib);
    QMessageBox::critical
      (0, "Ipe",
       QObject::tr("Ipe has been compiled using the header files of Freetype "
		   " version %1.%2.%3, but is linked with Freetype "
		   "library version %4.%5.%6.  No text display is possible")
       .arg(FREETYPE_MAJOR).arg(FREETYPE_MINOR).arg(FREETYPE_PATCH)
       .arg(major).arg(minor).arg(patch),
       QObject::tr("Dismiss"));
  }
#endif
  iOk = true;
}

FTFontEngine::~FTFontEngine()
{
  FT_Done_FreeType(iLib);
}

//------------------------------------------------------------------------

/*! \class FTFace
  \brief Represents a unique font dictionary in the PDF file.

  This is what Freetype calls a face. Each FTFace object is
  associated with a unique font dictionary in the PDF file, namely the
  object \c iId.

  A face can be loaded at various sizes, and using differnt
  transformations.  Each combination of size/transformation will be a
  FTFont.

*/

//! Construct font from external font file.
FTFace::FTFace(Ref *id, FTFontEngine *engine, char *fontFileName)
{
  iOk = false;
  iId = *id;
  iEngine = engine;
  iCodeMap = 0;
  if (FT_New_Face(engine->iLib, fontFileName, 0, &iFace))
    return;
  iOk = true;
}

//! Construct font from memory buffer.
FTFace::FTFace(Ref *id, FTFontEngine *engine,
	       const char *buffer, int bufsize)
{
  iOk = false;
  iId = *id;
  iEngine = engine;
  iCodeMap = 0;
  if (FT_New_Memory_Face(engine->iLib, (const unsigned char *) buffer,
			 bufsize, 0, &iFace))
    return;
  iOk = true;
}

//! 8-bit font, TrueType or Type 1/1C
void FTFace::SetMode(char **fontEnc, bool pdfFontHasEncoding, bool type1)
{
  char *name;
  int unicodeCmap, macRomanCmap, msSymbolCmap;
  int i, j;

  iOk = false;

  // xpdf finds this from freetype internals
  // if (!strcmp(iFace->driver->root.clazz->module_name, "type1") ||
  // !strcmp(iFace->driver->root.clazz->module_name, "cff")) {

  if (type1) {
    iMode = ftFontModeCodeMapDirect;
    iCodeMap = (Guint *)gmalloc(256 * sizeof(Guint));
    for (i = 0; i < 256; ++i) {
      iCodeMap[i] = 0;
      if ((name = fontEnc[i])) {
	iCodeMap[i] = FT_Get_Name_Index(iFace, name);
      }
    }

  } else {

    // To match up with the Adobe-defined behaviour, we choose a cmap
    // like this:
    // 1. If the PDF font has an encoding:
    //    1a. If the TrueType font has a Microsoft Unicode cmap, use it,
    //        and use the Unicode indexes, not the char codes.
    //    1b. If the TrueType font has a Macintosh Roman cmap, use it,
    //        and reverse map the char names through MacRomanEncoding to
    //        get char codes.
    // 2. If the PDF font does not have an encoding:
    //    2a. If the TrueType font has a Macintosh Roman cmap, use it,
    //        and use char codes directly.
    //    2b. If the TrueType font has a Microsoft Symbol cmap, use it,
    //        and use (0xf000 + char code).
    // 3. If none of these rules apply, use the first cmap and hope for
    //    the best (this shouldn't happen).
    unicodeCmap = macRomanCmap = msSymbolCmap = 0xffff;
    for (i = 0; i < iFace->num_charmaps; ++i) {
      if (iFace->charmaps[i]->platform_id == 3 &&
	  iFace->charmaps[i]->encoding_id == 1) {
	unicodeCmap = i;
      } else if (iFace->charmaps[i]->platform_id == 1 &&
		 iFace->charmaps[i]->encoding_id == 0) {
	macRomanCmap = i;
      } else if (iFace->charmaps[i]->platform_id == 3 &&
		 iFace->charmaps[i]->encoding_id == 0) {
	msSymbolCmap = i;
      }
    }
    i = 0;
    iMode = ftFontModeCharCode;
    iCharMapOffset = 0;
    if (pdfFontHasEncoding) {
      if (unicodeCmap != 0xffff) {
	i = unicodeCmap;
	iMode = ftFontModeUnicode;
      } else if (macRomanCmap != 0xffff) {
	i = macRomanCmap;
	iMode = ftFontModeCodeMap;
	iCodeMap = (Guint *)gmalloc(256 * sizeof(Guint));
	for (j = 0; j < 256; ++j) {
	  if (fontEnc[j]) {
	    iCodeMap[j] = globalParams->getMacRomanCharCode(fontEnc[j]);
	  } else {
	    iCodeMap[j] = 0;
	  }
	}
      }
    } else {
      if (macRomanCmap != 0xffff) {
	i = macRomanCmap;
	iMode = ftFontModeCharCode;
      } else if (msSymbolCmap != 0xffff) {
	i = msSymbolCmap;
	iMode = ftFontModeCharCodeOffset;
	iCharMapOffset = 0xf000;
      }
    }
    if (FT_Set_Charmap(iFace, iFace->charmaps[i]))
      return;
  }

  iOk = true;
}

//! CID font, TrueType
void FTFace::SetMode(Gushort *cidToGID, int cidToGIDLen)
{
  iCidToGID = cidToGID;
  iCidToGIDLen = cidToGIDLen;
  iMode = ftFontModeCIDToGIDMap;
}

// This is not currently used by pdftex:
//! CID font, Type 0C (CFF)
void FTFace::SetMode()
{
  iCidToGID = 0;
  iCidToGIDLen = 0;
  iMode = ftFontModeCFFCharset;
}

//! Destructor.
FTFace::~FTFace()
{
  if (iFace)
    FT_Done_Face(iFace);
  if (iCodeMap)
    gfree(iCodeMap);
}

//------------------------------------------------------------------------

/*! \class FTFont
  \brief A face at a given size and transformation.
*/

//! Construct from face and given transformation.
FTFont::FTFont(const FTFace *face,
	       double m11, double m12,
	       double m21, double m22)
{
  FT_Face fface;
  int x, xMin, xMax;
  int y, yMin, yMax;
  int i;

  iCache = 0;
  iCacheTags = 0;

  iOk = false;
  iFace = const_cast<FTFace *>(face);
  iM11 = m11;
  iM12 = m12;
  iM21 = m21;
  iM22 = m22;

  // create the transformed instance
  double m[4];
  m[0] = iM11;
  m[1] = -iM12;
  m[2] = iM21;
  m[3] = -iM22;

  FTFontEngine *engine = iFace->iEngine;
  fface = iFace->iFace;
  double size = sqrt(m[2]*m[2] + m[3]*m[3]);
  double div = fface->bbox.xMax > 20000 ? 65536 : 1;

  // transform the four corners of the font bounding box -- the min
  // and max values form the bounding box of the transformed font
  x = (int)((m[0] * fface->bbox.xMin + m[2] * fface->bbox.yMin) /
	    (div * fface->units_per_EM));
  xMin = xMax = x;
  y = (int)((m[1] * fface->bbox.xMin + m[3] * fface->bbox.yMin) /
	    (div * fface->units_per_EM));
  yMin = yMax = y;
  x = (int)((m[0] * fface->bbox.xMin + m[2] * fface->bbox.yMax) /
	    (div * fface->units_per_EM));
  if (x < xMin) {
    xMin = x;
  } else if (x > xMax) {
    xMax = x;
  }
  y = (int)((m[1] * fface->bbox.xMin + m[3] * fface->bbox.yMax) /
	    (div * fface->units_per_EM));
  if (y < yMin) {
    yMin = y;
  } else if (y > yMax) {
    yMax = y;
  }
  x = (int)((m[0] * fface->bbox.xMax + m[2] * fface->bbox.yMin) /
	    (div * fface->units_per_EM));
  if (x < xMin) {
    xMin = x;
  } else if (x > xMax) {
    xMax = x;
  }
  y = (int)((m[1] * fface->bbox.xMax + m[3] * fface->bbox.yMin) /
	    (div * fface->units_per_EM));
  if (y < yMin) {
    yMin = y;
  } else if (y > yMax) {
    yMax = y;
  }
  x = (int)((m[0] * fface->bbox.xMax + m[2] * fface->bbox.yMax) /
	    (div * fface->units_per_EM));
  if (x < xMin) {
    xMin = x;
  } else if (x > xMax) {
    xMax = x;
  }
  y = (int)((m[1] * fface->bbox.xMax + m[3] * fface->bbox.yMax) /
	    (div * fface->units_per_EM));
  if (y < yMin) {
    yMin = y;
  } else if (y > yMax) {
    yMax = y;
  }
  // This is a kludge: some buggy PDF generators embed fonts with
  // zero bounding boxes.
  if (xMax == xMin) {
    xMin = 0;
    xMax = (int) size;
  }
  if (yMax == yMin) {
    yMin = 0;
    yMax = (int)(1.2 * size);
  }
  // this should be (max - min + 1), but we add some padding to
  // deal with rounding errors
  iGlyphW = xMax - xMin + 3;
  iGlyphH = yMax - yMin + 3;
  // another kludge: some CJK TT fonts have bogus bboxes, so add more
  // padding
  if (fface->num_glyphs > 1000) {
    iGlyphW += iGlyphW >> 1;
    iGlyphH += iGlyphH >> 1;
  }
  if (engine->iAa) {
    iGlyphSize = iGlyphW * iGlyphH;
  } else {
    iGlyphSize = ((iGlyphW + 7) >> 3) * iGlyphH;
  }

  // set up the glyph pixmap cache
  iCacheAssoc = 8;
  if (iGlyphSize <= 256) {
    iCacheSets = 8;
  } else if (iGlyphSize <= 512) {
    iCacheSets = 4;
  } else if (iGlyphSize <= 1024) {
    iCacheSets = 2;
  } else {
    iCacheSets = 1;
  }
  iCache = (Guchar *)gmalloc(iCacheSets * iCacheAssoc * iGlyphSize);
  iCacheTags = (FTFontCacheTag *)gmalloc(iCacheSets * iCacheAssoc *
					 sizeof(FTFontCacheTag));
  for (i = 0; i < iCacheSets * iCacheAssoc; ++i) {
    iCacheTags[i].mru = i & (iCacheAssoc - 1);
  }

  // compute the transform matrix
  iMatrix.xx = (FT_Fixed)((m[0] / size) * 65536);
  iMatrix.yx = (FT_Fixed)((m[1] / size) * 65536);
  iMatrix.xy = (FT_Fixed)((m[2] / size) * 65536);
  iMatrix.yy = (FT_Fixed)((m[3] / size) * 65536);

  iOk = true;
}

FTFont::~FTFont()
{
  gfree(iCacheTags);
  gfree(iCache);
}

//! Retrieve rasterized glyph from cache, creating it if necessary.
/*! \a w and \a h are set to width and height of the resulting pixmap,
  \a x and \a y are set to the offset in the pixmap---that is,
  they have to be subtracted from the hotspot to get the coordinates
  of the top-left pixel to be transferred. */
Guchar *FTFont::GetGlyphPixmap(CharCode c, Unicode u,
			       int *x, int *y, int *w, int *h)
{
  int i, j, k;
  Guchar *ret;

  // check the cache
  i = (c & (iCacheSets - 1)) * iCacheAssoc;
  for (j = 0; j < iCacheAssoc; ++j) {
    if ((iCacheTags[i+j].mru & 0x8000) && iCacheTags[i+j].code == int(c)) {
      *x = iCacheTags[i+j].x;
      *y = iCacheTags[i+j].y;
      *w = iCacheTags[i+j].w;
      *h = iCacheTags[i+j].h;
      for (k = 0; k < iCacheAssoc; ++k) {
	if (k != j &&
	    (iCacheTags[i+k].mru & 0x7fff) < (iCacheTags[i+j].mru & 0x7fff)) {
	  ++iCacheTags[i+k].mru;
	}
      }
      iCacheTags[i+j].mru = 0x8000;
      return iCache + (i+j) * iGlyphSize;
    }
  }

  if (FT_Set_Pixel_Sizes(iFace->iFace, 0, (int) sqrt(iM21*iM21 + iM22*iM22)))
    return 0;
  FT_Set_Transform(iFace->iFace, &iMatrix, 0);
  FT_UInt idx = GetGlyphIndex(c, u);
  // if we have the FT2 bytecode interpreter, autohinting won't be used
#ifdef TT_CONFIG_OPTION_BYTECODE_INTERPRETER
  if (FT_Load_Glyph(iFace->iFace, idx, FT_LOAD_DEFAULT)) {
    return false;
  }
#else
  // XPDF comment:
  // FT2's autohinting doesn't always work very well (especially with
  // font subsets), so turn it off if anti-aliasing is enabled; if
  // anti-aliasing is disabled, this seems to be a tossup - some fonts
  // look better with hinting, some without, so leave hinting on
  if (FT_Load_Glyph(iFace->iFace, idx,
		    iFace->iEngine->iAa ? FT_LOAD_NO_HINTING
		    : FT_LOAD_DEFAULT))
    return 0;
#endif
  FT_GlyphSlot slot = iFace->iFace->glyph;
  if (FT_Render_Glyph(slot,
		      iFace->iEngine->iAa ? ft_render_mode_normal :
		      ft_render_mode_mono))
    return 0;
  *x = -slot->bitmap_left;
  *y = slot->bitmap_top;
  *w = slot->bitmap.width;
  *h = slot->bitmap.rows;
  if (*w > iGlyphW || *h > iGlyphH) {
#if 1 //~ debug
    fprintf(stderr, "Weird FreeType glyph size: %d > %d or %d > %d\n",
	    *w, iGlyphW, *h, iGlyphH);
#endif
    return 0;
  }

  // store glyph pixmap in iCache
  ret = 0;
  for (j = 0; j < iCacheAssoc; ++j) {
    if ((iCacheTags[i+j].mru & 0x7fff) == iCacheAssoc - 1) {
      iCacheTags[i+j].mru = 0x8000;
      iCacheTags[i+j].code = c;
      iCacheTags[i+j].x = *x;
      iCacheTags[i+j].y = *y;
      iCacheTags[i+j].w = *w;
      iCacheTags[i+j].h = *h;
      ret = iCache + (i+j) * iGlyphSize;
      /* The original Xpdf code assumed that Freetype padded rows up
	 to a byte.  This was apparently changed between FT 2.1.0
	 and FT 2.1.3, so we really have to check the pitch variable
	 of the FT bitmap. */
      int cacheRowSize = (*w + 7) >> 3;
      if (iFace->iEngine->iAa)
	cacheRowSize = *w;
      int pitch = slot->bitmap.pitch;
      if (pitch < 0) pitch = -pitch;
      Guchar *dst = ret;
      Guchar *src = slot->bitmap.buffer;
      for (int y = 0; y < *h; ++y) {
	memcpy(dst, src, cacheRowSize);
	dst += cacheRowSize;
	src += pitch;
      }
    } else {
      ++iCacheTags[i+j].mru;
    }
  }
  return ret;
}

//! Compute the glyph index in the face from the charcode or Unicode.
FT_UInt FTFont::GetGlyphIndex(CharCode c, Unicode u)
{
  FT_UInt idx;

  idx = 0; // make gcc happy
  switch (iFace->iMode) {
  case ftFontModeUnicode:
    idx = FT_Get_Char_Index(iFace->iFace, (FT_ULong)u);
    break;
  case ftFontModeCharCode:
    idx = FT_Get_Char_Index(iFace->iFace, (FT_ULong)c);
    break;
  case ftFontModeCharCodeOffset:
    idx = FT_Get_Char_Index(iFace->iFace,
			    (FT_ULong)(c + iFace->iCharMapOffset));
    break;
  case ftFontModeCodeMap:
    if (c <= 0xff) {
      idx = FT_Get_Char_Index(iFace->iFace, (FT_ULong)iFace->iCodeMap[c]);
    } else {
      idx = 0;
    }
    break;
  case ftFontModeCodeMapDirect:
    if (c <= 0xff) {
      idx = (FT_UInt)iFace->iCodeMap[c];
    } else {
      idx = 0;
    }
    break;
  case ftFontModeCIDToGIDMap:
    if (iFace->iCidToGIDLen) {
      if ((int)c < iFace->iCidToGIDLen) {
	idx = (FT_UInt)iFace->iCidToGID[c];
      } else {
	idx = (FT_UInt)0;
      }
    } else {
      idx = (FT_UInt)c;
    }
    break;
  case ftFontModeCFFCharset:
    // pdftex doesn't support CFF, no need to support it yet
    // (the xpdf code here depends on freetype internals)
#if 0 //~ cff cid->gid map
    CFF_Font cff = (CFF_Font)((TT_Face) iFace->iFace)->extra.data;
    idx = 0;
    for (j = 0; j < (int)cff->num_glyphs; ++j) {
      if (cff->charset.sids[j] == c) {
	idx = j;
	break;
      }
    }
#endif
    break;
  }
  return idx;
}

//! Draw a glyph.
/*! Glyph is drawn with hotspot at position (x,y) (on the pixmap) */
bool FTFont::DrawChar(QPixmap *pixmap, int x, int y,
		      QRgb rgb, CharCode c, Unicode u)
{
  FTFontEngine *engine = iFace->iEngine;
  int w = pixmap->width();
  int h = pixmap->height();

  // no Unicode index for this char - don't draw anything
  if (iFace->iMode == ftFontModeUnicode && u == 0)
    return false;

  // generate the glyph pixmap
  Guchar *p;
  int xOffset, yOffset, gw, gh;
  if (!(p = GetGlyphPixmap(c, u, &xOffset, &yOffset, &gw, &gh)))
    return false;

  // compute: (x0,y0) = position in destination pixmap
  //          (x1,y1) = position in glyph image
  //          (w0,h0) = size of image transfer
  int x0, y0, x1, y1, w0, h0;
  x0 = x - xOffset;
  y0 = y - yOffset;
  x1 = 0;
  y1 = 0;
  w0 = gw;
  h0 = gh;
  if (x0 < 0) {
    x1 = -x0;
    w0 += x0;
    x0 = 0;
  }
  if (x0 + w0 > w)
    w0 = w - x0;
  if (w0 < 0)
    return true;
  if (y0 < 0) {
    y1 = -y0;
    h0 += y0;
    y0 = 0;
  }
  if (y0 + h0 > h)
    h0 = h - y0;
  if (h0 < 0)
    return true;

  QPixmap dst(gw, gh);
  bitBlt(&dst, x1, y1, pixmap, x0, y0, w0, h0);
  QImage image = dst.convertToImage();

  if (engine->iAa) {
    int r = qRed(rgb);
    int g = qGreen(rgb);
    int b = qBlue(rgb);
    // stuff the glyph pixmap into the X image
    for (int yy = 0; yy < gh; ++yy) {
      uint *dst = (uint *) image.scanLine(yy);
      for (int xx = 0; xx < gw; ++xx) {
	int pix = *p++ & 0xff;
	int pix1 = 0xff - pix;
	if (pix > 0) {
	  uint bg = *dst;
	  int newr = (pix * r + pix1 * qRed(bg)) / 255;
	  int newg = (pix * g + pix1 * qGreen(bg)) / 255;
	  int newb = (pix * b + pix1 * qBlue(bg)) / 255;
	  *dst = qRgb(newr, newg, newb);
	}
	++dst;
      }
    }
  } else {
    // one color
    for (int yy = 0; yy < gh; ++yy) {
      uint *dst = (uint *) image.scanLine(yy);
      for (int xx = 0; xx < gw; xx += 8) {
	int pix = *p++;
	for (int xx1 = xx; xx1 < xx + 8 && xx1 < gw; ++xx1) {
	  if (pix & 0x80)
	    *dst = rgb;
	  pix <<= 1;
	  ++dst;
	}
      }
    }
  }
  dst.convertFromImage(image);
  bitBlt(pixmap, x0, y0, &dst, x1, y1, w0, h0);
  return true;
}

// --------------------------------------------------------------------
