//========================================================================
//
// ocfile.cpp
//
// Drop-in replacement for C <stdio.h> FILE 
//
//========================================================================

#include "ocfile.h"
#include <qfile.h>
#include <qstring.h>
#include <list>
#include <cassert>

// --------------------------------------------------------------------

// This function is from Xpdf. 
// It is here since the file that contained it originally is gone.
char *getLine(char *buf, int size, OCFILE *f)
{
  int c, i;

  i = 0;
  while (i < size - 1) {
    if ((c = fgetc(f)) == EOF) {
      break;
    }
    buf[i++] = (char)c;
    if (c == '\x0a') {
      break;
    }
    if (c == '\x0d') {
      c = fgetc(f);
      if (c == '\x0a' && i < size - 1) {
	buf[i++] = (char)c;
      } else if (c != EOF) {
	ungetc(c, f);
      }
      break;
    }
  }
  buf[i] = '\0';
  if (i == 0) {
    return NULL;
  }
  return buf;
}

// --------------------------------------------------------------------

/* All files ever opened are kept here. */

class OcFileStore {
public:  
  OCFILE *Open(const char *name, const char *mode);
  OCFILE *Open(QIODevice *device);
  ~OcFileStore();
private:
  std::list<OCFILE> iFiles;
};

static OcFileStore store;

OcFileStore::~OcFileStore()
{
  for (std::list<OCFILE>::iterator it = iFiles.begin(); 
       it != iFiles.end(); ++it) {
    if (it->iDevice) {
      delete it->iDevice;
      it->iDevice = 0;
    }
  }
}

OCFILE *OcFileStore::Open(const char *name, const char *mode)
{
  QFile *file = new QFile(QString::fromUtf8(name));
  int m = (mode[0] == 'w') ? IO_WriteOnly : IO_ReadOnly;
  if (!file->open(m)) {
    delete file;
    return 0;
  }
  iFiles.push_back(OCFILE());
  iFiles.back().iDevice = file;
  return &iFiles.back();
}

OCFILE *OcFileStore::Open(QIODevice *device)
{
  iFiles.push_back(OCFILE());
  iFiles.back().iDevice = device;
  return &iFiles.back();
}

// --------------------------------------------------------------------

extern OCFILE *ocfopen(const char *name, const char *mode)
{
  return store.Open(name, mode);
}

extern OCFILE *ocfopen(QIODevice *device)
{
  return store.Open(device);
}

extern void fclose(OCFILE *f)
{
  f->iDevice->close();
  f->iDevice = 0;
}

extern size_t fwrite(const void *buf, size_t size, size_t count, OCFILE *f)
{
  assert(size == 1);
  return f->iDevice->writeBlock((const char *) buf, count);
}

extern size_t fread(void *buf, size_t size, size_t count, OCFILE *f)
{
  assert(size == 1);
  return f->iDevice->readBlock((char *) buf, count);
}

extern int fseek(OCFILE *f, long offset, int whence)
{
  if (!f->iDevice)
    return -1;
  bool res;
  if (whence == SEEK_SET)
    res = f->iDevice->at(offset);
  else if (whence == SEEK_END)
    res = f->iDevice->at(offset + f->iDevice->size());
  else
    assert(false);
  return res ? 0 : -1;
}

extern long ftell(OCFILE *f)
{
  return f->iDevice->at();
}

extern int fgetc(OCFILE *f)
{
  return f->iDevice->getch();
}

extern int fputc(int ch, OCFILE *f)
{
  return f->iDevice->putch(ch);
}

extern void ungetc(int ch, OCFILE *f)
{
  f->iDevice->ungetch(ch);
}

extern void fputs(const char *s, OCFILE *f)
{
  f->iDevice->writeBlock(s, strlen(s));
}

extern void fprintf(OCFILE *f, const char *fmt, int a1, int a2)
{
  char buf[30];
  sprintf(buf, fmt, a1, a2);
  f->iDevice->writeBlock(buf, strlen(buf));
}

extern void fprintf(OCFILE *f, const char *fmt, const char *s)
{
  assert(strlen(fmt) + strlen(s) < 200);
  char buf[200];
  sprintf(buf, fmt, s);
  f->iDevice->writeBlock(buf, strlen(buf));
}

// --------------------------------------------------------------------
