// --------------------------------------------------------------------
// The Ipe document, with PDF support
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipemodel.h"

#include "ipeprefs.h"
#include "ipeq.h"

#include "ipelatex.h"
#include "ipestyle.h"
#include "ipeiml.h"

#include <QBuffer>
#include <QFile>
#include <QDir>
#include <QTextStream>
#include <QByteArray>

// --------------------------------------------------------------------

/*! \class IpeModel
  \brief Support for loading/saving/running Latex on IpeDocument.

  This class supports running pdflatex to generate the PDF output for
  text objects, and adds I/O using QFile's to IpeDocument.

  All members of the class are static.

 */

IpeDocument *IpeModel::New(const QString &fname, int &reason)
{
  QFile f(fname);
  reason = -1;
  if (!f.open(QIODevice::ReadOnly))
    return 0;
  QDataSource source(&f);
  IpeDocument::TFormat format = IpeDocument::FileFormat(source);
  source.reset();
  IpeDocument *self = IpeDocument::New(source, format, reason);
  f.close();
  return self;
}

bool IpeModel::Save(const IpeDocument *doc, const QString &fname,
		    IpeString creator, IpeDocument::TFormat format,
		    uint flags)
{
  QFile f(fname);
  if (!f.open(QIODevice::WriteOnly))
    return 0;
  IpeQStream stream(&f);
  bool result = doc->Save(stream, creator, format, flags, -1, -1,
			  IpePreferences::Static()->iCompressLevel);
  stream.flush();
  f.close();
  return result;
}

bool IpeModel::AddStyleSheet(IpeDocument *doc, const QString &fname)
{
  QFile file(fname);
  if (!file.open(QIODevice::ReadOnly))
    return false;
  QDataSource source(&file);
  bool result = doc->AddStyleSheet(source);
  file.close();
  return result;
}

static IpeStyleSheet *updateStyleSheet(IpeStyleSheet *sheet,
				       const QDir &dir,
				       IpeRepository *rep,
				       IpeString &log)
{
  IpeStyleSheet *cascade = sheet->Cascade();
  if (cascade) {
    IpeStyleSheet *nCascade = updateStyleSheet(cascade, dir, rep, log);
    sheet->SetCascade(nCascade);
  }
  if (sheet->IsStandard()) {
    log.append("Standard stylesheet\n");
    return sheet;
  }
  if (sheet->Name().empty()) {
    log.append("Unnamed stylesheet\n");
    return sheet;
  }
  // else try to load it
  QString fn = QIpe(sheet->Name()) + QLatin1String(".isy");
  if (!dir.exists(fn)) {
    log.append("'");
    log.append(sheet->Name());
    log.append("' not found\n");
    return sheet;
  }
  QFile file(dir.filePath(fn));
  if (!file.open(QIODevice::ReadOnly)) {
    log.append("'");
    log.append(sheet->Name());
    log.append("' cannot open file '");
    log.append(IpeQ(file.fileName()));
    log.append("'\n");
    return sheet;
  }
  QDataSource source(&file);
  IpeImlParser parser(source, rep);
  IpeStyleSheet *nSheet = parser.ParseStyleSheet();
  if (!nSheet) {
    log.append("'");
    log.append(sheet->Name());
    log.append("' cannot parse file '");
    log.append(IpeQ(file.fileName()));
    log.append("'\n");
    return sheet;
  }
  log.append("'");
  log.append(sheet->Name());
  log.append("' loaded from '");
  log.append(IpeQ(file.fileName()));
  log.append("'\n");
  nSheet->SetCascade(sheet->Cascade());
  sheet->SetCascade(0);
  delete sheet;
  return nSheet;
}

//! Update all style sheets in document from stylesheet file
/*! Looks for files in directory \a dir whose name matches the name of
  the style sheet and has the extension 'isy'.
  \returns a string describing what has been done.
*/
IpeString IpeModel::UpdateStyleSheets(IpeDocument *doc, const QDir &dir)
{
  IpeStyleSheet *top = doc->GetStyleSheet();
  IpeString log;
  IpeStyleSheet *ntop = updateStyleSheet(top, dir, doc->Repository(), log);
  doc->SetStyleSheet(ntop);
  // update all references to the new sheet
  IpeAttributeSeq seq;
  doc->CheckStyle(seq);
  return log;
}

// --------------------------------------------------------------------

//! Run PdfLatex
/*! Prepend \a styleDirs to Latex include path. */
int IpeModel::RunLatex(IpeDocument *doc, bool needLatex,
		       const QStringList &, QString &texLog)
{
  IpePreferences *prefs = IpePreferences::Static();
  texLog = QString::null;
  IpeLatex converter(doc->StyleSheet());

  IpeAttribute bgSym = doc->Repository()->
    MakeSymbol(IpeAttribute::ETemplate, "Background");
  const IpeObject *background = doc->StyleSheet()->FindTemplate(bgSym);
  if (background)
    converter.ScanObject(background);

  int count = 0;
  for (int i = 0; i < doc->pages(); ++i)
    count = converter.ScanPage(doc->page(i));
  if (count == 0)
    return ErrNoText;

  // First we need a directory
  QDir temp = QDir(prefs->iLatexDir);
  if (!temp.exists()) {
    // create Pdflatex directory if it doesn't exist
    QDir().mkdir(temp.absolutePath());
    temp = QDir(prefs->iLatexDir);
    if (!temp.exists())
      return ErrNoDir;
  }

  QString texFile = temp.filePath(QLatin1String("text.tex"));
  QString pdfFile = temp.filePath(QLatin1String("text.pdf"));
  QString logFile = temp.filePath(QLatin1String("text.log"));
  QFile::remove(logFile);

  QFile file(texFile);
  if(!file.open(QIODevice::WriteOnly))
    return -1;
  IpeQStream stream(&file);
  int err = converter.CreateLatexSource(stream, doc->Properties().iPreamble);
  stream.flush();
  file.close();

  if (err < 0)
    return ErrWritingSource;
  if (err == 0 && !needLatex)
    return ErrAlreadyHaveForm;
  // Latex source has been prepared correctly

#if 0
  // set up TEXINPUTS (but special handling for MikTeX below)
  QByteArray ti = "TEXINPUTS=";
  bool putTi = false;
  if (styleDirs.count() > 0) {
    putTi = true;
    for (int i = 0; i < styleDirs.count(); ++i)
      ti.append((styleDirs[i] + QLatin1String(":")).toLocal8Bit());
  }
  if (!prefs->iTexInputs.empty()) {
    putTi = true;
    ti.append(prefs->iTexInputs.CString());
  }
  // setting TEXINPUTS breaks MikTeX 2.5
  if (putTi)
    putenv(qstrdup(ti.data()));
#endif

#ifdef WIN32
  QString s = temp.filePath(QLatin1String("runlatex.bat"));
  QFile f(s);
  f.open(QIODevice::WriteOnly);
  QTextStream batFile(&f);
  QString dirPath = QDir::convertSeparators(temp.path());
  if (dirPath.length() > 2 && dirPath[1] == QLatin1Char(':'))
    batFile << dirPath.left(2) << "\r\n";
  batFile << "cd \"" << dirPath << "\"\r\n";
  batFile << "\"" << prefs->iPdfLatex << "\"";
#if 0
  if (prefs->iMikTeX) {
    for (int i = 0; i < styleDirs.count(); ++i) {
      batFile << " -include-directory=\""
	     << QDir::convertSeparators(styleDirs[i]) << "\"";
    }
  }
#endif
  batFile << " text.tex\r\n";
  f.close();
  s = QLatin1String("call \"") + QDir::convertSeparators(s)
    + QLatin1String("\"");
  system(s.toLocal8Bit());
#else
  QByteArray s = "cd " + QFile::encodeName(temp.path()) + "; "
    + QFile::encodeName(prefs->iPdfLatex) + " text.tex";
  int res = system(s.data());
  (void) res;
#endif
  // Check log file for Pdflatex version and errors
  QFile log(logFile);
  if (!log.open(QIODevice::ReadOnly))
    return ErrLatex;
  texLog = QLatin1String(log.readAll());
  log.close();
  if (texLog.left(14) != QLatin1String("This is pdfTeX") &&
      texLog.left(15) != QLatin1String("This is pdfeTeX"))
    return ErrLatex;
  prefs->iMikTeX = (texLog.left(80).contains(QLatin1String("MikTeX")));
  int i = texLog.indexOf(QLatin1Char('-'));
  if (i < 0)
    return ErrLatex;
  QString version = texLog.mid(i+1, 30);
  ipeDebug("pdfTeX version %s", version.toLatin1().constData());
  if (version[0] == QLatin1Char('1') &&
      QLatin1Char('0') <= version[1] && version[1] <= QLatin1Char('3'))
    return ErrOldPdfLatex;
  if (version.left(2) == QLatin1String("14") &&
      version.left(3) < QLatin1String("14f"))
    return ErrOldPdfLatex;
  // Check for error
  if (texLog.contains(QLatin1String("\n!")))
    return ErrLatex;

  QFile pdfF(pdfFile);
  if (!pdfF.open(QIODevice::ReadOnly))
    return ErrLatex;
  QDataSource source(&pdfF);
  if (converter.ReadPdf(source) && converter.UpdateTextObjects()) {
    doc->SetFontPool(converter.TakeFontPool());
    ipeDebug("Pdflatex run completed successfully");
    return ErrNone;
  }
  return ErrLatexOutput;
}

//! Run Pdflatex (suitable for console applications)
/*! Success/error is reported on stderr. */
int IpeModel::RunLatex(IpeDocument *doc)
{
  QString logFile;
  QStringList s;
  s.append(QDir::currentPath());
  switch (RunLatex(doc, true, s, logFile)) {
  case IpeModel::ErrNoText:
    fprintf(stderr, "No text objects in document, no need to run Pdflatex\n");
    return 0;
  case IpeModel::ErrNoDir:
    fprintf(stderr, "Directory '%s' does not exist and cannot be created\n",
	    ((const char *)
	     IpePreferences::Static()->iLatexDir.toLocal8Bit()));
    return 1;
  case IpeModel::ErrWritingSource:
    fprintf(stderr, "Error writing Latex source.\n");
    return 1;
  case IpeModel::ErrOldPdfLatex:
    fprintf(stderr, "Your installed version of Pdflatex is too old.\n");
    return 1;
  case IpeModel::ErrLatex:
    fprintf(stderr, "There were errors trying to run Pdflatex\n");
    return 1;
  case IpeModel::ErrNone:
  default:
    fprintf(stderr, "Pdflatex was run sucessfully.\n");
    return 0;
  }
}

// --------------------------------------------------------------------
