// -*- C++ -*-
// --------------------------------------------------------------------
// IpeCanvas
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPECANVAS_H
#define IPECANVAS_H

#include "ipegeo.h"
#include "ipecolor.h"
#include "ipesnap.h"
#include "ipepainter.h"
#include "ipefonts.h"

#include <QWidget>
// #include <QPixmap>
#include <QImage>

// --------------------------------------------------------------------

class IpeStyleSheet;
class IpePage;
class IpeOverlay;

class IpePdfObj;
class IpeFontPool;

class QMouseEvent;
class QPainter;

// --------------------------------------------------------------------

/*! \class IpeCanvasServices
  \brief Interface for canvas callbacks.
*/

class IpeCanvasServices : public IpeFontsServices {
public:
  virtual ~IpeCanvasServices() { /* nothing */ } // placate compiler
  //! Request an IpeOverlay object because of this mouse press.
  virtual void CvSvcRequestOverlay(QMouseEvent *ev) = 0;
  //! Notify application whether a drawing is active.
  virtual void CvSvcSetDrawingMode(bool mode) = 0;
  //! Notify application of change in zoom by wheel mouse.
  virtual void CvSvcWheelZoom(int delta) = 0;
  //! Notify application of current mouse position (in user coordinates).
  virtual void CvSvcMousePosition(const IpeVector &pos) = 0;
  //! Clear status bar message.
  virtual void CvSvcClearMessage() = 0;
};

// --------------------------------------------------------------------

class IpeCanvas : public QWidget {
  Q_OBJECT

public:
  IpeCanvas(IpeCanvasServices *services, bool doubleBuffer,
	    QWidget* parent, int bitmapSize, Qt::WFlags f=0);
  ~IpeCanvas();

  void SetPage(const IpePage *page, int view, const IpeStyleSheet *sheet,
	       IpeAttribute pageColor);
  void SetFontPool(const IpeFontPool *fontPool);
  void SetAntiAlias(bool antiAlias);
  void SetPan(const IpeVector &v);
  void SetZoom(double zoom);
  void SetSnap(const IpeSnapData &s);
  void SetDimmed(bool dimmed);
  void SetAutoOrigin(const IpeVector &v);
  IpeMatrix CanvasTfm() const;

  //! Return current pan.
  inline IpeVector Pan() const { return iPan; }
  //! Return current zoom.
  inline double Zoom() const { return iZoom; }
  //! Return current style sheet.
  inline const IpeStyleSheet *StyleSheet() const { return iStyleSheet; }
  //! Return current media box.
  // inline IpeRect MediaBox() const { return iMediaBox; }

  //! Return center of canvas.
  inline IpeVector Center() const {
    return 0.5 * IpeVector(width(), height()); }

  //! Return last mouse position (snapped!) in user coordinates.
  inline IpeVector Pos() const { return iMousePos; }

  //! Return last unsnapped mouse position in user coordinates.
  inline IpeVector UnsnappedPos() const { return iUnsnappedMousePos; }

  //! Return current snapping information.
  inline const IpeSnapData &SnapData() const { return iSnapData; }

  void Update();
  void UpdateOverlay();

  void SetOverlay(IpeOverlay *overlay);
  inline IpeOverlay *Overlay() const { return iOverlay; }
  void FinishOverlay();

  IpeVector DevToUser(const IpeVector &arg) const;
  IpeVector UserToDev(const IpeVector &arg) const;

protected:
  // virtual bool event(QEvent *ev);
  virtual void resizeEvent(QResizeEvent *ev);
  virtual void paintEvent(QPaintEvent *ev);
  virtual void mousePressEvent(QMouseEvent *ev) ;
  virtual void mouseReleaseEvent(QMouseEvent *ev);
  virtual void mouseMoveEvent(QMouseEvent *ev);
  virtual void wheelEvent(QWheelEvent *ev);
  virtual QSize sizeHint() const;

private:
  void DrawGrid(QPainter &qPainter);
  void RepaintObjects();
  void ComputeFifi(QMouseEvent *ev);
  void DrawAxes(QPainter &qPainter);
  void SetOrigin();
  bool snapToPaperAndFrame();

private:
  const IpePage *iPage;
  int iView;
  const IpeStyleSheet *iStyleSheet;
  IpeCanvasServices *iServices;
  IpeAttribute iPageColor;
  int iBitmapSize;

  IpeVector iPan;
  double iZoom;
  IpeSnapData iSnapData;
  bool iDimmed;
  bool iAutoSnap;
  bool iAntiAlias;
  IpeVector iAutoOrigin;

  IpeOverlay *iOverlay;

  bool iRepaintObjects;
  // QPixmap iOffscreen;
  QImage iOffscreen;
  IpeVector iUnsnappedMousePos;
  IpeVector iMousePos;
  QPointF iFifi;
  bool iFifiVisible;

  IpeCanvasFonts *iFonts;
};

// --------------------------------------------------------------------
#endif
