// --------------------------------------------------------------------
// incircle.cpp: Ipelet for computing the incircle of a triangle
//
// Author: Jari Lappalainen
// Date: April 14th 2006
//
// Description: Finds the incircle of a triangle and adds it to the page.
// Algorithm is based on the geometric observation that the angle bisectors
// intersect at the incentre.
//
// This ipelet is based on the smallest-circle ipelet by Chris Gray.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipelet.h"
#include "ipepath.h"
#include "ipepage.h"
#include "ipevisitor.h"
#include "ipemark.h"

#include <set>
#include <sstream>

using namespace std;

static const char *aboutText =
"Written by Jari Lappalainen";

class EuclidIpelet : public Ipelet {
public:
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  int NumFunctions() const { return 2; }
  virtual const char* Label() const { return "Euclidean Geometry"; }
  const char* SubLabel(int fCount) const
  {
    switch (fCount) {
      case 0:
	return "Incircle of Triangle";
      case 1:
	return "Excircles of Triangle";
    }
    return ""; // Never get here
  }
  virtual void Run(int, IpePage *page, IpeletHelper *helper);
  virtual const char *About() const { return aboutText; }
};

namespace {
  struct vector_comparator :
    public binary_function<IpeVector,IpeVector,bool> {
    bool operator()(const IpeVector& v1, const IpeVector& v2) const {
      if (v1.iX == v2.iX) {
	return v1.iY < v2.iY;
      }
      return v1.iX < v2.iX;
    }
  };

  typedef set<IpeVector,vector_comparator> VS_t;

  VS_t collect_vertices(IpePage* page)
  {
    VS_t points;
    for (IpePage::iterator it = page->begin(); it != page->end(); ++it) {
      if (it->Select() && it->Object()) {
	IpeMatrix m = it->Object()->Matrix();
	if (it->Object()->AsMark()) {
	  points.insert(m * it->Object()->AsMark()->Position());
	}
	else if (it->Object()->AsPath()) {
	  for (int j = 0; j<it->Object()->AsPath()->NumSubPaths(); ++j) {
	    const IpeSubPath* sp = it->Object()->AsPath()->SubPath(j);
	    if (sp->AsSegs()) {
	      for (int k = 0; k<sp->AsSegs()->NumSegments(); k++) {
		points.insert(m * sp->AsSegs()->Segment(k).CP(0));
		points.insert(m * sp->AsSegs()->Segment(k).CP(1));
	      }
	    }
	  }
	}
      }
    }
    return points;
  }

  IpeLine angle_bisector(const IpeVector& origin,
			 const IpeVector& dir1,
			 const IpeVector& dir2)
  {
    assert(dir1.SqLen() > 0);
    assert(dir2.SqLen() > 0);
    IpeVector bisector = dir1.Normalized() + dir2.Normalized();
    if (bisector.Len() == 0) {
      bisector = dir1.Orthogonal();
    }
    return IpeLine::Through(origin, origin + bisector);
  }

  void incircle(const IpeVector& a, const IpeVector& b,
		const IpeVector& c, IpeScalar& inradius, IpeVector& incentre)
  {
    IpeLine b1 = angle_bisector(a, b - a, c - a);
    IpeLine b2 = angle_bisector(b, c - b, a - b);
    if (b1.Intersects(b2, incentre)) {
      IpeLine AB = IpeLine::Through(a, b);
      inradius = AB.Distance(incentre);
    }
    else {
      inradius = 0;
      incentre = c;
    }
  }

  void excircle(const IpeVector& a, const IpeVector& b,
		const IpeVector& c, IpeScalar& radius, IpeVector& centre)
  {
    IpeLine b1 = angle_bisector(a, b - a, c - a);
    IpeLine b2 = angle_bisector(b, c - b, a - b);
    IpeVector n1 = b1.Normal();
    IpeVector n2 = b2.Normal();
    IpeLine nl1 = IpeLine::Through(a, a + n1);
    IpeLine nl2 = IpeLine::Through(b, b + n2);
    if (nl1.Intersects(nl2, centre)) {
      IpeLine AB = IpeLine::Through(a, b);
      radius = AB.Distance(centre);
    }
    else {
      // Should really be a line through a, b and c
      centre = c;
      radius = 0;
    }
  }

  void add_circle(IpeScalar radius, const IpeVector& centre, IpePage* page,
		  IpeletHelper* helper)
  {
    // Protect against ipe crash
    const double IpeLimit(10000);
    if (radius > IpeLimit
	&& fabs(centre.iX) > IpeLimit
	&& fabs(centre.iY) > IpeLimit) {
      return;
    }

    IpeMatrix m(radius, 0.0, 0.0, radius, centre.iX, centre.iY);
    IpeEllipse* ellipse = new IpeEllipse(m);
    IpePath* path = new IpePath(helper->Attributes());
    path->AddSubPath(ellipse);
    page->push_back(IpePgObject(IpePgObject::EPrimary,
				helper->CurrentLayer(), path));
  }

  void RunIncircle(IpePage *page, IpeletHelper *helper)
  {
    VS_t vertices = collect_vertices(page);
    if (vertices.size() != 3) {
      ostringstream ost;
      ost << "selection with " << vertices.size()
	  << " points is not a triangle";
      helper->Message(ost.str().c_str());
    }
    else {
      IpeVector a = *(vertices.begin());
      IpeVector b = *(++vertices.begin());
      IpeVector c = *(++++vertices.begin());
      IpeScalar inradius(0);
      IpeVector incentre;
      incircle(a, b, c, inradius, incentre);
      add_circle(inradius, incentre, page, helper);

      ostringstream ost;
      ost << "Created incircle with radius " << inradius
	  << " and centre " << incentre.iX << "," << incentre.iY;
      helper->Message(ost.str().c_str());
    }
  }

  void RunExcircles(IpePage *page, IpeletHelper *helper)
  {
    VS_t vertices = collect_vertices(page);
    if (vertices.size() != 3) {
      ostringstream ost;
      ost << "selection with " << vertices.size()
	  << " vertices is not a triangle";
      helper->Message(ost.str().c_str());
    }
    else {
      IpeVector a = *(vertices.begin());
      IpeVector b = *(++vertices.begin());
      IpeVector c = *(++++vertices.begin());
      IpeScalar radius(0);
      IpeVector centre;
      excircle(a, b, c, radius, centre);
      add_circle(radius, centre, page, helper);
      excircle(b, c, a, radius, centre);
      add_circle(radius, centre, page, helper);
      excircle(c, a, b, radius, centre);
      add_circle(radius, centre, page, helper);

      ostringstream ost;
      ost << "Created excircles";
      helper->Message(ost.str().c_str());
    }
  }
}

void EuclidIpelet::Run(int fCount, IpePage *page, IpeletHelper *helper)
{
  switch (fCount) {
  case 0:
    RunIncircle(page, helper);
    break;
  case 1:
    RunExcircles(page, helper);
    break;
  }
}

// --------------------------------------------------------------------

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new EuclidIpelet;
}

// --------------------------------------------------------------------
