// --------------------------------------------------------------------
// Ipelet for aligning objects
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipelet.h"
#include "ipepath.h"
#include "ipetext.h"
#include "ipepage.h"
#include "ipevisitor.h"

// --------------------------------------------------------------------

class AlignIpelet : public Ipelet {
public:
  AlignIpelet() : iSkip(0.0) { /* nothing */ }
  virtual int IpelibVersion() const { return IPELIB_VERSION; }
  virtual int NumFunctions() const { return 13; }
  virtual const char *Label() const { return "Align"; }
  virtual const char *SubLabel(int function) const;
  virtual void Run(int, IpePage *page, IpeletHelper *helper);
private:
  void simpleAlign(int fn, IpePage *page, IpeletHelper *helper);
  void sequenceAlign(int fn, IpePage *page, IpeletHelper *helper);
  void topToBottom(std::vector<IpePage::iterator> &sel, double skip);
  void leftToRight(std::vector<IpePage::iterator> &sel, double skip);
  double iSkip;
};

// --------------------------------------------------------------------

static const char * const sublabel[] = {
  "Top",
  "Bottom",
  "Left",
  "Right",
  "Center",
  "H Center",
  "V Center",
  "Baseline",
  "Left-to-right",
  "Left and right",
  "Top-to-bottom",
  "Top and bottom",
  "Set skip..."
};

// 1 : require horizontal movement
// 2 : require vertical movement
static char movementRequirement[] = { 2, 2, 1, 1, 3, 1, 2, 2, 1, 1, 2, 2 };

const char *AlignIpelet::SubLabel(int function) const
{
  return sublabel[function];
}

// --------------------------------------------------------------------

void AlignIpelet::Run(int fn, IpePage *page, IpeletHelper *helper)
{
  if (fn == 12) {
    IpeString str;
    IpeStringStream ss(str);
    ss << iSkip;
    if (!helper->GetString("Enter skip in points", str))
      return;
    iSkip = IpeLex(str).GetDouble();
    return;
  }

  IpePage::iterator it = page->PrimarySelection();
  if (it == page->end()) {
    helper->Message("Nothing selected");
    return;
  }

  // check pinning
  int count = 0;
  unsigned int requirement = movementRequirement[fn];
  for (IpePage::iterator it1 = page->begin(); it1 != page->end(); ++it1) {
    if (it1->Select() == IpePgObject::ESecondary ||
	(it1->Select() == IpePgObject::EPrimary && fn >= 8)) {
      ++count;
      if (it1->Object()->pinned() & requirement) {
	helper->Message("Some object is pinned and cannot be moved");
	return;
      }
    }
  }

  if (!count)
    helper->Message("No objects to align");

  if (fn < 8)
    simpleAlign(fn, page, helper);
  else
    sequenceAlign(fn, page, helper);
}

void AlignIpelet::simpleAlign(int fn, IpePage *page, IpeletHelper *)
{
  IpePage::iterator it = page->PrimarySelection();
  IpeRect pbox = it->BBox();
  IpeVector pref = pbox.Min();
  if (it->Object()->AsText())
    pref = it->Object()->Matrix() * it->Object()->AsText()->Position();

  for (IpePage::iterator it1 = page->begin(); it1 != page->end(); ++it1) {
    if (it1->Select() == IpePgObject::ESecondary) {
      IpeRect box = it1->BBox();
      IpeVector v(0, 0);
      IpeVector ref = box.Min();
      if (it1->Object()->AsText())
	ref = it1->Object()->Matrix() * it1->Object()->AsText()->Position();
      switch (fn) {
      case 0: // top
	v.iY = pbox.Max().iY - box.Max().iY;
	break;
      case 1: // bottom
	v.iY = pbox.Min().iY - box.Min().iY;
	break;
      case 2: // left
	v.iX = pbox.Min().iX - box.Min().iX;
	break;
      case 3: // right
	v.iX = pbox.Max().iX - box.Max().iX;
	break;
      case 4: // center
	v.iX = 0.5 * ((pbox.Min().iX + pbox.Max().iX) -
		      (box.Min().iX + box.Max().iX));
	v.iY = 0.5 * ((pbox.Min().iY + pbox.Max().iY) -
		      (box.Min().iY + box.Max().iY));
	break;
      case 5: // h center
	v.iX = 0.5 * ((pbox.Min().iX + pbox.Max().iX) -
		      (box.Min().iX + box.Max().iX));
	break;
      case 6: // v center
	v.iY = 0.5 * ((pbox.Min().iY + pbox.Max().iY) -
		      (box.Min().iY + box.Max().iY));
	break;
      case 7: // baseline
	v.iY = pref.iY - ref.iY;
	break;
      }
      it1->Transform(IpeMatrix(v));
    }
  }
}

class Comparer {
public:
  Comparer(bool vertical) : iVertical(vertical) { /* nothing else */ }
  bool operator()(const IpePage::iterator &lhs,
		  const IpePage::iterator &rhs);
private:
  bool iVertical;
};

bool Comparer::operator()(const IpePage::iterator &lhs,
			 const IpePage::iterator &rhs)
{
  return iVertical ?
    (lhs->BBox().Max().iY > rhs->BBox().Max().iY) :
    (lhs->BBox().Min().iX < rhs->BBox().Min().iX);
}

void AlignIpelet::leftToRight(std::vector<IpePage::iterator> &sel, double skip)
{
  double xtarget = sel[0]->BBox().Max().iX + skip;
  for (uint i = 1; i < sel.size(); ++i) {
    double xmov = xtarget - sel[i]->BBox().Min().iX;
    sel[i]->Transform(IpeMatrix(IpeVector(xmov, 0.0)));
    xtarget = sel[i]->BBox().Max().iX + skip;
  }
}

void AlignIpelet::topToBottom(std::vector<IpePage::iterator> &sel, double skip)
{
  double ytarget = sel[0]->BBox().Min().iY - skip;
  for (uint i = 1; i < sel.size(); ++i) {
    double ymov = ytarget - sel[i]->BBox().Max().iY;
    sel[i]->Transform(IpeMatrix(IpeVector(0.0, ymov)));
    ytarget = sel[i]->BBox().Min().iY - skip;
  }
}

void AlignIpelet::sequenceAlign(int fn, IpePage *page, IpeletHelper *)
{
  std::vector<IpePage::iterator> sel;
  for (IpePage::iterator it = page->begin(); it != page->end(); ++it) {
    if (it->Select())
      sel.push_back(it);
  }
  std::sort(sel.begin(), sel.end(), Comparer(fn >= 10));

  switch (fn) {
  case 8:
    leftToRight(sel, iSkip);
    break;
  case 9:
    {
      double totalWidth = 0.0;
      for (uint i = 0; i < sel.size(); ++i)
	totalWidth += sel[i]->BBox().Width();
      double skip = (sel.back()->BBox().Max().iX - sel.front()->BBox().Min().iX
		     - totalWidth) / (sel.size() - 1);
      leftToRight(sel, skip);
      break;
    }
  case 10:
    topToBottom(sel, iSkip);
    break;
  case 11:
    {
      double totalHeight = 0.0;
      for (uint i = 0; i < sel.size(); ++i)
	totalHeight += sel[i]->BBox().Height();
      double skip = (sel.front()->BBox().Max().iY - sel.back()->BBox().Min().iY
		     - totalHeight) / (sel.size() - 1);
      topToBottom(sel, skip);
      break;
    }
  default:
    break;
  }
}

// --------------------------------------------------------------------

IPELET_DECLARE Ipelet *NewIpelet()
{
  return new AlignIpelet;
}

// --------------------------------------------------------------------
