#
# iPodder based on the iSpider engine
#
# Periodically retrieves an rss file parses out the enclosures and downloads the content
# starts torrent if necessary. Downloaded content is archived on disk and managed
# by the a mediaplayer which updates the portable music device.
#
# credits Adam Curry,
#         Andrew Grumet,
#         Erik de Jonge,
#         Garth Kidd.
#         Aaron Mitti,
#         Ray Slakinski,
#         Perica Zivkovic,
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import platform
import sys
import urllib
import httplib
import os
from os.path import *
from string import join
import time
import re
from sha import *
import feedparser
from threading import Event
import logging, conlogging
import pickle, shelve
import socket # just so we can catch timeout

# Parts of iPodder
import configuration
import feeds
import hooks
import urlnorm
import downloaders

__version__ = configuration.__version__

log = logging.getLogger('iPodder')

def PLEASEREPORT():
    log.exception("Please report this traceback:")

class iPodder:
    def makesafeurl(self, url):
        "PENDING DEPRECATION: make a URL safe."
        return urlnorm.normalize(url)

    def feedstoscan(self, allfeeds=None): 
        "Look for feeds to scan."
        log.info("Figuring out which feeds to scan...")
        feeds = []
        now = time.time()
        politenesss = self.config.politeness * 60 # in seconds
        if allfeeds is None: 
            allfeeds = self.feeds
        for feedinfo in allfeeds: 
            if feedinfo.sub_state in ('unsubscribed', 'disabled'): 
                continue
            if feedinfo.sub_state in ('preview'): 
                log.warn("Feed sub_state %s not implemented yet; "\
                         "skipping feed %s", feedinfo)
                continue
            if feedinfo.checked is not None: 
                delta = now - time.mktime(feedinfo.checked)
                if delta < 0: # Daylight Savings? :)
                    delta = politenesss 
                log.debug("Check delta is %d versus politenesss of %d", 
                          delta, politenesss)
                if delta < politenesss: 
                    log.warn("It'd be impolite to scan %s again only "\
                             "%d seconds since we last did it.", feedinfo, 
                             delta)
                    continue
            feeds.append(feedinfo)
        return feeds
        
    def scanfeeds(self, feeds=None): 
        "Scan feeds looking for torrents and enclosures."
                
        if feeds is None: 
            feeds = self.feedstoscan()

        log.info("Downloading feeds and looking for enclosures...")
        
        # scan-enclosures-begin hooks get no arguments
        # scan-enclosures-announce hooks get feedinfo
        # scan-enclosures-count hooks get encnum, maxnum
        # encnum == maxnum on the last call
        # scan-enclosures-end hooks get no arguments
        beginner = self.hooks.get('scan-enclosures-begin')
        announcer = self.hooks.get('scan-enclosures-announce')
        backannouncer = self.hooks.get('scan-enclosures-backannounce')
        counter = self.hooks.get('scan-enclosures-count')
        ender = self.hooks.get('scan-enclosures-end')

        beginner() # hooked
        countmax = len(feeds)
        count = 0
        enclosures = []
      
        for feed in feeds: 
            counter(count, countmax) # hooked
            count = count + 1
            announcer(feed)
            enclosures.extend(self._scanfeed(feed))
            backannouncer(feed)
            
        counter(count, countmax) 
        ender()
        return enclosures
            
    def _scanfeed(self, feed): 
        """Scan a feed looking for enclosures.
        Returns a list of enclosures."""
        etag = feed.etag
        modified = feed.modified

        if self.config.force_playlist_updates or feed.sub_state == 'force':
            etag = modified = '' # needs its own option, but very useful
            if feed.sub_state == 'force':
                log.warn("Feed id %d (%s) is forced. You should turn "\
                         "that off once it's fixed whatever you turned "\
                         "it on to fix.", feed.id, feed)

        safe_url = self.makesafeurl(feed.url)
        feed.checked = time.localtime()

        results = downloaders.grabfeed(safe_url, etag, modified)
        if results is None:
            feed.status = -1 # flag as an error
            feed.half_flush()
            return []

        status = feed.status = results['status']

        if status == 200: 
            log.debug("Successfully downloaded and parsed.")
            return self._mineresults(feed, results)
        
        elif status == 304:
            # We *could* munch on saved RSS, but we're not saving it yet.
            log.info("That hasn't changed since we last checked.")
            
        elif status == 404:
            log.error("That doesn't exist any more (404).")

        else: 
            log.error("Unknown feed status %d.", feed.status)

        return [] # bummer!
        
    def _mineresults(self, feed, results): 
        """Mine feedparser.parse results for enclosures."""

        # Update the state database
        for key in ['etag', 'modified', 'tagline', 'generator', 'link',
                    'lastbuilddate', 'copyright']:
            setattr(feed, key, results.get(key, ''))
        for key in ['copyright_detail', 'title_detail']:
            setattr(feed, key, dict(results.get(key, {})))
        feed.version = results.get('version', 'unknown')
        channeltitle = results.feed.title
        flush = feed.half_flush
        if not feed.title:
            log.info("Naming feed #%d \"%s\"", feed.id, channeltitle)
            feed.title = channeltitle
            flush = feed.flush # very intensive
        elif feed.title != channeltitle:
            log.info("Feed %d renamed \"%s\"", feed.id, channeltitle)
            feed.title = channeltitle
        flush()

        enclosures = []
        
        # Look for enclosures
        entrynum = 0
        for entry in results.entries:
            entrynum = entrynum + 1
            enclosures.extend(self._mineresultentry(feed, entrynum, entry))

        return enclosures

    def _mineresultentry(self, feed, entrynum, entry): 
        """Mine an entry from a feedparser.parse result.
        Return a list of enclosures."""
        return [] 
        
    def getMetaInfoTorrentFile(self, metainfo_name):
        "TODO: DEPRECATE."
        return torrents.TorrentMetaInfo(metainfo_name)

    def progress(self, block_count, block_size, total_size):

        #Internal stats must be set for both console and GUI.
        self.m_f_downloaded_size = block_count*block_size;
        self.m_f_total_size = total_size;

        if self.ui_progress:
            self.ui_progress(block_count, block_size, total_size)

    def console_progress(self, block_count, block_size, total_size):

            print self.m_download_file + " - %.2f MB of %.2f MB\r" % (float(block_count*block_size)/1000000, float(total_size)/1000000),
            sys.stdout.flush()

    def downloadcontent(self):
        log.info("Pass #2: downloading enclosures...")
        
        # download-content-begin hooks get no arguments
        beginner = self.hooks.get('download-content-begin')
        
        # download-content-announce hooks get encinfo
        # encinfo is *not* None on the last call anymore
        announcer = self.hooks.get('download-content-announce')
        backannouncer = self.hooks.get('download-content-backannounce')
        
        # download-content-downloaded hooks get encinfo, destfile
        # (and thus all feed attrs via encinfo.feed)
        downloadedhooks = self.hooks.get('download-content-downloaded')
        
        # download-content-count hooks get encnum, maxnum
        # encnum == maxnum on the last call
        counter = self.hooks.get('download-content-count')

        # download-content-end hooks get no arguments
        ender = self.hooks.get('download-content-end')
        
        # Step #1: figure out what to download
        enclosures = [encinfo for encinfo in self.m_enclosures]

        if self.config.dry_run: 
            log.warn("dry_run set: no enclosures will be downloaded.")
            enclosures = []

        # Step #2: scan
        beginner()
        countmax = len(enclosures)
        count = 0
        for encinfo in enclosures:
            counter(count, countmax)
            count = count + 1
            announcer(encinfo)
            try:
                update_playlist=True
                self.m_f_downloaded_size = -1
                self.m_f_total_size = 0;
                self.m_download_complete=False
                first_time_skip = False

                torrentfile = False
                enc = encinfo.url
                feed = encinfo.feed
                if not enc:
                    # It happened before; it might happen again.
                    log.error("An enclosure URL for channel %s is empty.",
                                        encinfo.feed)
                    continue


                encsplit = enc.split('/')
                filename = encsplit[-1]
                name, ext = os.path.splitext(filename)

                if ext == 'torrent':
                    torrentfile = True

                historic = self.m_downloadlog.__contains__(filename)
                if historic \
                and not feed.sub_state == 'force':
                    #and not self.config.force_playlist_updates \
                    log.debug("History says we've already grabbed %s.", filename)
                else:
                    dirname = feed.dirname
                    if dirname is None:
                        dirname = feed.dirname = re.sub('[\\\\/?*:<>|;"\'\.]','', str(encinfo.feed))
                        feed.half_flush()
                    channelDir = os.path.join(self.config.download_dir, dirname)
                    if not os.path.isdir(channelDir):
                        log.info("Creating new directory %s for channel", dirname)
                        os.mkdir(channelDir)
                    destFile = os.path.join(channelDir, filename)
                    if os.path.exists(destFile):
                        if encinfo.marked:
                            log.debug("Destination file exists: %s", destFile)
                            self.m_download_complete = True
                            update_playlist = False
                    else:
                        try:
                            if encinfo.marked:
                                log.info("Downloading %s", enc)
                                self.m_download_file = filename
                                tinfo = urllib.urlretrieve(enc, destFile, self.progress)
                            else:
                                log.debug("Skipping %s", enc)
                                first_time_skip = True
                        except KeyError:
                            raise
                        except socket.timeout, ex:
                            log.error("... operation timed out.")
                        except socket.gaierror, ex:
                            log.error("... error connecting (%s, errno=%s)",
                                                ex.args[1], str(ex.args[0]))
                        except IOError, ex:
                            errno, message = ex.args
                            if errno == 2: # ENOFILE
                                log.error("'No such file or directory' (probably "\
                                                    "caused by an enclosure target lacking the "\
                                                    "http:// part of the URL) downloading %s", enc)
                            else:
                                log.exception("IOError (errno=%s, message=%s) "\
                                                            "downloading %s", repr(errno),
                                                            repr(message), enc)
                        except:
                            log.exception("Problem downloading %s", enc)

                    pl_file = os.path.join(channelDir, filename)

                    if torrentfile==True:
                        try:
                            if encinfo.marked:
                                torrentmeta = torrents.TorrentFile(prefix+filename)
                                if torrentmeta==-1:
                                    print "invalid torrentfile"
                                    encinfo.unmark()
                                    update_playlist = False
                                else:
                                    if self.m_downloadlog.__contains__(torrentmeta.name)==True:
                                        pass
                                    else:
                                        self.m_downloadlog.append(torrentmeta.name)
                                        self.m_download_file = torrentmeta.name
                                        bt = BitTorrentAdaptor(self)
                                        bt.TorrentDownload(enc, prefix+torrentmeta.name, 0, torrentmeta)

                                    pl_file = os.path.join(channelDir, torrentmeta.name)
                        except:
                            print "torrent error"

                    if self.m_f_downloaded_size>=self.m_f_total_size:
                        self.m_download_complete=True

                    if torrentfile==True:
                        self.m_download_complete=True

                    if first_time_skip==True:
                        self.m_downloadlog.append(filename)

                    if encinfo.marked:
                        if update_playlist or self.config.force_playlist_updates:
                            if self.m_download_complete:
                                self.updateplaylist(pl_file, str(encinfo.feed))
                                self.m_downloadlog.append(filename)
                                # Call hooks to let 'em know we got it. 
                                downloadedhooks(encinfo, destFile)
                            else:
                                log.info("Scheduling deletion for partial file %s", pl_file)
                                # TODO: figure out why we don't actually do anything with this information
                                del_log = open(self.config.delete_list_file, "a")
                                del_log.write(pl_file+'\n')
                                del_log.close()

                    sys.stdout.flush()
            except:
                log.exception("An error occurred; continuing with next feed.")
            backannouncer(encinfo)

        # Final hook calls.
        counter(count, countmax)
        ender()

    def updateplaylist(self, filename, playlistname):
        player = self.config.player
        if player is not None:
                log.info("Updating playlist %s with %s", playlistname,
                                  os.path.basename(filename))
                player.append_and_create(filename, playlistname)

    def syncdevices(self):
        player = self.config.player
        if player is not None:
                player.sync()

    def start(self, progress):
        self.ui_progress = progress

        self.absorbhistory()
        if not self.config.use_new_download_code: 
            self.handledeletes()
            self.scanenclosures()
            self.downloadcontent()
            self.syncdevices()
        else: 
            log.warn("Using new feed scanning code...")
            self.scanfeeds()
        self.stop()

    def __init__(self, config, state):
        """Initialise the iPodder.

        config -- configuration.Configuration object
        state -- shelve-style state database or dict
        """

        self.m_downloadlog = []
        self.m_user_os = platform.system();
        self.config = config
        self.state = state
        self.feeds = feeds.Feeds(self.config, self.state)
        self.ui_progress = None
        self.hooks = hooks.HookCollection()

    def stop(self):
        "Flush the history file."
        try:
            exclude = open(self.config.history_file, "w")
            for filename in self.m_downloadlog:
                print >> exclude, filename
            exclude.close()
        except (IOError, OSError), ex:
            log.exception("Problem flushing history to %s",
                                        self.config.history_file)

def main():
    # Initialise the logging module and configure it for our console logging.
    # I'll factor this out soon so it's less convoluted.
    logging.basicConfig()
    handler = logging.StreamHandler()
    handler.formatter = conlogging.ConsoleFormatter("%(message)s", wrap=False)
    log.addHandler(handler)
    log.propagate = 0
    
    # Parse our configuration files.
    # I'll factor this out soon so it's less convoluted.
    parser = configuration.makeCommandLineParser()
    options, args = parser.parse_args()
    if args:
        parser.error("only need options; no arguments.")
    if options.debug:
        log.setLevel(logging.DEBUG)
    else:
        log.setLevel(logging.INFO)
    config = configuration.Configuration(options)
    if options.debug: # just in case config file over-rode it
        log.setLevel(logging.DEBUG)
    else:
        log.setLevel(logging.INFO)

    # Tweak our sub-log detail.
    hooks.log.setLevel(logging.INFO)
    
    # Open our state database.
    log.debug("Opening state database file...")
    state = shelve.open(config.state_db_file, 'c',
                        writeback=False, protocol=pickle.HIGHEST_PROTOCOL)
    log.debug("State database opened with %d entries.", len(state.keys()))
    
    # Initialise our iPodder.
    ipodder = iPodder(config, state)
    return ipodder
  
if __name__ == '__main__':
    ipodder = main()
    ipodder.start(ipodder.console_progress)
