#
# iPodder file grabbing code
#
# TODO: grab timeoutsocket; put into contrib; make sure feedparser
#       can see it and is happy. 
# 
# TODO: check content type to see if something is a torrent, just 
#       in case the extension isn't '.torrent'
# 
# TODO: consider moving grabfeed to a Feed method. Or not. :) 

import sys
import time
import stat
import logging
import threading
import socket
import random
import urllib2
import sha

from configuration import __version__
import threads
import hooks

import contrib.feedparser as feedparser
from contrib.BitTorrent.bencode import bencode, bdecode
from contrib.BitTorrent.btformats import check_message
import contrib.BitTorrent.download as download
import contrib.BitTorrent.track as track

log = logging.getLogger('iPodder')

SPAM = int(logging.DEBUG/2) # even more detailed than logging.DEBUG
AGENT = "iPodder/%s +http://ipodder.sf.net/" % __version__
BLOCKSIZE = 1024

class GrabError(Exception): 
    """Raised when we fail to grab something."""
    def __init__(self, message, ex=None): 
        """Initialise the GrabError."""
        Exception.__init__(self, message, ex)
        self.message = message
        self.exception = ex

class GenericGrabber(threads.SelfLogger):
    """Generic grabber. Over-ride download()."""

    def __init__(self, what, destfilename, blocksize=BLOCKSIZE, ipodder=None): 
        """Initialise the GenericGrabber."""
        self.what = what # probably a URL
        self.destfilename = destfilename
        self.blocksize = blocksize

        self.stopflag = threading.Event() # set to tell us to stop
        self.doneflag = threading.Event() # set by us when we're done
        #log.debug("%s", repr(destfilename))
        self.name = os.path.basename(destfilename)
        
        threads.SelfLogger.__init__(self, tag=self.name)

        # Public attributes, meant for interrogation from other threads
        self.last_activity = ''
        self.upload_rate = 0.0
        self.upload_mb = 0.0
        self.fraction_done = 0.0
        self.download_rate = 0.0
        self.download_mb = 0.0
        self.eta = 0.0 # units should be seconds

    def __call__(self): 
        """download()"""
        self.debug("downloading...")
        return self.download()
        
    def download(self): 
        """Download whatever we're supposed to.
        
        Return filename, headers (if you can)."""
        self.done()

    def done(self): 
        """Declare that we're done."""
        self.fraction_done = 1.0
        self.eta = 0.0
        self.hooks.get('update')()
        self.hooks.get('done')()
        self.doneflag.set()

    def stop(self): 
        self.stopflag.set()
        self.debug("asked to shut down; waiting...")
        self.doneflag.wait()
        self.debug("shut down.")

class NullFile(object): 
    """Pretend to be a writable file, but just throw away data."""
    def write(self, data): 
        pass

def build_opener(*handlers):
    """Create an opener object from a list of handlers.

    The opener will use several default handlers, including support
    for HTTP and FTP.

    If any of the handlers passed as arguments are subclasses of the
    default handlers, the default handlers will not be used.
    """

    log.debug("Using custom build_opener.")
    opener = urllib2.OpenerDirector()
    default_classes = [
        urllib2.ProxyHandler, urllib2.UnknownHandler, urllib2.HTTPHandler,
        urllib2.HTTPDefaultErrorHandler, urllib2.HTTPRedirectHandler,
        urllib2.FTPHandler, urllib2.FileHandler]
    if hasattr(urllib2.httplib, 'HTTPS'):
        default_classes.append(urllib2.HTTPSHandler)
    skip = []
    for klass in default_classes:
        for check in handlers:
            if urllib2.inspect.isclass(check):
                if issubclass(check, klass):
                    skip.append(klass)
            elif isinstance(check, klass):
                skip.append(klass)
    for klass in skip:
        default_classes.remove(klass)

    for h in handlers:
        if urllib2.inspect.isclass(h):
            h = h()
        if isinstance(h, urllib2.ProxyHandler): 
            opener.add_handler(h)

    for klass in default_classes:
        opener.add_handler(klass())

    for h in handlers:
        if urllib2.inspect.isclass(h):
            h = h()
        if not isinstance(h, urllib2.ProxyHandler): 
            opener.add_handler(h)

    return opener

class BasicGrabber(GenericGrabber): 
    """Basic grabber. Uses feedparser to help."""
    
    def __init__(self, what, destfilename, blocksize=BLOCKSIZE, 
                 etag='', modified='', referrer='', proxies={}, ipodder=None): 
        """Initialise the BasicGrabber.

        what -- the URL to grab
        destfilename -- the destination filename
        blocksize -- the block size for reads
        etag -- the etag to give the server
        modified -- the last-modification timestamp
        proxies -- map protocol to proxy URL
        """
        GenericGrabber.__init__(self, what, destfilename, blocksize)
        self.etag = etag
        self.modified = modified
        self.referrer = referrer
        self.proxies = proxies
        self.hooks = hooks.HookCollection()
        # Private attributes
        self._began = self._last = time.time()
        self._last_bytes = 0

    def download(self): 
        """Download the file."""
        nocopy = False
        what = self.what
        destfilename = self.destfilename
        try: 
            if os.path.exists(what): 
                fp = urllib.urlopen(what)
                self.headers = fp.info()
                if os.path.exists(destfilename): 
                    if os.path.getsize(destfilename) == os.path.getsize(what): 
                        nocopy = True
            else: 
                handlers = []
                proxies = self.proxies
                if 0: #proxies: 
                    self.warn("Using proxies: %s", 
                             ", ".join(
                                 ["%s=%s" % (key, val) for (key, val) in proxies.items()]
                                 ))
                    handlers.append(urllib2.ProxyHandler(proxies))
                # This should work: 
                #   urllib2.install_opener(build_opener)
                # ... except feedparser bypasses it, so we need to do this: 
                urllib2.build_opener = build_opener
                fp = feedparser._open_resource(
                        self.what,
                        self.etag,
                        self.modified,
                        AGENT, # module global
                        self.referrer,
                        handlers)
            headers = self.headers = fp.info()
            size = int(headers.get('content-length', 0))

            # TODO: Add some error handling
            if not nocopy: 
                bytes = 0
                destfp = file(destfilename, 'wb')
                self.progress(bytes, size)
                try:
                    block = fp.read(self.blocksize)
                    while block: 
                        if self.stopflag.isSet(): 
                            self.warn("Download aborted: %s", self.what)
                            break # OUCH!
                            # TODO: delete the dest file? what else?
                        size += len(block)
                        self.progress(size, size)
                        destfp.write(block)
                        block = fp.read(self.blocksize)
                finally: 
                    destfp.close()
            fp.close()
            self.done()

            return destfilename, headers
            
        except KeyboardInterrupt: # not likely in a thread, but what the hey.
            raise
        
        except socket.timeout, ex:
            self.error("... operation timed out.")
            raise GrabError, ("timeout", ex)
            
        except socket.gaierror, ex:
            self.error("... error connecting (%s, errno=%s)",
                                ex.args[1], str(ex.args[0]))
            raise GrabError, ("gaierror", ex)
            
        except IOError, ex:
            errno, message = ex.args
            if errno == 2: # ENOFILE
                self.error("'No such file or directory' (probably "\
                           "caused by an enclosure target lacking the "\
                           "http:// part of the URL) downloading %s", 
                           self.url)
            else:
                self.exception("IOError (errno=%s, message=%s) "\
                               "downloading %s", repr(errno),
                               repr(message), self.url)
            raise GrabError, ("IOError", ex)

        except Exception, ex:
            self.exception("Problem downloading %s", what)
            # TODO: shouldn't I just raise?
            raise GrabError, ("unknown", ex)

    def done(self): 
        """Declare we're done. Special one-file version."""
        GenericGrabber.done(self) # call our parent
        self.hooks.get('report-finished-file')(self.destfilename, self.info)
 
    def progress(self, bytes, size):
        """Update our status variables."""
        mb = bytes / 1048576.0
        delta_b = self._last_bytes - bytes
        self._last_bytes = bytes

        now = time.time()
        delta_t = now - self._last
        self._last = now

        # After calculating delta_t and delta_b, I'm going to ignore them because 
        # I have no idea how short the durations will be. Instead, I'll go for the 
        # more boring "progress so far over time so far" number. 

        duration = now - self._began
        if duration > 1: 
            rate = bytes / float(duration)
        else: 
            rate = 0.0

        if rate and size: 
            left = size - bytes
            self.eta = left / rate
        else: 
            self.eta = 0 # use as a sentinel: no idea :)

        self.upload_rate = 0.0
        self.upload_mb = 0.0
        self.download_rate = rate
        self.download_mb = mb
        
        if size > 0: 
            self.fraction_done = (100.0 * bytes) / float(size)
        else: 
            self.fraction_done = random.random()

        self.hooks.get('updated')() # tell others to check our state vars

# Terminology changes from 1.1:
# 
# * Instead of a TorrentMetaInfoFile or whatever it was, we're using a 
#   TorrentFile. I suspect it should just be a Torrent now that I think 
#   about it. TODO: stop thinking about it. 
# 
# * The previous version of the code in TorrentFile.__init__ used 
#   'metainfo' a lot. I'm calling it 'response', just like the BitTorrent 
#   internals. 
#
# TODO: take off the read-only bit on finished files so we can change 
# their ID3 information, so that the user can delete them, etc. 

class TorrentFile(object):
    """Class to keep track of BitTorrent file information."""

    def __init__(self, responseorfilename): 
        """Initialise the TorrentFile."""

        object.__init__(self)

        # BitTorrent.download.download() can download the torrent file all 
        # by itself. I think the only reason we're doing it ourselves is 
        # so we can figure out the length and the filename. 
        # 
        # TODO: consider whether we could double-check to make sure that 
        # nobody is faking nasty paths. 
        # 
        # TODO: definitely use this information to make sure there's 
        # enough disk space for the maneouvre. 
        # 
        # TODO: intelligently handle multiple-file torrents, with users 
        # being able to choose whether to unpack each torrent's files 
        # into a subdirectory for that torrent, or to unpack each 
        # torrent into the same directory (which might also make sense). 
        # That'd also be a good time to ponder how to handle items with 
        # multiple enclosure tags. 

        try: 
            response = responseorfilename
            tinfo = bdecode(response)
            self.responsefilename = None
        except ValueError: 
            infofd = open(responseorfilename, 'rb')
            response = infofd.read()
            infofd.close() # explicit close seen as better
            try:
                tinfo = bdecode(response) 
            except ValueError: 
                log.error("Couldn't decode BitTorrent response file %s",
                          responseorfilename)
                log.error("Response was: %s", repr(response))
                raise
            self.responsefilename = responseorfilename

        try: 
            check_message(tinfo)
            self.response = response
        except ValueError, ex: 
            log.error("Couldn't decode BitTorrent response file: %s", 
                      str(ex))
            raise # Force whomever is initialising us to deal with it. 

        self.announce = tinfo['announce'] # not used?
        info = tinfo['info']
        # self.info_hash = sha.new(bencode(info)).digest() # also not used?
        self.name = info['name']

        if info.has_key('length'):
            self.length = info['length']
        else:
            # TODO: wade through real examples and figure out what this 
            # code is trying to accomplish. Okay, so it's calculating 
            # a length, but what's the path variable for? 
            #
            # TODO: decide how to correctly handle torrents that download 
            # multiple files, such as those whopping huge ones from 
            # LegalTorrents. LegalTorrents also deliver zip files, which 
            # also make our life hard. 
            length = 0
            for file in info['files']:
                path = ''
                for item in file['path']:
                    if (path != ''):
                      path = path + "/"
                    path = path + item
                length += file['length']
            self.length = length

        #piece_length = info['piece length'] # also not used?
        #piece_number, last_piece_length = divmod(self.length, piece_length)

    def __repr__(self): 
        """Issue a string representation of the TorrentFile."""
        hexid = '%08x' % id(self)
        atts = ['length', 
                'info_hash',
                'announcer',
                'length',
                'name',
                'responsefilename']
        attrep = ', '.join(["%s=%s" % (att, getattr(self, att))
                            for att in atts])
        return "<%s.%s instance at 0x%s with attributes %s>" % (
                __module__, 
                self.__class__.__name__, 
                hexid.upper,
                attrep)

class TorrentGrabber(GenericGrabber): 
    """BitTorrent downloader thread.
    
    Not interested in multi-threading? Call .run() instead of .start()."""
    
    # TODO: make this match GenericGrabber :)

    def __init__(self, what, destfilename, blocksize=BLOCKSIZE, keepserving=False): 
        """Initialise the TorrentGrabber.
        
        what -- a TorrentFile object."""

        GenericGrabber.__init__(self, what, destfilename, blocksize)

        self.name = os.path.basename(what.name)
        #threading.Thread.__init__(self, name = self.name)
        self.hooks = hooks.HookCollection()
        self.filefunc_called = False
        self.keepserving = keepserving

        # Declare a status map for both __init__ and statusfunc
        self.statusMap = {
            'activity': ('last_activity', ''),
            'upRate': ('upload_rate', 0.0), 
            'upTotal': ('upload_total', 0.0), 
            'fractionDone': ('fraction_done', 0.0),
            'downRate': ('download_rate', 0.0), 
            'downTotal': ('download_total', 0.0), 
            'timeEst': ('time_remaining', 0), # not sure about this one
            }

        # Set some defaults
        for statuskey, settings in self.statusMap.items(): 
            att, default = settings
            setattr(self, att, default)

    def errorfunc(self, message):
        """Issue an error on behalf of BT.d.d."""
        self.error("%s thread reports: %s", self.name, message)
        raise GrabError, message

    def statusfunc(self, statusdict): 
        """Called by BT.d.d to indicate its status."""
        for key, value in statusdict.items(): 
            self.spam("%s thread reports: %s: %s", 
                      self.name, key, value)
            settings = self.statusMap.get(key)
            if settings is not None: 
                att = settings[0]
                setattr(self, att, value)
        self.hooks.get('updated')() # tell others to check our state vars

    def finfunc(self): 
        """Called by BT.d.d when it's done."""
        # TODO: this is an indication of download success! 
        # report back so we can add to playlists, etc!
        # (good use for hooks, that... though we might have 
        #  to notify completion on multiple files)
        self.debug("BitTorrent.download.download says it's done.")
        # TODO: wait a while
        if not self.keepserving: 
            self.stopflag.set()
        # TODO: self.hooks.get('report-finished-file')(filename,fakeheaders) for each
        self.done()
        
    def filefunc(self, name, length, saveas, isdir):
        """Decide where to put the downloaded files. 
        
        Called by BT.d.d with the following arguments:

        name -- response['info']['name']
        length -- file length or total file length
        saveas -- BT's configured saveas target
        isdir -- False for single file, True for multiple files

        BT's behaviour with isdir true is a little unusual: if you hand 
        it an existing directory with none of the torrent's named files 
        in it, it'll create a subdirectory in it to put the files into. 
        If just one of the files exist, it won't create a subdirectory. 
        Your only clue as to what happened is pathfunc getting called.

        The GOOD news is that resumption is pretty much automatic.
        """
        if isdir: 
            self.warn("iPodder's handling of torrents with multiple "\
                      "files is a little primitive. Please let us "\
                      "know of any problems.")
        return saveas

    def pathfunc(self, target): 
        """Called by BT to reveal the newly created directory it'll be 
        downloading files into. Only called for torrents with multiple 
        files. See also the documentation for filefunc."""
        self.info("Multiple files will be put in: %s", target)
        # TODO: something more useful than that

    def paramfunc(self, params): 
        """Called by BT.d.d to let us know how we can change some key 
        controls on the fly."""
        pass
        # TODO: make good use of
        # 'max_upload_rate': change_max_upload_rate(<int bytes/sec>)
        # 'max_uploads': change_max_uploads(<int max uploads>)
        # 'listen_port': int
        # 'peer_id': string
        # 'info_hash': string (why calculate it ourselves?)
        # 'start_connection': start_connection((<string ip>, <int port>), <peer id>)
        
    # This was being called with: 
    #
    #   url = enclosure url, 
    #   file = destination filename
    #   maxUploadRate = 0 (unlimited)
    #   torrentmeta = the torrent meta bit thingy
    #
    # Interestingly, no attempt is made to re-use the downloaded torrent 
    # file; BitTorrent.download.download() was having to do it *again*. 
    # How droll. It's weird because we can pass --responsefile as easily 
    # as we can pass --url... 
    
    def download(self): 
        """Perform the download. Called by run()."""
        # TODO: user-configurable  from keeping seedingtimeouts
        # TODO: user-configurable upload rate limits
        responsefilename = self.what.responsefilename
        if responsefilename is None: # only likely during testing, but...
            rawfd, responsefilename = tempfile.mkstemp('.torrent')
            self.debug("Saving raw response to temporary file %s", 
                        responsefilename)
            fd = os.fdopen(rawfd, 'wb')
            fd.write(self.what.response)
            fd.close()
            delete_responsefile = True
        else: 
            delete_responsefile = False
            
        try: 
            params = ['--responsefile', responsefilename,
                      #'--max_upload_rate', maxUploadRate, 
                      '--saveas', self.destfilename,
                      '--timeout', 60.0,
                      '--timeout_check_interval', 10]

            download.download(
                    params, 
                    self.filefunc, 
                    self.statusfunc, 
                    self.finfunc, 
                    self.errorfunc,
                    self.stopflag, # set to stop BitTorrent from keeping seeding
                    80, # used to format complaint to errorfunc if no params
                    pathFunc = self.pathfunc, 
                    paramfunc = self.paramfunc)

        finally: 
            if delete_responsefile: 
                try: 
                    self.debug("Deleting temporary response file %s", 
                              responsefilename)
                    os.unlink(responsefilename)
                except OSError, ex:
                    pass

if __name__ == '__main__': 
    import BaseHTTPServer
    import SimpleHTTPServer
    import mimetypes
    import StringIO
    import shutil
    import tempfile
    import os
    import unittest
    import random
    import Queue

    TESTPORT = 58585
    TRACKPORT = 47474

    logging.basicConfig()
    log.setLevel(logging.DEBUG)

    def yield_random_content(length): 
        """Build some random content of the required length."""
        blocksize = 2048
        blocks = []
        while length: 
            size = min(blocksize, length)
            bytes = []
            for idx in range(size): 
                bytes.append(chr(random.randint(0,255)))
            yield ''.join(bytes)
            length -= size

    def write_random_content(fd, length): 
        """Write random content to an open file."""
        for block in yield_random_content(length):
            fd.write(block)
        
    def random_content(length): 
        """Return random content.""" 
        sio = StringIO.StringIO()
        write_random_content(sio, length)
        return sio.getvalue()

    class RandomContentTester(unittest.TestCase): 
        """It always pays to test your test code, too."""
        def test_random_content(self): 
            """Test our ability to generate random content."""
            lengths = [0, 1, 2047, 2048, 2049, 8000, 12000, 100000]
            for length in lengths: 
                log.debug("Checking random_content(%d)", length)
                content = random_content(length)
                assert len(content) == length

    # This isn't quite a mock object, but it's pretty close. 

    class GrabberTestFakeFile(object): 
        """A fake file for Grabber testing."""

        def __init__(self, 
                     content = '', # content
                     headers = {}, # overrides to default headers
                     code = None, # overrides to code
                     message = None): # overrides to message
            self.content = content
            self.headers = headers
            self.code = code
            self.message = message
            
    class GrabberTestServer(threads.SelfLogger, BaseHTTPServer.HTTPServer): 
        """A test server for Grabbers."""
        
        def __init__(self, server_address, RequestHandlerClass): 
            """Initialise the test server."""
            BaseHTTPServer.HTTPServer.__init__(self, server_address, RequestHandlerClass)
            threads.SelfLogger.__init__(self)
            self.name, self.port = server_address
            self.__contents = {}
            self.stopflag = threading.Event()
            self.doneflag = threading.Event()
            self.readyflag = threading.Event()
            self.promptqueue = Queue.Queue()
            
        def __setitem__(self, key, value): 
            self.__contents[key] = value

        def __getitem__(self, key): 
            return self.__contents[key]

        def get(self, key, default=None): 
            return self.__contents.get(key, default)
            
        def add(self, key, **kw): 
            assert key[:1] == '/'
            self.debug("Added fake file %s", key)
            self[key] = GrabberTestFakeFile(**kw)

        def prompt(self, count=1): 
            """Prompt the GrabberTestServer to serve count requests."""
            self.debug("prompted to serve %d request(s)", count)
            for idx in range(count): 
                self.promptqueue.put(None)
            self.readyflag.wait()
            self.readyflag.clear()

        def serve_forever(self):
            """Handle one request at a time until doomsday."""
            while not self.stopflag.isSet():
                try: 
                    prompt = self.promptqueue.get(True, 1)
                    self.debug("waiting to serve a request.")
                    self.readyflag.set()
                    self.handle_request()
                except Queue.Empty, ex: 
                    pass # loop around again
            self.debug("stopped.")
            self.server_close()
            self.doneflag.set()

        def stop(self): 
            """Ask us to stop. Assumption: you're calling from another thread."""
            self.debug("asking the test server to stop...")
            self.stopflag.set()
            self.doneflag.wait()

    class GrabberTestRequestHandler(SimpleHTTPServer.SimpleHTTPRequestHandler): 
        server_version = "iPodderGrabberTester/1.0"
        protocol_version = "HTTP/1.0"

        def __init__(self, request, client_address, server): 
            SimpleHTTPServer.SimpleHTTPRequestHandler.__init__(self, request, client_address, server)
            self.contents = {}
            
        def send_head(self):
            """Common code for GET and HEAD commands.

            This sends the response code and MIME headers.

            Return value is either a file object (which has to be copied
            to the outputfile by the caller unless the command was HEAD,
            and must be closed by the caller under all circumstances), or
            None, in which case the caller has nothing further to do.

            """
            fake = self.server.get(self.path)
            if fake is None: 
                self.send_error(404)
                return None

            code = fake.code
            if code is None: 
                code = 200

            message = fake.message # None => default

            ctype = self.guess_type(self.path)
            headers = {
                "Content-type": ctype, 
                "Content-Length": len(fake.content)
                }
            headers.update(fake.headers)
            
            self.send_response(code, message)
            for key, value in headers.items(): 
                self.send_header(key, value)
            self.end_headers()

            return StringIO.StringIO(fake.content)

        extensions_map = mimetypes.types_map.copy()
        extensions_map.update({
            '.torrent': 'application/bittorrent',
            '.tor': 'application/bittorrent' # cheaty!
            })

    class BasicGrabberTests(unittest.TestCase): 
        def setUp(self): 
            addrinfo = ('', TESTPORT)
            self.httpd = GrabberTestServer(addrinfo, GrabberTestRequestHandler)
            self.tempdir = tempfile.mkdtemp('.tests', 'ipodder.grabber.')
            log.debug("Temporary directory is %s", self.tempdir)
            self.httpd_thread = httpd_thread = threads.OurThread(target=self.httpd.serve_forever)
            httpd_thread.setDaemon(True)
            log.debug("Starting HTTP daemon for test...")
            httpd_thread.start()

        def tearDown(self): 
            log.debug("Removing temporary directory %s...", self.tempdir)
            shutil.rmtree(self.tempdir, True, 
                    lambda fun, path, exc_info: \
                           log.error("%s can't delete %s: %s", fun, path, exc_info))
            self.httpd.stop()

        def prep(self, filename, *a, **kw): 
            self.httpd.add("/%s" % filename, *a, **kw)
            self.httpd.prompt()
            return 'http://127.0.0.1:%s/%s' % (TESTPORT, filename)

        def tempify(self, filename): 
            return os.path.join(self.tempdir, filename)
            
        def test_grab1(self): 
            url = self.prep('rss.xml', content='This is a test')
            bg = BasicGrabber(url, self.tempify('rss.xml'))
            bg()

        def test_grabmany(self):
            success = 0
            target = 10
            for i in range(target):
                try: 
                    self.test_grab1()
                    success += 1
                except GrabError: 
                    pass
            assert success == target, "%d < %d" % (success, target)

        def test_proxy1(self): 
            raise AssertionError, "Not safe at all."
            url = self.prep('rss.xml', content='This is a test')
            bg = BasicGrabber(url, self.tempify('rss.xml'), 
                    proxies = {'http': 'http://127.0.0.1:3128'})
            bg()
                    
    class TorrentGrabberTestTracker(threads.SelfLogger): 
        def __init__(self, dfile, servedir, **kwargs): 
            """Initialise the tracker."""
            threads.SelfLogger.__init__(self)
            self.debug("initialising...")
            rawconfig = { 'port': TRACKPORT,
                          'dfile': dfile, 
                          #'allowed_dir': servedir,
                          'parse_allowed_interval': 1 }
            rawconfig.update(kwargs)
            self.args = []
            for key, value in rawconfig.items(): 
                self.args.extend(['--%s' % key, value])
            self.config, files = track.parseargs(self.args, track.defaults, 0, 0)
            # files is ignored in BitTorrent.track.track(), too. 
            self.stopflag = threading.Event()
            self.doneflag = threading.Event()
               
        def __call__(self): 
            """Run the tracker."""
            config = self.config
            r = track.RawServer(self.stopflag, 
                                config['timeout_check_interval'], 
                                config['socket_timeout'])
            t = track.Tracker(config, r)
            r.bind(config['port'], config['bind'], True)
            self.debug("started.")
            r.listen_forever(track.HTTPHandler(t.get, 
                config['min_time_between_log_flushes']))
            t.save_dfile()
            self.debug("shut down.")
            self.doneflag.set()
            
        def stop(self): 
            """Ask the tracker to stop."""
            self.stopflag.set()
            self.debug("waiting for shutdown...")
            self.doneflag.wait()
       
    class TorrentGrabberTests(unittest.TestCase): 
        def __init__(self, *a, **kw): 
            unittest.TestCase.__init__(self, *a, **kw)
            # TODO: figure out how to re-use servers or kill ports

        def setUp(self): 
            addrinfo = ('', TESTPORT)
            self.tempdir = tempfile.mkdtemp('.tests', 'ipodder.grabber.')
            self.servedir = os.path.join(self.tempdir, 'serve')
            self.grabdir = os.path.join(self.tempdir, 'consume')
            self.dfile = os.path.join(self.tempdir, 'dfile.txt')
            os.mkdir(self.servedir)
            os.mkdir(self.grabdir)
            log.debug("Temporary directory is %s", self.tempdir)
            self.tracker = TorrentGrabberTestTracker(self.dfile, self.servedir)
            self.tracker_thread = tracker_thread = threads.OurThread(
                    target = self.tracker)
            tracker_thread.setDaemon(True)
            log.debug("Starting tracker for test...")
            tracker_thread.start()
            self.httpd = GrabberTestServer(addrinfo, GrabberTestRequestHandler)
            self.httpd_thread = httpd_thread = threads.OurThread(target=self.httpd.serve_forever)
            httpd_thread.setDaemon(True)
            log.debug("Starting HTTP daemon for test...")
            httpd_thread.start()

        def tearDown(self): 
            log.debug("Removing temporary directory %s...", self.tempdir)
            self.httpd.stop()
            self.tracker.stop()
            shutil.rmtree(self.tempdir, True, 
                    lambda fun, path, exc_info: \
                           log.error("%s can't delete %s: %s", fun, path, exc_info))

        def makeResponse(self, filename, piecelength=None): 
            if piecelength is None: 
                piecelength = 2**18
            fd = open(filename, 'rb')
            pieces = []
            while 1: 
                block = fd.read(piecelength) 
                if len(block) or not len(pieces): 
                    sha_ = sha.new()
                    sha_.update(block)
                    pieces.append(sha_.digest())
                if not block:
                    break
            fd.close()

            response = {
                'announce': 'http://127.0.0.1:%d/announce' % TRACKPORT,
                'creation date': int(time.time()),
                'info': {
                    'pieces': ''.join(pieces),
                    'piece length': piecelength,
                    'name': os.path.basename(filename), 
                    'length': os.path.getsize(filename)
                    }
                }

            return bencode(response)

        def makeFileAndTorrent(self, filename=None, length=None): 
            """Make a file and torrent. 

            Returns filepath, responsefilecontents, torrenturl."""
            
            if filename is None: 
                filename = 'blah%d.mp3' % random.randint(1000,2000)
            base, ext = os.path.splitext(filename)
            if length is None: 
                length = random.randint(1024*64, 1024*256)
            filepath = os.path.join(self.servedir, filename)
            fd = open(filepath, 'wb')
            write_random_content(fd, length)
            fd.close()
            response = self.makeResponse(filepath)
            torrentname = '%s.torrent' % base
            self.httpd.add("/%s" % torrentname, content=response) # next arg is header dict
            self.httpd.prompt()
            torrenturl = 'http://127.0.0.1:%s/%s' % (TESTPORT, torrentname) 
            return filepath, response, torrenturl
                
        def test_checkalreadygot(self): 
            """Make sure we can check what we already have."""
            filepath, response, torrenturl = self.makeFileAndTorrent()
            torrentpath = os.path.join(self.grabdir, 'test.torrent')
            # First, download the torrent file and torrentgrab it
            grabber = BasicGrabber(torrenturl, torrentpath)
            grabber()
            grabber = TorrentGrabber(TorrentFile(torrentpath), filepath)
            grabber()
            # Second, just use the raw response
            grabber = TorrentGrabber(TorrentFile(response), filepath)
            grabber()

        def test_download(self): 
            filepath, response, torrenturl = self.makeFileAndTorrent()
            torrentpath = os.path.join(self.grabdir, 'test.torrent')
            mp3path = os.path.join(self.grabdir, os.path.basename(filepath))
            # first, serve it up in another thread
            log.debug("Serving up %s", filepath)
            seeder = TorrentGrabber(TorrentFile(response), filepath, keepserving=True)
            seedthread = threads.OurThread(target=seeder)
            seedthread.setDaemon(True)
            seedthread.start()
            try: 
                # then, grab it
                log.debug("Grabbing %s", torrenturl)
                grabber = BasicGrabber(torrenturl, torrentpath)
                grabber()
                log.debug("Grabbing target %s", mp3path)
                grabber = TorrentGrabber(TorrentFile(torrentpath), mp3path)
                grabber()
                seedthread.catch()
                assert grabber.doneflag.isSet()
            finally: 
                # and tidy up
                seeder.stop()
                log.debug("Waiting for seeder thread...")
                seedthread.join()
                seedthread.catch()
            
    unittest.main()
