# Outline code for iPodder

"""
Let's embrace outlines fully in iPodder: not just for the feed management,
but also for playlist management, listing items, and so on. 
"""

import xml.sax
import xml.dom.minidom
import StringIO

import logging
log = logging.getLogger('iPodder.Outline')
SPAM = logging.DEBUG >> 1
spam = lambda *a, **kw: log.log(SPAM, *a, **kw)

class Node(list): 
    """An outliner node."""
    
    def __init__(self, *args, **kwargs): 
        """Initialise the outliner node. 

        kwargs -- used to initialise the attributes
        args -- each argument is assumed to be a Node, and is added as 
                a member. If the first element is a string, the node 
                type is set to 'text' and the 'text' attribute is set 
                to the first element. 
        """
        # Attributes
        self._atts = {'type': ''}
        assert not kwargs.has_key('type') # force kids to set .type
        self._atts.update(kwargs)
        # Members
        list.__init__(self)
        if len(args): 
            if isinstance(args[0], str): 
                self.text = args[0]
                self.type = 'text'
                args = args[1:]
        self.extend(args)
        
    def attadd(self, key, more): 
        """Add more to self[key], defaulting self[key] to ''."""
        self[key] = self.get(key, '') + more
        
    # Dict-style access
    def get(self, key, default=None): return self._atts.get(key, default)
    def has_key(self, key): return self._atts.has_key(key)
    def items(self): return self._atts.items()
    def keys(self): return self._atts.keys()
    def values(self): return self._atts.values()
    def __getitem__(self, key): 
        if isinstance(key, (int, slice)): 
            return list.__getitem__(self, key)
        else: 
            return self._atts[key]
    def __setitem__(self, key, value): 
        if isinstance(key, (int, slice)): 
            list.__setitem__(self, key, value)
        else: 
            self._atts[key] = value

    # Att-style access
    def __getattr__(self, att): 
        if att[:1] == '_': 
            return object.__getattribute__(self, att)
        try: 
            return object.__getattribute__(self, '_atts')[att]
        except KeyError: 
            raise AttributeError, att
        
    def __setattr__(self, att, value): 
        if att[:1] == '_': 
            object.__setattr__(self, att, value)
        else: 
            object.__getattribute__(self, '_atts')[att] = value

    def __repr__(self): 
        atts = self._atts
        attnames = atts.keys() # take a copy
        attnames.sort()
        bits = ['%s=%s' % (attname, repr(atts[attname]))
                for attname in attnames]
        bits.extend([repr(member) for member in self])
        return '[%s]' % ', '.join(bits)

    def __eq__(self, other): 
        """Compare one tree of Nodes with another tree of Nodes.
        Ignores class, just in case we read from OPML."""
        spam("comparing...")
        if len(self) != len(other): 
            spam("length mismatch: %d != %d", len(self), len(other))
            return False 
        myattnames = self.keys()
        myattnames.sort()
        attnames = other.keys()
        attnames.sort()
        if attnames != myattnames: 
            return False
        for attname in myattnames: 
            if self[attname] != other[attname]: 
                return False
        for index in range(len(self)): 
            if self[index] != other[index]: 
                return False
        return True

class Link(Node): 
    def __init__(self, text, href='', *args, **kwargs): 
        Node.__init__(self, text, href=href, *args, **kwargs)
        self.type = 'link'

class Text(Node): 
    def __init__(self, text, *args, **kwargs): 
        Node.__init__(self, text, *args, **kwargs)
        self.type = 'text'

class Head(Node): 
    def __init__(self, title, *args, **kwargs): 
        Node.__init__(self, title=title, *args, **kwargs)
        self.type = 'head'
    
    def toopml(self): 
        """Convert everything under the Head to XML."""
        impl = xml.dom.minidom.getDOMImplementation()
        doc = impl.createDocument(None, 'opml', None)
        top = doc.documentElement
        head = doc.createElement('head')
        top.appendChild(head)
        headbits = {}
        headbits.update(self)
        if not headbits.has_key('dateCreated'):  
            pass # we should do something
        if not headbits.has_key('dateModified'):  
            pass # we should do something
        if headbits.has_key('type'):
            del headbits['type']
        for bit, value in headbits.items(): 
            elem = doc.createElement(bit)
            data = doc.createTextNode(str(value).strip())
            elem.appendChild(data)
            head.appendChild(elem)
        body = doc.createElement('body')
        top.appendChild(body)
        
        def suck(node): 
            elem = doc.createElement('outline')
            atts = {}
            atts.update(node)
            #if node is self: 
            #    if atts.has_key('title'): 
            #        if not atts.has_key('text'): 
            #            atts['text'] = atts['title']
            #        del atts['title']
            #    atts['type'] = 'text'
            if atts['type'] == 'text': 
                del atts['type']
            attnames = atts.keys()
            attnames.sort()
            for attname in attnames: 
                value = atts[attname]
                elem.setAttribute(attname, str(value).strip())
            for child in node: 
                elem.appendChild(suck(child))
            return elem

        for node in self:
            body.appendChild(suck(node))
        #body.appendChild(suck(self))

        # return doc.toxml()
        outs = StringIO.StringIO()
        doc.writexml(outs, '', '    ', '\n')
        return outs.getvalue()

    def fromopml(self, resource): 
        """Return a Head corresponding to an OPML file.
        
        resource -- XML string, or a stream."""
        
        opmlhandler = OPMLEventHandler()
        if isinstance(resource, str): 
            sio = StringIO.StringIO(resource)
            xml.sax.parse(sio, opmlhandler)
            sio.close()
        else: 
            xml.sax.parse(resource, opmlhandler)
        return opmlhandler.head

    fromopml = classmethod(fromopml)
        
class OPMLEventHandler(xml.sax.handler.ContentHandler): 
    def __init__(self): 
        xml.sax.handler.ContentHandler.__init__(self)
        self.stack = []
        self.context = []
        self.onchars = None
    def startElement(self, name, atts): 
        stack = self.stack
        context = self.context
        self.onchars = None # reset it
        slen = len(stack)
        if not slen: 
            assert name == 'opml'
            # no effect on context
        elif slen == 1: 
            assert name in ['head', 'body']
            if name == 'head': 
                self.head = Head('')
                context.append(self.head)
        elif slen == 2: 
            if stack[1] == 'head': 
                assert name != 'outline'
                head = context[0]
                self.onchars = lambda chars: head.attadd(name, chars)
            else: 
                assert name == 'outline'
                self.startOutline(atts)
        else: 
            assert stack[-1] == 'outline'
            assert name == 'outline'
            self.startOutline(atts)
        self.stack.append(name)
        spam("starting %s", name)
        for att, val in atts.items():
            spam("  %s = %s", att, repr(val))
    def startOutline(self, atts): 
        parent = self.context[-1]
        child = Node()
        for att, val in atts.items(): 
            child[att] = val
        parent.append(child)
        self.context.append(child)
    def endElement(self, name): 
        spam("ending %s", name)
        assert self.stack[-1] == name
        if name == 'outline': 
            self.context.pop()
        self.stack.pop()
    def characters(self, chars): 
        if self.onchars is not None: 
            spam("handling characters: %s", repr(chars))
            self.onchars(chars.strip())
        else: 
            spam("ignoring characters: %s", repr(chars))
        
if __name__ == '__main__': 
    import unittest
    
    class NodeTest(unittest.TestCase): 

        def test_create_noargs(self): 
            node = Node()
            assert node.type == ''
            
        def test_create_autotext(self): 
            node = Node('boingoingoing')
            assert node.type == 'text'
            assert node.text == 'boingoingoing'

        def test_head(self): 
            tree = Head('title')

        def test_nest(self): 
            tree = Node('title', Node('boing'))

        def create_tree(self): 
            tree = Head("Playlists", 
                #Node("Playlists", 
                    Link("iPodder.org: Podcasting Central", 
                         href="http://www.ipodder.org/discuss/reader$4.opml"),
                    Link("iPodder Team Directory", 
                         href="http://ipodder.sf.net/opml/ipodder.opml"),
                    Link("iPodderX Top Picks", 
                         href="http://directory.ipodderx.com/opml/iPodderX_Picks.opml"), 
                    Link("iPodderX Most Popular", 
                         href="http://directory.ipodderx.com/opml/iPodderX_Popular.opml"),
                #    )
                )
            return tree

        def test_create_tree(self): 
            self.create_tree()
            
        def test_tree_repr(self): 
            tree = self.create_tree()
            repr(tree)

        def test_tree_toopml(self): 
            tree = self.create_tree()
            tree.toopml()

        def test_tree_eq_silly(self): 
            tree = self.create_tree()
            tree2 = self.create_tree()
            assert tree == tree2
            
        def test_tree_fromopml(self): 
            tree = self.create_tree()
            xml = tree.toopml()
            spam(xml)
            newtree = Head.fromopml(xml)
            spam(tree)
            spam(newtree)
            assert newtree == tree

    logging.basicConfig()
    log.setLevel(logging.DEBUG)
    unittest.main()
