/***************************************************************************
 *   Copyright (C) 2004 by Michael Schulze                                 *
 *   mike.s@genion.de                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "trackmetadata.h"
#include "tracklist.h"

#include <qfileinfo.h>
#include <qstringlist.h>

#include <kdebug.h>

#include <id3/tag.h>
#include <id3/misc_support.h>

TrackMetadata::TrackMetadata()
    : itunesdb::Track()
{
    samplerate = 0xAC440000;    // hardwired
    setFDesc( QString( "MPEG audio file"));    // hardwired for now
}


TrackMetadata::TrackMetadata(Q_UINT32 trackid)
    : itunesdb::Track()
{
    id = trackid;
    samplerate = 0xAC440000;    // hardwired
    setFDesc( QString( "MPEG audio file"));    // hardwired for now
}


TrackMetadata::TrackMetadata(const Track& track)
    : itunesdb::Track(track)
{
    samplerate = 0xAC440000;    // hardwired
    setFDesc( QString( "MPEG audio file"));    // hardwired for now
}

TrackMetadata::~TrackMetadata()
{
}

/*!
    \fn TrackMetadata::getFileExtension()
 */
const QString& TrackMetadata::getFileExtension() const
{
    return file_extension;
}

/*!
    \fn TrackMetadata::setFileExtension( QString& extension)
 */
void TrackMetadata::setFileExtension(const QString& extension)
{
    file_extension= extension;
}


/*!
    \fn TrackMetadata::readFromFile(const QString& filename)
 */
bool TrackMetadata::readFromFile(const QString& filename)
{
    QFileInfo fileinfo( filename);
    if( !fileinfo.exists())
        return false;    // whatever happened here
    
    setFileExtension( fileinfo.extension(FALSE));
    file_size = fileinfo.size();
    lastmodified = fileinfo.lastModified().toTime_t();
    
    // TODO there are other formats than mp3 - decide on file extension what to do
    
    // put the following code in an extra helper class for mp3 metadata extraction
    ID3_Tag id3tag(filename);
    if (id3tag.NumFrames() <= 0)    // no mp3 information found: exit for now (maybe it's not even an mp3 file)
        return false;
    
    char * yearstring = ID3_GetYear(&id3tag);
    if (yearstring != NULL) {
        year = atoi( yearstring);
    } else {
        year = 0;
    }
    
    // get Composer
    ID3_Frame * frame = id3tag.Find(ID3FID_COMPOSER);
    if (frame != NULL) {
        setComposer( QString(ID3_GetString(frame, ID3FN_TEXT)));
    }
    
    // get volumeadjust
    frame = id3tag.Find(ID3FID_VOLUMEADJ);
    if (frame != NULL) {
        volumeadjust = frame->GetField(ID3FN_VOLUMEADJ)->Get();
    }
    
    frame = id3tag.Find(ID3FID_PLAYCOUNTER);
    if (frame != NULL) {
        playcount = frame->GetField(ID3FN_COUNTER)->Get();
    }

    setAlbum( QString(ID3_GetAlbum(&id3tag)));
    setArtist( QString(ID3_GetArtist(&id3tag)));
    setTitle( QString(ID3_GetTitle(&id3tag)));
    setComment( QString(ID3_GetComment(&id3tag)));
    
    int genrenum = ID3_GetGenreNum(&id3tag);
    if (genrenum != 0xFF) {
        setGenre(QString(ID3_V1GENRE2DESCRIPTION(genrenum)));
    } else {
        setGenre(QString(ID3_GetGenre(&id3tag)));
    }
    
    const Mp3_Headerinfo * mp3header = id3tag.GetMp3HeaderInfo();
    if (mp3header != NULL) {
        tracklen = mp3header->time * 1000;    // time in milliseconds
        tracknum = ID3_GetTrackNum(&id3tag);
        if (mp3header->vbr_bitrate) {
            bitrate = mp3header->vbr_bitrate/1000;
            vbr = 1;
        } else {
            bitrate = mp3header->bitrate/1000;    // Bitrate in KBps
            vbr = 0;
        }
    }
    type = 1;
    compilation = 0;
    rating = 0;
    
    // some sanity checks
    if (getArtist().isEmpty() || getTitle().isEmpty())
        return false;    // for now
    
    doneAddingData();
    
    return true;    // for now
}


/*!
    \fn TrackMetadata::toLogEntry()
 */
QStringList& TrackMetadata::toLogEntry(QStringList& valuebuffer) const
{
    valuebuffer << QString().setNum( getID(), 36 );
    valuebuffer << getPath();
    valuebuffer << getArtist();
    valuebuffer << getAlbum();
    valuebuffer << getTitle();
    valuebuffer << getComment();
    valuebuffer << getGenre();
    valuebuffer << getComposer();
    valuebuffer << QString().setNum( getYear(), 36 );
    valuebuffer << QString().setNum( getFileSize(), 36 );
    valuebuffer << QString().setNum( getLastModified(), 36 );
    valuebuffer << QString().setNum( getBitrate(), 36 );
    valuebuffer << QString().setNum( getSamplerate(), 36 );
    valuebuffer << QString().setNum( getTrackLength(), 36 );
    valuebuffer << QString().setNum( getTrackNumber(), 36 );
    valuebuffer << QString().setNum( getCdNumber(), 36 );
    valuebuffer << QString().setNum( getCdCount(), 36 );
    valuebuffer << QString().setNum( getVolumeAdjust(), 36 );
    valuebuffer << getFileExtension();
    valuebuffer << QString().setNum( vbr, 36 );
    valuebuffer << QString().setNum( type, 36 );
    valuebuffer << QString().setNum( compilation, 36 );
    valuebuffer << QString().setNum( rating, 36 );
    valuebuffer << QString().setNum( playcount, 36 );
    valuebuffer << QString().setNum( getDBID(), 36 );

    return valuebuffer;
}


/*!
    \fn TrackMetadata::readFromLogEntry()
 */
bool TrackMetadata::readFromLogEntry(const QStringList& values)
{
    bool result = TRUE;

    // some sanity checks first
    if (values.size() < 25)
        return false;    // wtf?
    
    for( uint i= 0; i < values.size(); i++) {
        QString data( values[ i ] );
        switch( i) {
            case 0:
                id = data.toUInt( &result, 36 );
                if ( !result || id == 0 )
                    return false;
                break;
            case 1: setPath( data ); break;
            case 2: setArtist( data ); break;
            case 3: setAlbum( data ); break;
            case 4: setTitle( data ); break;
            case 5: setComment( data ); break;
            case 6: setGenre( data ); break;
            case 7: setComposer( data ); break;
            case 8: year = data.toUInt( &result, 36 ); break;
            case 9: file_size = data.toUInt( &result, 36 ); break;
            case 10: lastmodified = data.toUInt( &result, 36 ); break;
            case 11: bitrate = data.toUInt( &result, 36 ); break;
            case 12: samplerate = data.toUInt( &result, 36 ); break;
            case 13: tracklen = data.toUInt( &result, 36 ); break;
            case 14: tracknum = data.toUInt( &result, 36 ); break;
            case 15: cdnum = data.toUInt( &result, 36 ); break;
            case 16: numcds = data.toUInt( &result, 36 ); break;
            case 17: volumeadjust = data.toUInt( &result, 36 ); break;
            case 18: setFileExtension( data ); break;
            case 19: vbr = data.toUShort( &result, 36 ) & 255; break;
            case 20: type = data.toUShort( &result, 36 ) & 255; break;
            case 21: compilation = data.toUShort( &result, 36 ) & 255; break;
            case 22: rating = data.toUShort( &result, 36 ) & 255; break;
            case 23: playcount = data.toUInt( &result, 36 ); break;
            case 24: dbid = data.toULong( &result, 36 ); break;
        }
    }
    
    doneAddingData();
    
    return result;
}

