/*
 * iroffer by PMG
 * Copyright (C) 1998-2003 PMG
 * 
 * By using this file, you agree to the terms and conditions set
 * forth in the GNU General Public License.  More information is
 * available in the README file.
 * 
 * If you received this file without documentation, it can be
 * downloaded from http://iroffer.org/
 * 
 * @(#) iroffer_admin.c 1.106@(#)
 * pmg@wellington.i202.centerclick.org|src/iroffer_admin.c|20030914020532|06029
 * 
 */

/* include the headers */
#include "iroffer_config.h"
#include "iroffer_defines.h"
#include "iroffer_headers.h"
#include "iroffer_globals.h"

/* local functions */
static void u_respond(const userinput * const u, const char *format, ...);

static void u_help(const userinput * const u);
static void u_xdl(const userinput * const u);
static void u_xds(const userinput * const u);
static void u_dcl(const userinput * const u);
static void u_dcld(const userinput * const u);
static void u_qul(const userinput * const u);
static void u_close(const userinput * const u);
static void u_nomin(const userinput * const u);
static void u_nomax(const userinput * const u);
static void u_rmq_rpq(const userinput * const u);
static void u_raw(const userinput * const u);
static void u_redraw(const userinput * const u);
static void u_info(const userinput * const u);
static void u_remove(const userinput * const u);
static void u_send(const userinput * const u);
static void u_psend(const userinput * const u);
static void u_msg(const userinput * const u);
static void u_mesg(const userinput * const u);
static void u_mesq(const userinput * const u);
static void u_closedcc(const userinput * const u);
static void u_status(const userinput * const u);
static void u_chfile(const userinput * const u);
static void u_chdesc(const userinput * const u);
static void u_chnote(const userinput * const u);
static void u_chmins(const userinput * const u);
static void u_chmaxs(const userinput * const u);
static void u_chgets(const userinput * const u);
static void u_add(const userinput * const u);
static void u_adddir(const userinput * const u);
static void u_chatme(const userinput * const u);
static void u_rehash(const userinput * const u);
static void u_botinfo(const userinput * const u);
static void u_ignl(const userinput * const u);
static void u_ignore(const userinput * const u);
static void u_unignore(const userinput * const u);
static void u_nosave(const userinput * const u);
static void u_nosend(const userinput * const u);
static void u_nolist(const userinput * const u);
static void u_renumber(const userinput * const u);
static void u_msgread(const userinput * const u);
static void u_msgdel(const userinput * const u);
static void u_memstat(const userinput * const u);
static void u_qsend(const userinput * const u);
static void u_shutdown(const userinput * const u);
static void u_debug(const userinput * const u);
static void u_jump(const userinput * const u);
static void u_servqc(const userinput * const u);
static void u_servers(const userinput * const u);
static void u_trinfo(const userinput * const u);
static void u_listul(const userinput * const u);
static void u_clearrecords(const userinput * const u);
static void u_rmul(const userinput * const u);
static void u_crash(const userinput * const u);
static void u_chanl(const userinput * const u);


/* local info */
static const userinput_parse_t userinput_parse[] = {
{1,method_allow_all,u_help,     "HELP",NULL,"Shows Help"},
{1,method_allow_all_xdl,u_xdl,  "XDL",NULL,"Lists Offered Files"},
{1,method_allow_all,u_xds,      "XDS",NULL,"Save XDCC File"},
{1,method_allow_all,u_dcl,      "DCL",NULL,"Lists Current Transfers"},
{1,method_allow_all,u_dcld,     "DCLD",NULL,"Lists Current Transfers with Details"},
{1,method_allow_all,u_trinfo,   "TRINFO","n","Lists Information About Transfer n"},
{1,method_allow_all,u_qul,      "QUL",NULL,"Lists Current Queue"},
{1,method_allow_all,u_ignl,     "IGNL",NULL,"Show Ignored List"},
{1,method_allow_all,u_listul,   "LISTUL",NULL,"Shows contents of upload directory"},
{1,method_allow_all,u_chanl,    "CHANL",NULL,"Shows channel list with member list"},

{2,method_allow_all,u_close,    "CLOSE","n","Cancels Transfer with ID = n"},
{2,method_allow_all,u_rmq_rpq,  "RMQ","n","Removes Main Queue Number n"},
{2,method_allow_all,u_rmq_rpq,  "RPQ","n","Removes Pack Queue Number n"},
{2,method_allow_all,u_nomin,    "NOMIN","n","Disables Minspeed For Transfer ID n"},
{2,method_allow_all,u_nomax,    "NOMAX","n","Disables Maxspeed For Transfer ID n"},
{2,method_allow_all,u_send,     "SEND","nick n","Sends Pack n to nick"},
{2,method_allow_all,u_psend,    "PSEND","<channel> <style>","Sends <style> (full|minimal|summary) XDCC LIST to <channel>"},
{2,method_allow_all,u_qsend,    "QSEND",NULL,"Sends Out The First Queued Pack"},

{3,method_allow_all,u_info,     "INFO","n","Show Info for Pack n"},
{3,method_allow_all,u_remove,   "REMOVE","n","Removes Pack n"},
{3,method_allow_all,u_renumber, "RENUMBER","x y","Moves Pack x to y"},
{3,method_allow_all,u_add,      "ADD","<filename>","Add New Pack With <filename>"},
{3,method_allow_all,u_adddir,   "ADDDIR","<dir>","Add Every File in <dir>"},
{3,method_allow_all,u_chfile,   "CHFILE","n <msg>","Change File of pack n to <msg>"},
{3,method_allow_all,u_chdesc,   "CHDESC","n <msg>","Change Description of pack n to <msg>"},
{3,method_allow_all,u_chnote,   "CHNOTE","n <msg>","Change Note of pack n to <msg>"},
{3,method_allow_all,u_chmins,   "CHMINS","n x","Change min speed of pack n to x KB"},
{3,method_allow_all,u_chmaxs,   "CHMAXS","n x","Change max speed of pack n to x KB"},
{3,method_allow_all,u_chgets,   "CHGETS","n x","Change the get count of a pack"},

{4,method_allow_all,u_msg,      "MSG","<nick> <message>","Send a message to a user"},
{4,method_allow_all,u_mesg,     "MESG","<message>","Sends msg to all users who are transferring"},
{4,method_allow_all,u_mesq,     "MESQ","<message>","Sends msg to all users in a queue"},
{4,method_allow_all,u_ignore,   "IGNORE","n <hostmask>","Ignore hostmask (nick!user@host) for n minutes, wildcards allowed"},
{4,method_allow_all,u_unignore, "UNIGNORE","<hostmask>","Un-Ignore hostmask"},
{4,method_allow_all,u_nosave,   "NOSAVE","n","Disables XDCC AutoSave for next n minutes"},
{4,method_allow_all,u_nosend,   "NOSEND","n","Disables XDCC Send for next n minutes"},
{4,method_allow_all,u_nolist,   "NOLIST","n","Disables XDCC List and Plist for next n mins"},
{4,method_allow_all,u_msgread,  "MSGREAD",NULL,"Show MSG log"},
{4,method_allow_all,u_msgdel,   "MSGDEL",NULL,"Delete MSG log"},
{4,method_allow_all,u_rmul,     "RMUL","<file>","Delete a file in the Upload Dir"},
{4,method_allow_all,u_raw,      "RAW","<command>","Send <command> to server (RAW IRC)"},

{5,method_allow_all,u_servers,  "SERVERS",NULL,"Shows the server list"},
{5,method_allow_all,u_jump,     "JUMP","<num>","Switches to a random server or server <num>"},
{5,method_allow_all,u_servqc,   "SERVQC",NULL,"Clears the server send queue"},
{5,method_allow_all,u_status,   "STATUS",NULL,"Show Useful Information"},
{5,method_allow_all,u_rehash,   "REHASH",NULL,"Re-reads config file(s) and reconfigures"},
{5,method_allow_all,u_botinfo,  "BOTINFO",NULL,"Show Information about the bot status"},
{5,method_allow_all,u_memstat,  "MEMSTAT",NULL,"Show Information about memory usage"},
{5,method_allow_all,u_clearrecords, "CLEARRECORDS",NULL,"Clears transfer, bandwidth, uptime, and total sent"},
{5,method_stdin,    u_redraw,   "REDRAW",NULL,"Redraws the Screen"},
{5,method_dcc,      u_closedcc, "QUIT",NULL,"Close DCC Chat"},
{5,method_msg,      u_chatme,   "CHATME",NULL,"Sends you a DCC Chat Request"},
{5,method_stdin,    u_debug,    "DEBUG","n","Set Debugging level to n"},
{5,method_allow_all,u_shutdown, "SHUTDOWN","<act>","Shutdown iroffer, <act> is \"now\", \"delayed\", or \"cancel\""},

{6,method_stdin,    u_crash, "CRASH",NULL,"Cause a segmentation fault"},
};



void u_fillwith_stdin (userinput * const u, char *line) {
   char* tempstr1 = mycalloc(maxtextlength);
   char* tempstr2 = mycalloc(maxtextlength);
   int i;
   
   updatecontext();
   
   if (line[strlen(line)-1] == '\n') line[strlen(line)-1] = '\0';
   
   u->method = method_stdin;
   u->snick = NULL;

   u->cmd = caps(getpart(line,1));
   u->arg1 = getpart(line,2);
   u->arg2 = getpart(line,3);
   u->arg3 = getpart(line,4);
   
   strncpy(tempstr1,"",maxtextlength-1);
   if (u->arg1) {
      strncpy(tempstr1,line,maxtextlength-1);
      for (i=strlen(u->cmd)+1; i<=sstrlen(line); i++)
         tempstr1[i-strlen(u->cmd)-1] = tempstr1[i];
      }
   u->arg1e = tempstr1;
   
   strncpy(tempstr2,"",maxtextlength-1);
   if (u->arg1 && u->arg2) {
      strncpy(tempstr2,line,maxtextlength-1);
      for (i=sstrlen(u->cmd)+sstrlen(u->arg1)+2; i<=sstrlen(line); i++)
         tempstr2[i-strlen(u->cmd)-strlen(u->arg1)-2] = tempstr2[i];
      }
   u->arg2e = tempstr2;
   
   }

void u_fillwith_dcc (userinput * const u, char *line) {
   char* tempstr1 = mycalloc(maxtextlength);
   char* tempstr2 = mycalloc(maxtextlength);
   int i;
   
   updatecontext();
   
   if (line[strlen(line)-1] == '\n') line[strlen(line)-1] = '\0';
   
   u->method = method_dcc;
   u->snick = NULL;

   u->cmd = caps(getpart(line,1));
   u->arg1 = getpart(line,2);
   u->arg2 = getpart(line,3);
   u->arg3 = getpart(line,4);
   
   strncpy(tempstr1,"",maxtextlength-1);
   if (u->arg1) {
      strncpy(tempstr1,line,maxtextlength-1);
      for (i=sstrlen(u->cmd)+1; i<=sstrlen(line); i++)
         tempstr1[i-strlen(u->cmd)-1] = tempstr1[i];
      }
   u->arg1e = tempstr1;
   
   strncpy(tempstr2,"",maxtextlength-1);
   if (u->arg1 && u->arg2) {
      strncpy(tempstr2,line,maxtextlength-1);
      for (i=sstrlen(u->cmd)+sstrlen(u->arg1)+2; i<=sstrlen(line); i++)
         tempstr2[i-strlen(u->cmd)-strlen(u->arg1)-2] = tempstr2[i];
      }
   u->arg2e = tempstr2;
   
   }

void u_fillwith_msg (userinput * const u, char* n, const char *line) {
   char* tempstr1 = mycalloc(maxtextlength);
   char* tempstr2 = mycalloc(maxtextlength);
   char *t1,*t2,*t3,*t4,*t5;
   int i,len;
   
   updatecontext();
   
   u->method = method_msg;
   u->snick = n;
   t1 = getpart(line,1);
   t2 = getpart(line,2);
   t3 = getpart(line,3);
   t4 = getpart(line,4);
   t5 = getpart(line,5);
   
   u->cmd = caps(getpart(line,6));
   u->arg1 = getpart(line,7);
   u->arg2 = getpart(line,8);
   u->arg3 = getpart(line,9);
   
   len = strlen(t1) + strlen(t2) + strlen(t3) + strlen(t4) + strlen(t5) +5;
   if (u->cmd) len += strlen(u->cmd);
   
   strncpy(tempstr1,"",maxtextlength-1);
   if (u->arg1) {
      strncpy(tempstr1,line,maxtextlength-1);
      for (i=len+1; i<=sstrlen(line); i++)
         tempstr1[i-len-1] = tempstr1[i];
      }
   u->arg1e = tempstr1;
   
   strncpy(tempstr2,"",maxtextlength-1);
   if (u->arg1 && u->arg2) {
      len += strlen(u->arg1);
      strncpy(tempstr2,line,maxtextlength-1);
      for (i=len+2; i<=sstrlen(line); i++)
         tempstr2[i-len-2] = tempstr2[i];
      }
   u->arg2e = tempstr2;
   
   mydelete(t1);
   mydelete(t2);
   mydelete(t3);
   mydelete(t4);
   mydelete(t5);
   }

void u_fillwith_clean (userinput * const u) {
   
   updatecontext();
   
   mydelete(u->cmd);
   mydelete(u->arg1e);
   mydelete(u->arg2e);
   mydelete(u->arg1);
   mydelete(u->arg2);
   mydelete(u->arg3);
   }

static void u_respond(const userinput * const u, const char *format, ...)
{
  va_list args;

  updatecontext();
  
  va_start(args, format);
  
  switch (u->method)
    {
    case method_stdin:
      vioutput(CALLTYPE_NORMAL, OUT_S, COLOR_NO_COLOR, format, args);
      break;
    case method_dcc:
      vioutput(CALLTYPE_NORMAL, OUT_D, COLOR_NO_COLOR, format, args);
      break;
    case method_out_all:
      vioutput(CALLTYPE_NORMAL, OUT_S|OUT_L|OUT_D, COLOR_NO_COLOR, format, args);
      break;
    case method_fd:
      {
        ssize_t retval;
        char tempstr[maxtextlength];
        vsnprintf(tempstr,maxtextlength-2,format,args);
        removenonprintablectrl(tempstr);
        retval = write(u->fd, tempstr, strlen(tempstr));
        if (retval < 0)
          {
            outerror(OUTERROR_TYPE_WARN_LOUD,"Write failed: %s", strerror(errno));
          }
        tempstr[0] = '\n';
        retval = write(u->fd, tempstr, 1);
        if (retval < 0)
          {
            outerror(OUTERROR_TYPE_WARN_LOUD,"Write failed: %s", strerror(errno));
          }
      }
      break;
    case method_msg:
    case method_xdl_channel:
    case method_xdl_channel_min:
    case method_xdl_channel_sum:
      vprivmsg(u->snick, format, args);
      break;
    case method_xdl_user:
      vnotice(u->snick, format, args);
      break;
    case method_allow_all:
    default:
      break;
    }
  
  va_end(args);
}

void u_parseit(userinput * const u) {
   int i,found = 0;
   
   updatecontext();
   
   if (!u->cmd || !strlen(u->cmd)) {
      u_respond(u,"*** Missing Command, try again");
      u_fillwith_clean(u);
      return;
      }
   
   for (i=0; !found && i<(sizeof(userinput_parse)/sizeof(userinput_parse_t)); i++)
      if ( (!strcmp(userinput_parse[i].command,u->cmd)) &&
           (userinput_parse[i].methods_allowed & u->method) ) {
         found=1;
         userinput_parse[i].handler(u);
         }
   
   if (!found)
      u_respond(u,"*** User Command Not Recognized, try \"HELP\"");
   
   if (found && u->method==method_stdin)
      ioutput(CALLTYPE_NORMAL,OUT_S|OUT_L|OUT_D,COLOR_MAGENTA,"ADMIN %s Requested (stdin)",u->cmd);
   if (found && u->method==method_dcc)
      ioutput(CALLTYPE_NORMAL,OUT_S|OUT_L|OUT_D,COLOR_MAGENTA,"ADMIN %s Requested (DCC Chat)",u->cmd);
   if (found && u->method==method_msg)
      ioutput(CALLTYPE_NORMAL,OUT_S|OUT_L|OUT_D,COLOR_MAGENTA,"ADMIN %s Requested (MSG: %s)",u->cmd,u->snick);
   
   u_fillwith_clean(u);
   
   }

static void u_help(const userinput * const u)
{
  int i,which=0;
  char *tempstr2;
  
  updatecontext();
  
  tempstr2 = mycalloc(maxtextlengthshort);
  
  for (which=1; which<6; which++)
    {
      
      u_respond(u,"-- %s Commands --",
                which==1?"Info":
                which==2?"Transfer":
                which==3?"Pack":
                which==4?"Misc":
                which==5?"Bot":"<Unknown>");
      
      for (i=0; i<(sizeof(userinput_parse)/sizeof(userinput_parse_t)); i++)
        {
          if (userinput_parse[i].methods_allowed & u->method &&
              userinput_parse[i].help_section == which)
            {
              if (userinput_parse[i].args)
                {
                  snprintf(tempstr2,maxtextlengthshort-2,"%s %s",
                           userinput_parse[i].command,userinput_parse[i].args);
                }
              else
                {
                  snprintf(tempstr2,maxtextlengthshort-2,"%s",
                           userinput_parse[i].command);
                }
              u_respond(u,"  %-16s - %s", tempstr2, userinput_parse[i].desc);
            }
        }
    }
  u_respond(u,"For additional help, see the complete documentation at http://iroffer.org/");
  
  mydelete(tempstr2);
}

static void u_xdl(const userinput * const u) {
   char *tempstr,*tempstr2,*tempstr3;
   const char *spaces[] = { ""," ","  ","   ","    ","     ","      " };
   int a,i,p,m,m1,s;
   
   updatecontext();
   
   if ((u->method == method_xdl_channel || u->method == method_xdl_channel_min || u->method == method_xdl_channel_sum) && (gdata.nolisting > gdata.curtime)) return;
   
   if ((u->method == method_xdl_user) && (gdata.nolisting > gdata.curtime))
     {
       u_respond(u,"The Owner Has Requested That No Lists Be Sent In The Next %li Minute%s",
                 1+(gdata.nolisting-gdata.curtime)/60,((1+(gdata.nolisting-gdata.curtime)/60)!=1?"s":""));
       return;
     }

   tempstr  = mycalloc(maxtextlength);
   tempstr2 = mycalloc(maxtextlength);
   tempstr3 = mycalloc(maxtextlengthshort);

   if (u->method==method_xdl_channel_min) m = 1; else m = 0;
   if (u->method==method_xdl_channel_sum) m1 = 1; else m1 = 0;
   
   if (gdata.headline)
     {
       u_respond(u,"\2**\2 %s \2**\2",gdata.headline);
     }
   
   if (!m && !m1) {
      if (gdata.slotsmax-gdata.slotsfull < 0) a = gdata.slotsfull;
      else a = gdata.slotsmax;
      
      snprintf(tempstr,maxtextlength-2,
          "\2**\2 %i pack%s \2**\2  %i of %i slot%s open",
          gdata.numpacks,gdata.numpacks!=1?"s":"",a-gdata.slotsfull,a,a!=1?"s":"");
      if (gdata.slotsmax <= gdata.slotsfull) {
         snprintf(tempstr2,maxtextlength-2,"%s, Queue: %i/%i",tempstr,gdata.inqueue,gdata.queuesize);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      if (gdata.overallminspeed > 0) {
         snprintf(tempstr2,maxtextlength-2,"%s, Min: %1.1fKB/s",tempstr,gdata.overallminspeed);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      if (gdata.transfermaxspeed > 0) {
         snprintf(tempstr2,maxtextlength-2,"%s, Max: %1.1fKB/s",tempstr,gdata.transfermaxspeed);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      if (gdata.record > 0.5) {
         snprintf(tempstr2,maxtextlength-2,"%s, Record: %1.1fKB/s",tempstr,gdata.record);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      u_respond(u,"%s",tempstr);
      
      for (i=0,a=0; i<120; i++)
         a += gdata.xdccsent[i];
      snprintf(tempstr,maxtextlength-2,
          "\2**\2 Bandwidth Usage \2**\2 Current: %1.1fKB/s,",((float)a)/120.0/1024.0);
      if (gdata.maxb) {
         snprintf(tempstr2,maxtextlength-2,"%s Cap: %i.0KB/s,",tempstr,gdata.maxb/4);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      if (gdata.sentrecord > 0.5) {
         snprintf(tempstr2,maxtextlength-2,"%s Record: %1.1fKB/s",tempstr,gdata.sentrecord);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      u_respond(u,"%s",tempstr);
   
      u_respond(u,"\2**\2 To request a file type: \"/msg %s xdcc send #x\" \2**\2",
                gdata.user_nick);
      }
   
   if (m1) {
      u_respond(u,"\2**\2 For a listing type: \"/msg %s xdcc list\" \2**\2",
                gdata.user_nick);
      
      if (gdata.creditline)
	{
	  u_respond(u,"\2**\2 %s \2**\2",gdata.creditline);
	}
      
      mydelete(tempstr);
      mydelete(tempstr2);
      mydelete(tempstr3);
      return;
      }
   
   s = 0;
   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if (p>(MAXXDCCS-1))
         break;
      s = max2(s,gdata.xdccs[p]->gets);
      }
   i = s; s = 5;
   if (i < 10000) s = 4;
   if (i < 1000) s = 3;
   if (i < 100) s = 2;
   if (i < 10) s = 1;
   snprintf(tempstr3,maxtextlengthshort-2,"\2#%%-2i\2 %%%iix [%%s] %%s",s);
   
   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if (p>(MAXXDCCS-1))
         break;
      
      snprintf(tempstr,maxtextlength-2,
         tempstr3,
         i,
         gdata.xdccs[p]->gets,
         sizestr(1,tempstr2,gdata.xdccs[p]->st_size),
         gdata.xdccs[p]->desc);
      
      if (gdata.xdccs[p]->minspeed > 0 && gdata.xdccs[p]->minspeed != gdata.overallminspeed) {
         snprintf(tempstr2,maxtextlength-2,"%s [%1.1fiK Min]",tempstr,gdata.xdccs[p]->minspeed);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      
      if (gdata.xdccs[p]->maxspeed > 0 && gdata.xdccs[p]->maxspeed != gdata.transfermaxspeed) {
         snprintf(tempstr2,maxtextlength-2,"%s [%1.1fiK Max]",tempstr,gdata.xdccs[p]->maxspeed);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      
      u_respond(u,"%s",tempstr);
      
      if (gdata.xdccs[p]->note && strlen(gdata.xdccs[p]->note)) {
         u_respond(u," \2^-\2%s%s",spaces[s],gdata.xdccs[p]->note);
         }
      
      if (i == gdata.slotsmaxpack && gdata.inslotsmaxqueue) {
         u_respond(u," \2^-\2%sPack %i Has Slot Queue: %i/%i",
            spaces[s],i,gdata.inslotsmaxqueue,gdata.slotsmaxqueue);
         }
      }
   
   if (gdata.creditline) {
      u_respond(u,"\2**\2 %s \2**\2",gdata.creditline);
      }
   
   if (!m) {
      float toffered = 0;
   
      for (i=0; i<MAXXDCCS; i++)
         if (gdata.xdccs[i])
            toffered += (float)gdata.xdccs[i]->st_size;
   
      u_respond(u,
         "Total Offered: %1.1f MB  Total Transferred: %1.2f %cB",
         toffered/1024.0/1024.0,
         (gdata.totalsent/1024/1024) > 1024 ? ( (gdata.totalsent/1024/1024/1024) > 1024 ? ((float)gdata.totalsent)/1024/1024/1024/1024 : ((float)gdata.totalsent)/1024/1024/1024 ) : ((float)gdata.totalsent)/1024/1024 ,
         (gdata.totalsent/1024/1024) > 1024 ? ( (gdata.totalsent/1024/1024/1024) > 1024 ? 'T' : 'G' ) : 'M'
         );
      }
   
   mydelete(tempstr);
   mydelete(tempstr2);
   mydelete(tempstr3);
   }

static void u_xds(const userinput * const u) {
   updatecontext();
   
   xdccsave(0);
   }

static void u_dcl(const userinput * const u) {
   const char *y;
   int i,j;
   
   updatecontext();
   
   for (j=i=0; i<MAXUPLDS; i++)
      if (gdata.uploads[i] != NULL) j++;

   if (!gdata.slotsfull && !j) {
      u_respond(u,"No Active Transfers");
      return;
      }
   
   u_respond(u,"Current Transfer%s",gdata.slotsfull!=1?"s":"");
   u_respond(u,"   ID  User        File                               Status");
   
   for (i=0; i<MAXTRANS; i++)
      if (gdata.trans[i] != NULL) {
         
        switch (gdata.trans[i]->tr_status)
          {
          case TRANSFER_STATUS_LISTENING:
            y = "Listening";
            break;
            
          case TRANSFER_STATUS_SENDING:
            y = "Sending";
            break;
            
          case TRANSFER_STATUS_WAITING:
            y = "Finishing";
            break;
            
          case TRANSFER_STATUS_DONE:
            y = "Closing";
            break;
            
          default:
            y = "Unknown!";
            break;
          }
        
         if (gdata.trans[i]->tr_status == TRANSFER_STATUS_SENDING) {
            u_respond(u,"   %2i  %-9s   %-32s   %s %2.0f%%",
               i,gdata.trans[i]->nick,getfilename(gdata.trans[i]->xpack->file),y,
               ((float)gdata.trans[i]->bytessent)*100.0/((float)gdata.trans[i]->xpack->st_size));
            }
         else
            u_respond(u,"   %2i  %-9s   %-32s   %s",
               i,gdata.trans[i]->nick,getfilename(gdata.trans[i]->xpack->file),y);
         
         }
   
   for (i=0; i<MAXUPLDS; i++)
      if (gdata.uploads[i] != NULL) {
         
        switch (gdata.uploads[i]->ul_status)
          {
          case UPLOAD_STATUS_CONNECTING:
            y = "Connecting";
            break;
            
          case UPLOAD_STATUS_GETTING:
            y = "Getting";
            break;
            
          case UPLOAD_STATUS_WAITING:
            y = "Finishing";
            break;
            
          case UPLOAD_STATUS_DONE:
            y = "Done";
            break;
            
          default:
            y = "Unknown!";
            break;
          }
        
         if (gdata.uploads[i]->ul_status == UPLOAD_STATUS_GETTING) {
            u_respond(u,"   Ul  %-9s   %-32s   %s %2.0f%%",
               gdata.uploads[i]->nick,getfilename(gdata.uploads[i]->file),y,
               ((float)gdata.uploads[i]->bytesgot)*100.0/((float)gdata.uploads[i]->totalsize));
            }
         else
            u_respond(u,"   Ul  %-9s   %-32s   %s",
               gdata.uploads[i]->nick,getfilename(gdata.uploads[i]->file),y);
         
         }
   
   }

static void u_dcld(const userinput * const u) {
   char *tempstr2, *tempstr3, *tempstr4;
   const char *y;
   int i,j,left,started;
   
   updatecontext();
   
   for (j=i=0; i<MAXUPLDS; i++)
      if (gdata.uploads[i] != NULL) j++;

   if (!gdata.slotsfull && !j) {
      u_respond(u,"No Active Transfers");
      return;
      }
   
   tempstr2 = mycalloc(maxtextlength);
   tempstr3 = mycalloc(maxtextlengthshort);
   tempstr4 = mycalloc(maxtextlengthshort);

   u_respond(u,"Current Transfer%s",gdata.slotsfull!=1?"s":"");
   u_respond(u," ID  User        File                               Status");
   u_respond(u,"  ^-    Speed    Current/    End   Start/Remain    Min/  Max  Resumed");
   u_respond(u," --------------------------------------------------------------------");
   
   for (i=0; i<MAXTRANS; i++)
      if (gdata.trans[i] != NULL) {
         
        switch (gdata.trans[i]->tr_status)
          {
          case TRANSFER_STATUS_LISTENING:
            y = "Listening";
            break;
            
          case TRANSFER_STATUS_SENDING:
            y = "Sending";
            break;
            
          case TRANSFER_STATUS_WAITING:
            y = "Finishing";
            break;
            
          case TRANSFER_STATUS_DONE:
            y = "Closing";
            break;
            
          default:
            y = "Unknown!";
            break;
          }
         
         if (gdata.trans[i]->tr_status == TRANSFER_STATUS_SENDING) {
            u_respond(u," %2i  %-9s   %-32s   %s %2.0f%%",
               i,gdata.trans[i]->nick,getfilename(gdata.trans[i]->xpack->file),y,
               ((float)gdata.trans[i]->bytessent)*100.0/((float)gdata.trans[i]->xpack->st_size));
            }
         else
            u_respond(u," %2i  %-9s   %-32s   %s",
               i,gdata.trans[i]->nick,getfilename(gdata.trans[i]->xpack->file),y);
         
         if (gdata.trans[i]->tr_status == TRANSFER_STATUS_SENDING || gdata.trans[i]->tr_status == TRANSFER_STATUS_WAITING) {
            left = min2(359999,(gdata.trans[i]->xpack->st_size-gdata.trans[i]->bytessent)/((int)(max2(gdata.trans[i]->lastspeed,0.001)*1024)));
            started = min2(359999,gdata.curtime-gdata.trans[i]->connecttime);
            snprintf(tempstr2,maxtextlength-2,"%1.1fK",gdata.trans[i]->xpack->minspeed);
            snprintf(tempstr3,maxtextlengthshort-2,"%6liK",(long)(gdata.trans[i]->startresume)/1024);
            snprintf(tempstr4,maxtextlengthshort-2,"%1.1fK",gdata.trans[i]->xpack->maxspeed);
            
            u_respond(u,
                "  ^- %5.1fK/s    %6lliK/%6lliK  %2i%c%02i%c/%2i%c%02i%c  %5s/%5s  %7s",
                gdata.trans[i]->lastspeed,
                (long long)((gdata.trans[i]->bytessent)/1024),
                (long long)((gdata.trans[i]->xpack->st_size)/1024),
                started < 3600 ? started/60 : started/60/60 ,
                started < 3600 ? 'm' : 'h',
                started < 3600 ? started%60 : (started/60)%60 ,
                started < 3600 ? 's' : 'm',
                left < 3600 ? left/60 : left/60/60 ,
                left < 3600 ? 'm' : 'h',
                left < 3600 ? left%60 : (left/60)%60 ,
                left < 3600 ? 's' : 'm',
                (gdata.trans[i]->nomin || (gdata.trans[i]->xpack->minspeed == 0.0)) ? "no" : tempstr2 ,
                (gdata.trans[i]->nomax || (gdata.trans[i]->xpack->maxspeed == 0.0)) ? "no" : tempstr4 ,
                gdata.trans[i]->startresume ? tempstr3 : "no"
                );
            }
         else {
            u_respond(u,"  ^-    -----    -------/-------   -----/------  -----/-----      ---");
            }
         }
   
   for (i=0; i<MAXUPLDS; i++)
      if (gdata.uploads[i] != NULL) {
         
        switch (gdata.uploads[i]->ul_status)
          {
          case UPLOAD_STATUS_CONNECTING:
            y = "Connecting";
            break;
            
          case UPLOAD_STATUS_GETTING:
            y = "Getting";
            break;
            
          case UPLOAD_STATUS_WAITING:
            y = "Finishing";
            break;
            
          case UPLOAD_STATUS_DONE:
            y = "Done";
            break;
            
          default:
            y = "Unknown!";
            break;
          }
         
         if (gdata.uploads[i]->ul_status == UPLOAD_STATUS_GETTING) {
            u_respond(u," Ul  %-9s   %-32s   %s %2.0f%%",
               gdata.uploads[i]->nick,getfilename(gdata.uploads[i]->file),y,
               ((float)gdata.uploads[i]->bytesgot)*100.0/((float)gdata.uploads[i]->totalsize));
            }
         else
            u_respond(u," Ul  %-9s   %-32s   %s",
               gdata.uploads[i]->nick,getfilename(gdata.uploads[i]->file),y);
         
         if (gdata.uploads[i]->ul_status == UPLOAD_STATUS_GETTING || gdata.uploads[i]->ul_status == UPLOAD_STATUS_WAITING) {
            left = min2(359999,(gdata.uploads[i]->totalsize-gdata.uploads[i]->bytesgot)/((int)(max2(gdata.uploads[i]->lastspeed,0.001)*1024)));
            started = min2(359999,gdata.curtime-gdata.uploads[i]->connecttime);
            
            u_respond(u,
                "  ^- %5.1fK/s    %6lliK/%6lliK  %2i%c%02i%c/%2i%c%02i%c  -----/-----      ---",
                gdata.uploads[i]->lastspeed,
                (long long)((gdata.uploads[i]->bytesgot)/1024),
                (long long)((gdata.uploads[i]->totalsize)/1024),
                started < 3600 ? started/60 : started/60/60 ,
                started < 3600 ? 'm' : 'h',
                started < 3600 ? started%60 : (started/60)%60 ,
                started < 3600 ? 's' : 'm',
                left < 3600 ? left/60 : left/60/60 ,
                left < 3600 ? 'm' : 'h',
                left < 3600 ? left%60 : (left/60)%60 ,
                left < 3600 ? 's' : 'm'
                );
            }
         else {
            u_respond(u, "  ^-    -----    -------/-------   -----/------  -----/-----      ---");
            }
         
         }
      
   u_respond(u," --------------------------------------------------------------------");
   
   mydelete(tempstr2);
   mydelete(tempstr3);
   mydelete(tempstr4);
   }

static void u_qul(const userinput * const u) {
   int i, j;
   unsigned long rtime, lastrtime; 
  
   updatecontext();
   
   if (!gdata.inqueue && !gdata.inslotsmaxqueue) {
      u_respond(u,"No Users Queued");
      return;
      }
   
   u_respond(u,"Current Queue:");
   u_respond(u,"    #Q  User        File                              Waiting     Left");
   
   lastrtime=0;
   
   /* if we are sending more than allowed, we need to skip the difference */
   for (i=0; i<gdata.slotsfull-gdata.slotsmax; i++)
     {
       rtime=-1;
       for (j=0; j<MAXTRANS; j++)
	 if (gdata.trans[j]) {
	   int left = min2(359999,(gdata.trans[j]->xpack->st_size-gdata.trans[j]->bytessent)/((int)(max2(gdata.trans[j]->lastspeed,0.001)*1024)));
	   if (left > lastrtime && left < rtime)
	     rtime = left;
	 }
       if (rtime < 359999)
	 lastrtime=rtime;
     }
   
   for (i=1; i<=gdata.inqueue; i++) {
     
     rtime=-1;
     for (j=0; j<MAXTRANS; j++)
       if (gdata.trans[j]) {
	 int left = min2(359999,(gdata.trans[j]->xpack->st_size-gdata.trans[j]->bytessent)/((int)(max2(gdata.trans[j]->lastspeed,0.001)*1024)));
	 if (left > lastrtime && left < rtime)
	   rtime = left;
       }
     lastrtime=rtime;
     
     if (rtime < 359999)
      u_respond(u,"   %2iM  %-9s   %-32s   %2lih%2lim   %2lih%2lim",
         i,
         gdata.mainqueue[i-1]->nick,
         getfilename(gdata.mainqueue[i-1]->xpack->file),
         (gdata.curtime-gdata.mainqueue[i-1]->queuedtime)/60/60,
         ((gdata.curtime-gdata.mainqueue[i-1]->queuedtime)/60)%60,
         rtime/60/60,
         (rtime/60)%60);
     else
      u_respond(u,"   %2iM  %-9s   %-32s   %2lih%2lim  Unknown",
         i,
         gdata.mainqueue[i-1]->nick,
         getfilename(gdata.mainqueue[i-1]->xpack->file),
         (gdata.curtime-gdata.mainqueue[i-1]->queuedtime)/60/60,
         ((gdata.curtime-gdata.mainqueue[i-1]->queuedtime)/60)%60);
      }
   
   for (i=1; i<=gdata.inslotsmaxqueue; i++) {
      u_respond(u,"   %2iP  %-9s   %-32s   %2lih%2lim",
         i,
         gdata.packqueue[i-1]->nick,
         getfilename(gdata.packqueue[i-1]->xpack->file),
         (gdata.curtime-gdata.packqueue[i-1]->queuedtime)/60/60,
         ((gdata.curtime-gdata.packqueue[i-1]->queuedtime)/60)%60);
      }
   
   }

static void u_close(const userinput * const u) {
   
   int num = -1;
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (num > MAXTRANS-1 || num < 0 || gdata.trans[num] == NULL)
      u_respond(u,"Invalid ID number, Try \"DCL\" for a list");
   else
      t_closeconn(gdata.trans[num],"Owner Requested Close",0);
   
   }

static void u_nomin(const userinput * const u) {
   
   int num = -1;
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (num > MAXTRANS-1 || num < 0 || gdata.trans[num] == NULL)
      u_respond(u,"Invalid ID number, Try \"DCL\" for a list");
   else
      gdata.trans[num]->nomin = 1;
   
   }

static void u_nomax(const userinput * const u) {
   
   int num = -1;
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (num > MAXTRANS-1 || num < 0 || gdata.trans[num] == NULL)
      u_respond(u,"Invalid ID number, Try \"DCL\" for a list");
   else
      gdata.trans[num]->nomax = 1;
   
   }

static void u_rmq_rpq(const userinput * const u) {
   int i;
   int num = 0;
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (u->cmd[1] == 'M') {
      if ( (num > MAXQUEUE) || (num < 1) || gdata.mainqueue[num-1] == NULL )
         u_respond(u,"Invalid ID number, Try \"QUL\" for a list");
      else {
         notice(gdata.mainqueue[num-1]->nick,"*** Removed From Queue: Owner Requested Remove");
         mydelete(gdata.mainqueue[num-1]);
         for (i=num; i<gdata.inqueue; i++)
            gdata.mainqueue[i-1] = gdata.mainqueue[i];
         gdata.mainqueue[i] = NULL;
         gdata.inqueue--;
         }
      }
   else if (u->cmd[1] == 'P') {
      if ( (num > MAXQUEUE) || (num < 1) || gdata.packqueue[num-1] == NULL )
         u_respond(u,"Invalid ID number, Try \"QUL\" for a list");
      else {
         notice(gdata.packqueue[num-1]->nick,"*** Removed From Queue: Owner Requested Remove");
         mydelete(gdata.packqueue[num-1]);
         for (i=num; i<gdata.inslotsmaxqueue; i++)
            gdata.packqueue[i-1] = gdata.packqueue[i];
         gdata.packqueue[i] = NULL;
         gdata.inslotsmaxqueue--;
         }
      }
   else {
      u_respond(u,"internal iroffer error: u_rmq_rpq(): unexpected case, report problem to PMG");
      }
   
   }

static void u_raw(const userinput * const u) {
   updatecontext();
   
   writeserver2(u->arg1e,1);
   }

static void u_info(const userinput * const u) {
   int num = 0;
   int i,p;
   char *tempstr2;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (num < 1 || num > gdata.numpacks) {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
      }
   
   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if ( i == num )
         break;
      }
   
   tempstr2 = mycalloc(maxtextlength);
   
   u_respond(u,"Pack Info for Pack #%i:",num);
   
   u_respond(u,"Gets: %i, Size: %siB (%lli), Min: %1.1fKB/sec, Max: %1.1fKB/sec, Xfered: >%liMB",
             gdata.xdccs[p]->gets,
             sizestr(1,tempstr2,gdata.xdccs[p]->st_size),
             (long long)gdata.xdccs[p]->st_size,
             gdata.xdccs[p]->minspeed,
             gdata.xdccs[p]->maxspeed,
             (long)(((double)gdata.xdccs[p]->gets)*((double)gdata.xdccs[p]->st_size)/1024.0/1024.0));
   u_respond(u,"Filename:    %s",gdata.xdccs[p]->file);
   u_respond(u,"Description: %s",gdata.xdccs[p]->desc);
   u_respond(u,"Note:        %s",gdata.xdccs[p]->note);
   
   mydelete(tempstr2);
   }

static void u_remove(const userinput * const u) {
   int i,t,p,q,x;
   int num = 0;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if ( num < 1 || num > gdata.numpacks ) {
      u_respond(u,"Try a valid pack number");
      return;
      }
   
   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if ( i == num )
         break;
      }
   
   for (t=0; t<MAXTRANS; t++)
     {
       if ((gdata.trans[t] != NULL) &&
           (gdata.trans[t]->tr_status != TRANSFER_STATUS_DONE) &&
           (gdata.trans[t]->xpack == gdata.xdccs[p]))
         {
           t_closeconn(gdata.trans[t],"Pack removed",0);
         }
     }
   
   for (q=0; q<gdata.inqueue; q++)
     {
       if ((gdata.mainqueue[q] != NULL) &&
           (gdata.mainqueue[q]->xpack == gdata.xdccs[p]))
         {
           notice(gdata.mainqueue[q]->nick,"*** Removed From Queue: Pack removed");
           mydelete(gdata.mainqueue[q]);
           for (x=q+1; x<gdata.inqueue; x++)
             {
               gdata.mainqueue[x-1] = gdata.mainqueue[x];
             }
           gdata.mainqueue[x] = NULL;
           gdata.inqueue--;
           q--;
         }
     }
   
   for (q=0; q<gdata.inslotsmaxqueue; q++)
     {
       if ((gdata.packqueue[q] != NULL) &&
           (gdata.packqueue[q]->xpack == gdata.xdccs[p]))
         {
           notice(gdata.packqueue[q]->nick,"*** Removed From Queue: Pack removed");
           mydelete(gdata.packqueue[q]);
           for (x=q+1; x<gdata.inslotsmaxqueue; x++)
             {
               gdata.packqueue[x-1] = gdata.packqueue[x];
             }
           gdata.packqueue[x] = NULL;
           gdata.inslotsmaxqueue--;
           q--;
         }
     }
   
   u_respond(u,"Removed Pack %i [%s]", num, gdata.xdccs[p]->desc);
   
   mydelete(gdata.xdccs[p]->file);
   mydelete(gdata.xdccs[p]->desc);
   mydelete(gdata.xdccs[p]->note);
   mydelete(gdata.xdccs[p]);
   
   for (i=p+1; i<=gdata.numpacks; i++)
      gdata.xdccs[i-1] = gdata.xdccs[i];
   
   gdata.numpacks--;
   xdccsave(0);
   }

static void u_redraw(const userinput * const u) {
   updatecontext();
   
   initscreen();
   gotobot();
   }

static void u_send(const userinput * const u) {
   int num = 0;
   updatecontext();
   
   if (u->arg2) num = atoi(u->arg2);
   
   if (!u->arg1 || !strlen(u->arg1)) {
      u_respond(u,"Try Specifying a Nick");
      return;
      }
   
   if (num > gdata.numpacks || num < 1) {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
      }
   
   u_respond(u,"Sending %s pack %i",u->arg1,num);
   
   sendxdccfile(u->arg1,"man",num,NULL);
   
   }

static void u_psend(const userinput * const u) {
   userinput manplist;
  
   updatecontext();
   
   if (!u->arg1 || !strlen(u->arg1)) {
      u_respond(u,"Try Specifying a Channel");
      return;
      }

   u_fillwith_msg(&manplist,u->arg1,"A A A A A xdl");
   
   if (!u->arg2 || !strlen(u->arg2)) {
      manplist.method = method_xdl_channel;
      u_respond(u, "No PLIST style specified. Using style full");
      }
   else {
      if (strcmp(u->arg2,"full") == 0)
         manplist.method = method_xdl_channel;
      else if (strcmp(u->arg2,"minimal") == 0)
         manplist.method = method_xdl_channel_min;
      else if (strcmp(u->arg2,"summary") == 0)
         manplist.method = method_xdl_channel_sum;
      else {
         u_respond(u, "PLIST format is not (full|minimal|summary)");
         return;
         }
      }
  
   u_parseit(&manplist);

   u_respond(u,"Sending PLIST with style %s to %s",u->arg2,u->arg1);
   }

static void u_msg(const userinput * const u)
{
  updatecontext();
  
  if (!u->arg1 || !strlen(u->arg1))
    {
      u_respond(u,"Try Specifying a Nick");
      return;
    }
  
  if (!u->arg2e || !strlen(u->arg2e))
    {
      u_respond(u,"Try Specifying a Message");
      return;
    }
  
  privmsg(u->arg1,"%s",u->arg2e);
  
}

static void u_mesg(const userinput * const u) {
   int i;
   
   updatecontext();
   
   if (!u->arg1e || !strlen(u->arg1e)) {
      u_respond(u,"Try Specifying a Message");
      return;
      }
   
   for (i=0; i<MAXTRANS; i++)
      if (gdata.trans[i] != NULL) {
      notice(gdata.trans[i]->nick,"MESSAGE FROM OWNER: %s",u->arg1e);
      }
   
   u_respond(u,"Sent message to %i user%s",gdata.slotsfull,gdata.slotsfull!=1?"s":"");
   
   }

static void u_mesq(const userinput * const u) {
   int i,count;
   
   updatecontext();
   
   if (!u->arg1e || !strlen(u->arg1e)) {
      u_respond(u,"Try Specifying a Message");
      return;
      }
   
   count=0;
   for (i=1; i<=gdata.inqueue; i++) {
      notice(gdata.mainqueue[i-1]->nick,"MESSAGE FROM OWNER: %s",u->arg1e);
      count++;
      }
   for (i=1; i<=gdata.inslotsmaxqueue; i++) {
      notice(gdata.packqueue[i-1]->nick,"MESSAGE FROM OWNER: %s",u->arg1e);
      count++;
      }
   u_respond(u,"Sent message to %i user%s",count,count!=1?"s":"");
   
   }

static void u_closedcc(const userinput * const u) {
   updatecontext();
   
      ioutput(CALLTYPE_NORMAL,OUT_S|OUT_L,COLOR_MAGENTA,"DCC CHAT: QUIT");
      writedccchat("Bye.");
      FD_CLR(gdata.dccchat, &gdata.readset);
      usleep(100*1000);
      /*
       * cygwin close() is broke, if outstanding data is present
       * it will block until the TCP connection is dead, sometimes
       * upto 10-20 minutes, calling shutdown() first seems to help
       */
      shutdown(gdata.dccchat, SHUT_RDWR);
      close(gdata.dccchat);
      gdata.dccchat = FD_UNUSED;
      highestsock();
   }

static void u_status(const userinput * const u) {
   char *tempstr = mycalloc(maxtextlength);
   
   updatecontext();
   
   getstatusline(tempstr);
   u_respond(u,"%s",tempstr);
   
   mydelete(tempstr);
   }

static void u_chfile(const userinput * const u) {
   int num = 0;
   int i,p,t,xfiledescriptor;
   struct stat st;
   char tempstr[maxtextlength];
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (num < 1 || num > gdata.numpacks) {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
      }
   
   if (!u->arg2e || !strlen(u->arg2e) ||
       (strlen(u->arg2e) > (maxtextlength-1))) {
      u_respond(u,"Try Specifying a Filename");
      return;
      }
   
   /* verify file is ok first */
   tempstr[0] = '\0';
   
   convert_to_unix_slash(u->arg2e);
   
   xfiledescriptor=open(u->arg2e, O_RDONLY | ADDED_OPEN_FLAGS);
   
   if (xfiledescriptor < 0 && (errno == ENOENT) && gdata.filedir) {
      snprintf(tempstr,maxtextlength-2,"%s/%s",gdata.filedir,u->arg2e);
      convert_to_unix_slash(tempstr);
      xfiledescriptor=open(tempstr, O_RDONLY | ADDED_OPEN_FLAGS);
      }
   
   if (xfiledescriptor < 0) {
      u_respond(u,"Cant Access File: %s",strerror(errno));
      return;
      }
   
   if (fstat(xfiledescriptor,&st) < 0)
     {
      u_respond(u,"Cant Access File Details: %s",strerror(errno));
      close(xfiledescriptor);
      return;
     }
      
   close(xfiledescriptor);
   
   if ( st.st_size == 0 ) {
      u_respond(u,"File has size of 0 bytes!");
      return;
      }
   
   if ((st.st_size > gdata.max_file_size) || (st.st_size < 0)) {
      u_respond(u,"File is too large.");
      return;
      }
   
   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if ( i == num )
         break;
      }
   
   for (t=0; t<MAXTRANS; t++)
     {
       if ((gdata.trans[t] != NULL) &&
           (gdata.trans[t]->tr_status != TRANSFER_STATUS_DONE) &&
           (gdata.trans[t]->xpack == gdata.xdccs[p]))
         {
           t_closeconn(gdata.trans[t],"Pack file changed",0);
         }
     }
   
   u_respond(u, "CHFILE: [Pack %i] Old: %s New: %s",
             num,gdata.xdccs[p]->file,tempstr[0] ? tempstr : u->arg2e);
   
   mydelete(gdata.xdccs[p]->file);
   gdata.xdccs[p]->file = mycalloc(strlen(tempstr[0] ? tempstr : u->arg2e) + 1);
   
   strcpy(gdata.xdccs[p]->file, tempstr[0] ? tempstr : u->arg2e);
   gdata.xdccs[p]->st_size  = st.st_size;
   gdata.xdccs[p]->st_dev   = st.st_dev;
   gdata.xdccs[p]->st_ino   = st.st_ino;
   gdata.xdccs[p]->mtime    = st.st_mtime;
   
   
   xdccsave(0);
   
   }

static void u_add(const userinput * const u) {
   int i,xfiledescriptor;
   struct stat st;
   
   updatecontext();
   
   if (!u->arg1e || !strlen(u->arg1e)) {
      u_respond(u,"Try Specifying a Filename");
      return;
      }
   
   for (i=0; gdata.xdccs[i]; i++) ;
   
   if (i==MAXXDCCS) {
      u_respond(u,"Max Number of Packs Reached");
      return;
      }
   
   gdata.xdccs[i] = mycalloc(sizeof(xdcc));
   
   gdata.xdccs[i]->file = mycalloc(strlen(u->arg1e)+1);
   strcpy(gdata.xdccs[i]->file,u->arg1e);
   
   gdata.xdccs[i]->note = mycalloc(1);
   strcpy(gdata.xdccs[i]->note,"");
   
   gdata.xdccs[i]->desc = mycalloc(strlen(getfilename(u->arg1e)) + 1);
   strcpy(gdata.xdccs[i]->desc,getfilename(u->arg1e));
   
   convert_to_unix_slash(gdata.xdccs[i]->file);
   
   gdata.xdccs[i]->gets = 0;
   gdata.xdccs[i]->minspeed = gdata.overallminspeed;
   gdata.xdccs[i]->maxspeed = gdata.transfermaxspeed;

   xfiledescriptor=open(gdata.xdccs[i]->file, O_RDONLY | ADDED_OPEN_FLAGS);
   
   if (xfiledescriptor < 0 && (errno == ENOENT) && gdata.filedir)
     {
       mydelete(gdata.xdccs[i]->file);
       gdata.xdccs[i]->file = mycalloc(strlen(gdata.filedir)+1+strlen(u->arg1e)+1);
       snprintf(gdata.xdccs[i]->file,maxtextlength-2,"%s/%s",gdata.filedir,u->arg1e);
       convert_to_unix_slash(gdata.xdccs[i]->file);
       xfiledescriptor=open(gdata.xdccs[i]->file, O_RDONLY | ADDED_OPEN_FLAGS);
     }
   
   if (xfiledescriptor < 0) {
      u_respond(u,"Cant Access File: %s",strerror(errno));
      mydelete(gdata.xdccs[i]->file);
      mydelete(gdata.xdccs[i]->desc);
      mydelete(gdata.xdccs[i]->note);
      mydelete(gdata.xdccs[i]);
      return;
      }
   
   if (fstat(xfiledescriptor,&st) < 0)
     {
      u_respond(u,"Cant Access File Details: %s",strerror(errno));
      mydelete(gdata.xdccs[i]->file);
      mydelete(gdata.xdccs[i]->desc);
      mydelete(gdata.xdccs[i]->note);
      mydelete(gdata.xdccs[i]);
      close(xfiledescriptor);
      return;
     }
      
   close(xfiledescriptor);
   
   gdata.xdccs[i]->st_size  = st.st_size;
   gdata.xdccs[i]->st_dev   = st.st_dev;
   gdata.xdccs[i]->st_ino   = st.st_ino;
   gdata.xdccs[i]->mtime    = st.st_mtime;
   
   if ( gdata.xdccs[i]->st_size == 0 ) {
      u_respond(u,"File has size of 0 bytes!");
      mydelete(gdata.xdccs[i]->file);
      mydelete(gdata.xdccs[i]->desc);
      mydelete(gdata.xdccs[i]->note);
      mydelete(gdata.xdccs[i]);
      return;
      }
   
   if ((gdata.xdccs[i]->st_size > gdata.max_file_size) || (gdata.xdccs[i]->st_size < 0)) {
      u_respond(u,"File is too large.");
      mydelete(gdata.xdccs[i]->file);
      mydelete(gdata.xdccs[i]->desc);
      mydelete(gdata.xdccs[i]->note);
      mydelete(gdata.xdccs[i]);
      return;
      }
   
   gdata.numpacks++;
   
   u_respond(u, "ADD PACK: [Pack: %i] [File: %s] Use CHDESC to change description",
             gdata.numpacks,gdata.xdccs[i]->file);
   
   xdccsave(0);
   
   }

static void u_adddir(const userinput * const u) {
   int count=0;
   DIR *d;
   struct dirent *f;
   char *tempstr, *thedir;
   
   updatecontext();
   
   if (!u->arg1e || !strlen(u->arg1e)) {
      u_respond(u,"Try Specifying a Directory");
      return;
      }
   
   if (u->arg1e[strlen(u->arg1e)-1] == '/') u->arg1e[strlen(u->arg1e)-1] = '\0';
   if (u->arg1e[strlen(u->arg1e)-1] == '\\') u->arg1e[strlen(u->arg1e)-1] = '\0';
   
   thedir = mycalloc(maxtextlength);
   strncpy(thedir,u->arg1e,maxtextlength-1);
   
   d = opendir(thedir);
   
   if (!d && (errno == ENOENT) && gdata.filedir) {
      snprintf(thedir,maxtextlength-2,"%s/%s",gdata.filedir,u->arg1e);
      d = opendir(thedir);
      }
   
   if (!d) {
      u_respond(u,"Cant Access Directory: %s",strerror(errno));
      return;
      }
   
   tempstr = mycalloc(maxtextlength);
   
   u_respond(u,"Adding contents of %s...",thedir);
   
   while ((f = readdir(d))) {
      if (strcmp(".",f->d_name) && strcmp("..",f->d_name)) {
         count++;
         u_respond(u,"  Adding %s:",f->d_name);
         snprintf(tempstr,maxtextlength-2,"%s/%s",thedir,f->d_name);
         strncpy(u->arg1e,tempstr,maxtextlength-1);
         u_add(u);
         }
      }
   
   u_respond(u,"Total of %i files added",count);
   
   mydelete(tempstr);
   mydelete(thedir);
   closedir(d);
   
   }

static void u_chdesc(const userinput * const u) {
   int num = 0;
   int i,p;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   if (num < 1 || num > gdata.numpacks) {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
      }
   
   if (!u->arg2e || !strlen(u->arg2e)) {
      u_respond(u,"Try Specifying a Description");
      return;
      }
   
   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if ( i == num )
         break;
      }
   
   u_respond(u, "CHDESC: [Pack %i] Old: %s New: %s",
      num,gdata.xdccs[p]->desc,u->arg2e);
   
   mydelete(gdata.xdccs[p]->desc);
   gdata.xdccs[p]->desc = mycalloc(strlen(u->arg2e) + 1);
   
   strcpy(gdata.xdccs[p]->desc,u->arg2e);
   
   xdccsave(0);
   
   }

static void u_chnote(const userinput * const u) {
   int num = 0;
   int i,p;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (num < 1 || num > gdata.numpacks) {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
      }
   
   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if ( i == num )
         break;
      }
   
   u_respond(u, "CHNOTE: [Pack %i] Old: %s New: %s",
             num,gdata.xdccs[p]->note,u->arg2e);
   
   mydelete(gdata.xdccs[p]->note);
   
   if (!u->arg2e)
     {
       gdata.xdccs[p]->note = mycalloc(1);
       strcpy(gdata.xdccs[p]->note,"");
     }
   else
     {
       gdata.xdccs[p]->note = mycalloc(strlen(u->arg2e) + 1);
       strcpy(gdata.xdccs[p]->note,u->arg2e);
     }
   
   xdccsave(0);
   
   }

static void u_chmins(const userinput * const u) {
   int num = 0;
   int i,p;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   if (num < 1 || num > gdata.numpacks) {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
      }
   
   if (!u->arg2 || !strlen(u->arg2)) {
      u_respond(u,"Try Specifying a Minspeed");
      return;
      }

   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if ( i == num )
         break;
      }
   
   u_respond(u, "CHMINS: [Pack %i] Old: %1.1f New: %1.1f",
             num,gdata.xdccs[p]->minspeed,atof(u->arg2));
   
   gdata.xdccs[p]->minspeed = gdata.overallminspeed;
   if ( atof(u->arg2) != gdata.overallminspeed )
      gdata.xdccs[p]->minspeed = atof(u->arg2);
   
   xdccsave(0);
   
   }

static void u_chmaxs(const userinput * const u) {
   int num = 0;
   int i,p;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   if (num < 1 || num > gdata.numpacks) {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
      }
   
   if (!u->arg2 || !strlen(u->arg2)) {
      u_respond(u,"Try Specifying a Maxspeed");
      return;
      }

   for (i=1, p=0; i<=gdata.numpacks; i++, p++) {
      while (gdata.xdccs[p] == NULL)
         p++;
      if ( i == num )
         break;
      }
   
   u_respond(u, "CHMAXS: [Pack %i] Old: %1.1f New: %1.1f",
             num,gdata.xdccs[p]->maxspeed,atof(u->arg2));
   
   gdata.xdccs[p]->maxspeed = gdata.transfermaxspeed;
   if ( atof(u->arg2) != gdata.transfermaxspeed )
      gdata.xdccs[p]->maxspeed = atof(u->arg2);
   
   xdccsave(0);
   
   }

static void u_chgets(const userinput * const u)
{
  int num = 0;
  int i,p;
  
  updatecontext();
  
  if (u->arg1)
    {
      num = atoi(u->arg1);
    }
  
  if (num < 1 || num > gdata.numpacks)
    {
      u_respond(u,"Try Specifying a Valid Pack Number");
      return;
    }
  
  if (!u->arg2 || !strlen(u->arg2))
    {
      u_respond(u,"Try Specifying a Count");
      return;
    }
  
  for (i=1, p=0; i<=gdata.numpacks; i++, p++)
    {
      while (gdata.xdccs[p] == NULL)
        {
          p++;
        }
      if ( i == num )
        {
          break;
        }
    }
  
  u_respond(u, "CHGETS: [Pack %i] Old: %d New: %d",
            num,gdata.xdccs[p]->gets,atoi(u->arg2));
  
  gdata.xdccs[p]->gets = atoi(u->arg2);
  
  xdccsave(0);
}

static void u_chatme(const userinput * const u) {
   
   updatecontext();
   
   u_respond(u,"Sending You A DCC Chat Request");
   
   if (setupdccchatout(u->snick))
      u_respond(u,"[Failed to listen, try again]");
   
   
   }

static void u_rehash(const userinput * const u) {
   
   /* other variables */
   char *tempbuf = mycalloc(maxtextlength);
   char *templine = mycalloc(maxtextlength);
   int h,i,j,k,len,found,filedescriptor,needtojump;
   
   updatecontext();
   
   if (u->method==method_out_all) u_respond(u,"Caught SIGUSR2, Rehashing...");
   
   /* clear old config items */
   for (i=0; i<MAXAHOST; i++)
      mydelete(gdata.adminhost[i]);
   for (i=0; i<MAXSRVS; i++)
      mydelete(gdata.server[i]);
   for (i=0; i<MAXCHNLS; i++)
      gdata.t_channel[i] = gdata.r_channel[i] = NULL;
   
   gdata.r_overallminspeed = gdata.overallminspeed;
   gdata.r_transfermaxspeed = gdata.transfermaxspeed;
   
   if (gdata.logfd != FD_UNUSED)
     {
       close(gdata.logfd);
       gdata.logfd = FD_UNUSED;
     }
   
   mydelete(gdata.logfile);
   gdata.logrotate = gdata.logstats = 0;
   mydelete(gdata.ignorefile);
   mydelete(gdata.messagefile);
   mydelete(gdata.user_realname);
   mydelete(gdata.user_modes);
   gdata.firewall = 0;
   gdata.dccrangestart = 4000;
   mydelete(gdata.proxyinfo);
   mydelete(gdata.proxyinfo2);
   gdata.usenatip = 0;
   gdata.slotsmax = gdata.queuesize = gdata.slotsmaxpack = gdata.slotsmaxslots = gdata.slotsmaxqueue = 0;
   gdata.maxtransfersperperson = 1;
   gdata.maxqueueditemsperperson = 1;
   mydelete(gdata.filedir);
   gdata.lowbdwth = 0;
   gdata.overallminspeed = gdata.transfermaxspeed = 0.0;
   gdata.overallmaxspeed = gdata.overallmaxspeeddayspeed = 0;
   gdata.overallmaxspeeddaytimestart = gdata.overallmaxspeeddaytimeend = 0;
   gdata.overallmaxspeeddaydays = 0x7F; /* all days */
   gdata.debug = 0;
   gdata.autosend = gdata.autopack = 0;
   mydelete(gdata.autoword);
   mydelete(gdata.automsg);
   gdata.xdccautosavetime = 0;
   gdata.hideos = 0;
   gdata.lognotices = 0;
   gdata.notifytime = 5;
   gdata.respondtochannelxdcc = 0;
   gdata.respondtochannellist = 0;
   gdata.smallfilebypass = 0;
   mydelete(gdata.creditline);
   mydelete(gdata.headline);
   mydelete(gdata.nickserv_pass);
   mydelete(gdata.periodicmsg_nick);
   mydelete(gdata.periodicmsg_msg);
   gdata.periodicmsg_time = 0;
   gdata.uploadmaxsize = 0;
   gdata.uploadallowed = 0;
   mydelete(gdata.uploaddir);
   gdata.restrictlist = gdata.restrictsend = gdata.restrictprivlist = 0;
   
   mydelete(gdata.loginname);
   set_loginname();
   
   /* go */
   for (h=0; h<MAXCONFIG && gdata.configfile[h]; h++) {
      u_respond(u,"Reloading %s ...",gdata.configfile[h]);
      
      filedescriptor=open(gdata.configfile[h], O_RDONLY | ADDED_OPEN_FLAGS);
      if (filedescriptor < 0) {
         u_respond(u,"Cant Access File, Aborting rehash: %s",strerror(errno));
         u_respond(u,"***WARNING*** missing vital information, fix and re-rehash ASAP");
         mydelete(tempbuf);
         mydelete(templine);
         return;
         }
   
      i = 0;
      while((len = read(filedescriptor,tempbuf,maxtextlength)) > 0)
        {
          for (j=0; j<len; j++)
             {
               if (tempbuf[j] == '\n' || tempbuf[j] == 13)  /* 13 is ^M */
                 {
                   templine[i] = '\0';
                   if ( templine[0] != '#' && strlen(templine) != 0 )
                     {
                       for (i=strlen(templine)-1; i>2 && templine[i] == ' '; i--)
                         {
                           templine[i] = '\0';
                         }
                       getconfig_set(templine,1);
                     }
                   i = 0;
                 }
               else
                 {
                   templine[i] = tempbuf[j];
                   i++;
                 }
             }
        }
      
      close(filedescriptor);
     }
   
   /* see what needs to be redone */
   
   u_respond(u,"Reconfiguring...");
   
   needtojump=0;
   if ( gdata.virthost != gdata.r_virthost ||
       (gdata.virthost && gdata.r_virthost && gdata.vhost_ip && gdata.r_vhost_ip && strcmp(gdata.vhost_ip,gdata.r_vhost_ip)) )
      needtojump=1;
   
   k=0;
   /* part deleted channels, add common channels */
   for (i=0; i<MAXCHNLS && gdata.channels[i]; i++) {
      found = 0;
      for (j=0; j<MAXCHNLS && gdata.r_channel[j] && !found; j++)
         if (!strcmp(gdata.channels[i]->name,gdata.r_channel[j]->name)) found = 1;
      j--;
      if (!found) {
         snprintf(templine,maxtextlength-2,"PART %s",gdata.channels[i]->name);
         if (!needtojump) writeserver(templine);
	 clearmemberlist(gdata.channels[i]);
         if (gdata.debug > 2)
           {
             ioutput(CALLTYPE_NORMAL,OUT_S,COLOR_NO_COLOR,
                     "1 channels[%i] = %s parted\n",i,gdata.channels[i]->name);
           }
         }
      else {
         gdata.t_channel[k] = mycalloc(sizeof(channel_t));
         memcpy(gdata.t_channel[k],gdata.r_channel[j],sizeof(channel_t));
         gdata.t_channel[k]->flags |= gdata.channels[i]->flags & CHAN_ONCHAN;
	 gdata.t_channel[k]->members = gdata.channels[i]->members;
         if (gdata.debug > 2)
           {
             ioutput(CALLTYPE_NORMAL,OUT_S,COLOR_NO_COLOR,
                     "2 t_channel[%i] <- r_channel[%i] = %s common\n",
                     k,i,gdata.channels[i]->name);
           }
         k++;
         }
      }
   
   /* join/add new channels */
   for (i=0; i<MAXCHNLS && gdata.r_channel[i]; i++) {
      found = 0;
      for (j=0; j<MAXCHNLS && gdata.channels[j]; j++)
         if (!strcmp(gdata.r_channel[i]->name,gdata.channels[j]->name)) found = 1;
      if (!found) {
         gdata.t_channel[k] = mycalloc(sizeof(channel_t));
         memcpy(gdata.t_channel[k],gdata.r_channel[i],sizeof(channel_t));
         if (!needtojump) joinchannel(gdata.t_channel[k]);
         gdata.t_channel[k]->flags &= ~CHAN_ONCHAN;
         if (gdata.debug > 2)
           {
             ioutput(CALLTYPE_NORMAL,OUT_S,COLOR_NO_COLOR,
                     "3 t_channel[%i] <- r_channel[%i] = %s new\n",
                     k,i,gdata.r_channel[i]->name);
           }
         k++;
         }
      }
   
   /* kill old channel lists */
   for (i=0; i<MAXCHNLS && gdata.channels[i]; i++)
      mydelete(gdata.channels[i]);
   for (i=0; i<MAXCHNLS && gdata.r_channel[i]; i++)
      mydelete(gdata.r_channel[i]);
   
   /* switch to new channel list */
   for (i=0; i<MAXCHNLS && gdata.t_channel[i]; i++) {
      gdata.channels[i] = gdata.t_channel[i];
      gdata.t_channel[i] = NULL;
      
      if (gdata.debug > 2)
        {
          ioutput(CALLTYPE_NORMAL,OUT_S,COLOR_NO_COLOR,
                  "4 channels[%i] = %s\n",i,gdata.channels[i]->name);
        }
      }
   
   if (needtojump) {
      u_respond(u,"vhost changed, reconnecting");
      mydelete(gdata.vhost_ip);
      gdata.vhost_ip = gdata.r_vhost_ip;
      gdata.r_vhost_ip = NULL;
      gdata.virthost = gdata.r_virthost;
      switchserver(-1);
      /* switchserver takes care of joining channels */
      }
   else {
      mydelete(gdata.vhost_ip);
      gdata.vhost_ip = gdata.r_vhost_ip;
      gdata.r_vhost_ip = NULL;
      gdata.virthost = gdata.r_virthost;
      }
   
   if (strcmp(gdata.pidfile,gdata.r_pidfile)) {
      u_respond(u,"pidfile changed, switching");
      unlink(gdata.pidfile);
      writepidfile(gdata.r_pidfile);
      }
   mydelete(gdata.pidfile);
   gdata.pidfile = gdata.r_pidfile;
   gdata.r_pidfile = NULL;
   
   if (strcmp(gdata.config_nick,gdata.r_config_nick)) {
      u_respond(u,"user_nick changed, renaming nick to %s", gdata.r_config_nick);
      snprintf(templine,maxtextlength-2,"NICK %s",gdata.r_config_nick);
      writeserver2(templine,1);
      }
   mydelete(gdata.config_nick);
   gdata.config_nick = gdata.r_config_nick;
   gdata.r_config_nick = NULL;
   
   
   if (strcmp(gdata.xdccfile,gdata.r_xdccfile))
      u_respond(u,"sorry, changing xdccfile is yet implemented");
   mydelete(gdata.r_xdccfile);
   
   
   gdata.maxb = gdata.overallmaxspeed;
   if (gdata.overallmaxspeeddayspeed != gdata.overallmaxspeed) {
      struct tm *localt;
      localt = localtime(&gdata.curtime);

      if (localt->tm_hour >= gdata.overallmaxspeeddaytimestart
          && localt->tm_hour < gdata.overallmaxspeeddaytimeend
          && ( gdata.overallmaxspeeddaydays & (1 << localt->tm_wday)) )
         gdata.maxb = gdata.overallmaxspeeddayspeed;
      }
   
   if ( gdata.r_overallminspeed != gdata.overallminspeed)
      for (i=0; i<MAXXDCCS; i++)
         if (gdata.xdccs[i] && (gdata.xdccs[i]->minspeed == gdata.r_overallminspeed))
            gdata.xdccs[i]->minspeed = gdata.overallminspeed;
   
   if ( gdata.r_transfermaxspeed != gdata.transfermaxspeed)
      for (i=0; i<MAXXDCCS; i++)
         if (gdata.xdccs[i] && (gdata.xdccs[i]->maxspeed == gdata.r_transfermaxspeed))
            gdata.xdccs[i]->maxspeed = gdata.transfermaxspeed;
   
   /* check for completeness */
   u_respond(u,"Checking for completeness of config file ...");
   
   if ( gdata.server[0] == NULL
        || gdata.config_nick == NULL || gdata.user_realname == NULL
        || gdata.user_modes == NULL || gdata.channels[0] == NULL
        || gdata.slotsmax == 0 || gdata.xdccautosavetime == 0 || gdata.xdccfile == 0)
      u_respond(u,"***WARNING*** missing vital information, fix and re-rehash ASAP");
   
   if ( gdata.autosend && ( gdata.autoword == NULL || gdata.automsg == NULL || !gdata.autopack ) )
      u_respond(u,"***WARNING*** incomplete autosend information, fix and re-rehash ASAP");
   
   if ( gdata.uploadallowed && ( gdata.uploaddir == NULL || strlen(gdata.uploaddir) < 2 ) )
      u_respond(u,"***WARNING*** incomplete upload information, fix and re-rehash ASAP");
      
   u_respond(u,"Done.");
   
   for (i=0; i<100; i++)
     {
       if (!gdata.exiting &&
           (gdata.inqueue || gdata.inslotsmaxqueue) &&
           (gdata.slotsfull < min2(ACTUAL_MAXTRANS,gdata.slotsmax)))
         {
           sendaqueue(0);
         }
     }
   
   mydelete(tempbuf);
   mydelete(templine);

   }

static void u_botinfo(const userinput * const u) {
   char *tempstr = mycalloc(maxtextlength);
   char *tempstr2 = mycalloc(maxtextlength);
   int i;
   struct rusage r;

   updatecontext();
   
   u_respond(u,"BotInfo:");
   
   u_respond(u,"iroffer v" VERSIONLONG ", http://iroffer.org/%s%s",
             gdata.hideos ? "" : " - ",
             gdata.hideos ? "" : gdata.osstring);

   getuptime(tempstr2,0,gdata.startuptime);
   u_respond(u,"iroffer started up %s ago",tempstr2);

   getuptime(tempstr2,0,gdata.curtime-gdata.totaluptime);
   u_respond(u,"total running time of %s",tempstr2);

   getrusage(RUSAGE_SELF,&r);
   
   u_respond(u,"cpu usage: %2.2fs user (%2.5f%%), %2.2fs system (%2.5f%%)",
                  ((float)r.ru_utime.tv_sec+(((float)r.ru_utime.tv_usec)/1000000.0)),
            100.0*((float)r.ru_utime.tv_sec+(((float)r.ru_utime.tv_usec)/1000000.0))/((float)(max2(1,gdata.curtime-gdata.startuptime))),
                  ((float)r.ru_stime.tv_sec+(((float)r.ru_stime.tv_usec)/1000000.0)),
            100.0*((float)r.ru_stime.tv_sec+(((float)r.ru_stime.tv_usec)/1000000.0))/((float)(max2(1,gdata.curtime-gdata.startuptime)))
            );

   u_respond(u,"configured nick: %s, actual nick: %s, realname: %s, modes: %s",
             gdata.config_nick,
             gdata.user_nick,
             gdata.user_realname,
             gdata.user_modes);
   
   switch (gdata.connectionmethod.how)
     {
     case how_direct:
       u_respond(u,"current server: %s %s (direct)",
                 gdata.curserverip,gdata.curserverport);
       break;
     case how_bnc:
       if (gdata.connectionmethod.vhost[0])
         {
           u_respond(u,"current server: %s %s (bnc at %s:%i with %s)",
                     gdata.curserverip,gdata.curserverport,gdata.connectionmethod.host,gdata.connectionmethod.port,gdata.connectionmethod.vhost);
         }
       else
         {
           u_respond(u,"current server: %s %s (bnc at %s:%i)",
                     gdata.curserverip,gdata.curserverport,gdata.connectionmethod.host,gdata.connectionmethod.port);
         }
       break;
     case how_wingate:
       u_respond(u,"current server: %s %s (wingate at %s:%i)",
                 gdata.curserverip,gdata.curserverport,gdata.connectionmethod.host,gdata.connectionmethod.port);
       break;
     case how_custom:
       u_respond(u,"current server: %s %s (custom at %s:%i)",
                 gdata.curserverip,gdata.curserverport,gdata.connectionmethod.host,gdata.connectionmethod.port);
       break;
     }
   
   u_respond(u,"current server actual name: %s ",
             gdata.curserveractualname ? gdata.curserveractualname : "<unknown>");
   
   for (i=0; gdata.channels[i] && i<MAXCHNLS; i++) {
      snprintf(tempstr,maxtextlength-2,"channel %10s: joined: %3s",
               gdata.channels[i]->name,gdata.channels[i]->flags & CHAN_ONCHAN ? "yes" : "no ");
      
      if (strlen(gdata.channels[i]->key)) {
         snprintf(tempstr2,maxtextlength-2,"%s, key: %s",tempstr,
                  gdata.channels[i]->key);
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
               
      if (gdata.channels[i]->plisttime) {
         snprintf(tempstr2,maxtextlength-2,"%s, plist every %2i min (%s)",tempstr,
                  gdata.channels[i]->plisttime,
                  gdata.channels[i]->flags & CHAN_MINIMAL ? "minimal" : (gdata.channels[i]->flags & CHAN_SUMMARY ? "summary" : "full"));
         strncpy(tempstr,tempstr2,maxtextlength-1);
         }
      
      u_respond(u,"%s",tempstr);
      }
   
   u_respond(u, "bandwidth: lowsend: %i, minspeed: %1.1f, maxspeed: %1.1f, overallmaxspeed: %i",
             gdata.lowbdwth,gdata.overallminspeed,gdata.transfermaxspeed,gdata.maxb/4);
   
   if (gdata.overallmaxspeed != gdata.overallmaxspeeddayspeed)
     {
       u_respond(u, "           default max: %i, day max: %i ( %i:00 -> %i:59, days=\"%s%s%s%s%s%s%s\" )",
                 gdata.overallmaxspeed/4,gdata.overallmaxspeeddayspeed/4,
                 gdata.overallmaxspeeddaytimestart,gdata.overallmaxspeeddaytimeend-1,
                 (gdata.overallmaxspeeddaydays & (1 << 1)) ? "M":"",
                 (gdata.overallmaxspeeddaydays & (1 << 2)) ? "T":"",
                 (gdata.overallmaxspeeddaydays & (1 << 3)) ? "W":"",
                 (gdata.overallmaxspeeddaydays & (1 << 4)) ? "R":"",
                 (gdata.overallmaxspeeddaydays & (1 << 5)) ? "F":"",
                 (gdata.overallmaxspeeddaydays & (1 << 6)) ? "S":"",
                 (gdata.overallmaxspeeddaydays & (1 << 0)) ? "U":"");
     }
   else
     {
       u_respond(u, "           default max: %i, day max: (same)",
                 gdata.overallmaxspeed/4);
     }
   
   u_respond(u, "files: xdcc: %s, pid: %s, log: %s, msg: %s, ignl: %s",
             (gdata.xdccfile?gdata.xdccfile:"(none)"),
             (gdata.pidfile?gdata.pidfile:"(none)"),
             (gdata.logfile?gdata.logfile:"(none)"),
             (gdata.messagefile?gdata.messagefile:"(none)"),
             (gdata.ignorefile?gdata.ignorefile:"(none)"));
   
   u_respond(u, "config file%s: %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
             gdata.configfile[1]?"s":"",
             gdata.configfile[0],
             gdata.configfile[1]?", ":"",gdata.configfile[1]?gdata.configfile[1]:"",
             gdata.configfile[2]?", ":"",gdata.configfile[2]?gdata.configfile[2]:"",
             gdata.configfile[3]?", ":"",gdata.configfile[3]?gdata.configfile[3]:"",
             gdata.configfile[4]?", ":"",gdata.configfile[4]?gdata.configfile[4]:"",
             gdata.configfile[5]?", ":"",gdata.configfile[5]?gdata.configfile[5]:"",
             gdata.configfile[6]?", ":"",gdata.configfile[6]?gdata.configfile[6]:"",
             gdata.configfile[7]?", ":"",gdata.configfile[7]?gdata.configfile[7]:"",
             gdata.configfile[8]?", ":"",gdata.configfile[8]?gdata.configfile[8]:"",
             gdata.configfile[9]?", ":"",gdata.configfile[9]?gdata.configfile[9]:""
             );
   
   if (gdata.uploadallowed) {
      snprintf(tempstr2,maxtextlength-2,"%lliMB",(long long)(gdata.uploadmaxsize/1024/1024));
      u_respond(u, "upload allowed, dir: %s, max size: %s",
                gdata.uploaddir,gdata.uploadmaxsize?tempstr2:"none");
      }
   
   if (gdata.stdout_buffer_init)
     {
       u_respond(u,"stdout buffering: %d bytes",
                 gdata.stdout_buffer_in_cnt - gdata.stdout_buffer_out_cnt);
     }
   
   if (gdata.delayedshutdown) {
      u_respond(u,"NOTICE: Delayed shutdown activated, iroffer will shutdown once there are no active transfers");
      u_respond(u,"NOTICE: To cancel the delayed shutdown, issue \"SHUTDOWN CANCEL\"");
      }
   
   mydelete(tempstr);
   mydelete(tempstr2);
   }


static void u_ignl(const userinput * const u)
{
  int i, first;
  time_t left, ago;
  
  updatecontext();
  
  if (!gdata.ignorelist[0])
    {
      u_respond(u,"No Hosts Ignored or Watched");
      return;
    }
  
  first = 1;
  for (i=0; i<MAXIGNL && gdata.ignorelist[i]; i++)
    {
      if (gdata.ignorelist[i]->flags & IGN_IGNORING)
        {
          if (first)
            {
              u_respond(u,"Current Ignore List:");
              u_respond(u,"  Last Request  Un-Ignore    Type  Hostmask");
              first = 0;
            }
          
          ago = gdata.curtime-gdata.ignorelist[i]->lastcontact;
          left = IGN_TL*(gdata.ignorelist[i]->bucket-IGN_OFF+1);
          
          u_respond(u,"  %4i%c%02i%c ago   %4i%c%02i%c  %6s  %-32s",
                    ago < 3600 ? ago/60 : ago/60/60 ,
                    ago < 3600 ? 'm' : 'h',
                    ago < 3600 ? ago%60 : (ago/60)%60 ,
                    ago < 3600 ? 's' : 'm',
                    left < 3600 ? left/60 : left/60/60 ,
                    left < 3600 ? 'm' : 'h',
                    left < 3600 ? left%60 : (left/60)%60 ,
                    left < 3600 ? 's' : 'm',
                    gdata.ignorelist[i]->flags & IGN_MANUAL ? "manual" : "auto",
                    gdata.ignorelist[i]->hostmask);
        }
    }

  first = 1;
  for (i=0; i<MAXIGNL && gdata.ignorelist[i]; i++)
    {
      if (!(gdata.ignorelist[i]->flags & IGN_IGNORING))
        {
          if (first)
            {
              u_respond(u,"Current Watch List:");
              u_respond(u,"  Last Request   Un-Watch          Hostmask");
              first = 0;
            }
          
          ago = gdata.curtime-gdata.ignorelist[i]->lastcontact;
          left = IGN_TL*(gdata.ignorelist[i]->bucket-IGN_OFF+1);
          
          u_respond(u,"  %4i%c%02i%c ago   %4i%c%02i%c          %-32s",
                    ago < 3600 ? ago/60 : ago/60/60 ,
                    ago < 3600 ? 'm' : 'h',
                    ago < 3600 ? ago%60 : (ago/60)%60 ,
                    ago < 3600 ? 's' : 'm',
                    left < 3600 ? left/60 : left/60/60 ,
                    left < 3600 ? 'm' : 'h',
                    left < 3600 ? left%60 : (left/60)%60 ,
                    left < 3600 ? 's' : 'm',
                    gdata.ignorelist[i]->hostmask);
        }
    }

  return;
}

static void u_ignore(const userinput * const u) {
   int i, num=0, found;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   
   if (!u->arg1)  {
      u_respond(u,"Try specifying an amount of time to ignore");        
      return;
      }

   if (!u->arg2 || strlen(u->arg2) < 4) {
      u_respond(u,"Try specifying a hostmask longer than 4 characters");
      return;
      }
   
   for (found=i=0; (i<MAXIGNL && gdata.ignorelist[i] && !found); i++)
      if (gdata.ignorelist[i]->regexp != NULL && !regexec(gdata.ignorelist[i]->regexp,u->arg2,0,NULL,0))
         found++;
   
   if (i == MAXIGNL) {
      outerror(OUTERROR_TYPE_WARN_LOUD,"Out of ignore slots!, This shouldn't happen");
      u_respond(u,"Out of ignore slots!, This shouldn't happen");
      return;
      }
   
   if (!found) {
      char *tempstr = mycalloc(maxtextlength);
      
      gdata.ignorelist[i] = mycalloc(sizeof(igninfo));
      gdata.ignorelist[i]->regexp = mycalloc(sizeof(regex_t));
      
      strncpy(gdata.ignorelist[i]->hostmask,u->arg2,maxtextlength-1);
      
      strncpy(tempstr,u->arg2,maxtextlength-1);
      hostmasktoregex(tempstr);
      if (regcomp(gdata.ignorelist[i]->regexp,tempstr,REG_ICASE|REG_NOSUB))
         gdata.ignorelist[i]->regexp = NULL;
      
      gdata.ignorelist[i]->flags |= IGN_IGNORING;
      gdata.ignorelist[i]->lastcontact = gdata.curtime;
      
      mydelete(tempstr);
      }
   else
      i--;
   gdata.ignorelist[i]->flags |= IGN_MANUAL;
   gdata.ignorelist[i]->bucket = (num*60)/IGN_TL;
   
   u_respond(u, "Ignore activated for %s which will last %i min",
             u->arg2,num);
   write_ignorefile();

   }


static void u_unignore(const userinput * const u)
{
  int i;
  int found;
  
  updatecontext();
  
  if (!u->arg1)
    {
      u_respond(u,"Try specifying a hostmask to un-ignore");
      return;
    }
  
  found=0;
  for (i=0; i<MAXIGNL && gdata.ignorelist[i]; i++)
    {
      if (strncmp(gdata.ignorelist[i]->hostmask,u->arg1,maxtextlength-1) == 0)
        {
          mydelete(gdata.ignorelist[i]->regexp);
          mydelete(gdata.ignorelist[i]);
          
          for (; i<(MAXIGNL-1); i++)
            {
              gdata.ignorelist[i] = gdata.ignorelist[i+1];
            }
          gdata.ignorelist[i] = NULL;
          
          u_respond(u, "Ignore removed for %s",u->arg1);
          write_ignorefile();
          
          found=1;
          break;
        }
    }
  
  if (!found)
    {
      u_respond(u,"Hostmask not found");
    }
  
  return;
}


static void u_nosave(const userinput * const u) {
   int num = 0;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   gdata.noautosave=gdata.curtime + 60*num - 1;
   u_respond(u,"*** XDCC AutoSave has been disabled for the next %i minute%s",num,num!=1?"s":"");
   
   }


static void u_nosend(const userinput * const u) {
   int num = 0;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   gdata.nonewcons=gdata.curtime + 60*num - 1;
   u_respond(u,"*** XDCC Send has been disabled for the next %i minute%s",num,num!=1?"s":"");
   
   }


static void u_nolist(const userinput * const u) {
   int num = 0;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   gdata.nolisting=gdata.curtime + 60*num - 1;
   u_respond(u,"*** XDCC List and PLIST have been disabled for the next %i minute%s",num,num!=1?"s":"");
   
   }


static void u_renumber(const userinput * const u) {
   int oldp = 0, newp = 0;
   int i,op,np;
   xdcc *t;
   
   updatecontext();
   
   if (u->arg1) oldp = atoi(u->arg1);
   if (u->arg2) newp = atoi(u->arg2);
   
   if (oldp < 1 || oldp > gdata.numpacks || newp < 1 || newp > gdata.numpacks || newp == oldp ) {
      u_respond(u,"Invalid pack number");
      return;
      }
   
   /* find old */
   for (i=1, op=0; i<=gdata.numpacks; i++, op++) {
      while (gdata.xdccs[op] == NULL)
         op++;
      if ( i == oldp )
         break;
      }
   
   /* find new */
   for (i=1, np=0; i<=gdata.numpacks; i++, np++) {
      while (gdata.xdccs[np] == NULL)
         np++;
      if ( i == newp )
         break;
      }
   
   t = gdata.xdccs[op];
   if (np > op)
      for (i=op; i<np; i++)
         gdata.xdccs[i] = gdata.xdccs[i+1];
   if (np < op)
      for (i=op; i>np; i--)
         gdata.xdccs[i] = gdata.xdccs[i-1];
   gdata.xdccs[np] = t;
   
   u_respond(u,"*** Moved pack %i to %i",oldp,newp);
   
   xdccsave(0);
   
   }


static void u_msgread(const userinput * const u) {
   int fd,count=0,i;
   char *linedate, *linemask;
   int msgoffset;
   char *tempstr;
   
   updatecontext();
   
   if (!gdata.messagefile) {
      u_respond(u,"no messagefile defined in config");
      return;
      }
   
   fd=open(gdata.messagefile, O_RDONLY | ADDED_OPEN_FLAGS);
   if (fd < 0) {
      u_respond(u,"Cant Access Message File: %s",strerror(errno));
      return;
      }
   
   tempstr = mycalloc(maxtextlength);
   while (getfline(tempstr,fd,0) != NULL) {
      count++;
      linedate = getpart(tempstr,1);
      linemask = getpart(tempstr,2);
      msgoffset = sstrlen(linedate) + sstrlen(linemask) + 1;
      
      for (i=msgoffset; i<sstrlen(tempstr)+1; i++)
         tempstr[i-msgoffset] = tempstr[i];
      
      
      linedate = getdatestr(linedate,atoul(linedate));
      
      u_respond(u, "%s: %s",linedate,linemask);
      u_respond(u, " ^- %s",tempstr);
      
      mydelete(linedate); mydelete(linemask);
      }
   
   u_respond(u, "msglog: %i message%s in log%s%s",
             count,
             count!=1?"s":"",
             count?", use MSGDEL to remove ":"",
             count>1?"them":(count==1?"it":""));
   
   mydelete(tempstr);
   close(fd);
   
   }


static void u_msgdel(const userinput * const u) {
   
   updatecontext();
   
   if (!gdata.messagefile) {
      u_respond(u,"no messagefile defined in config");
      return;
      }
   
   if (truncate(gdata.messagefile,0) < 0) {
      u_respond(u,"couldn't delete messages");
      return;
      }
   
   u_respond(u,"msglog: deleted all messages");
   
   }


static void u_memstat(const userinput * const u) {
   int i;
   long numcountrecent, sizecount;
   struct rusage r;
   
   updatecontext();
   
   u_respond(u,"iroffer memory usage:");

   getrusage(RUSAGE_SELF,&r);
   
   u_respond(u,"rusage: maxrss %li, ixrss %li, idrss %li, isrss %li, minflt %li, majflt %li, nswap %li",
             r.ru_maxrss,r.ru_ixrss,r.ru_idrss,r.ru_isrss,r.ru_minflt,r.ru_majflt,r.ru_nswap);
   u_respond(u,"        inbloc %li, oublock %li, msgsnd %li, msgrcv %li, nsignals %li, nvcsw %li, nivcsw %li",
             r.ru_inblock,r.ru_oublock,r.ru_msgsnd,r.ru_msgrcv,r.ru_nsignals,r.ru_nvcsw,r.ru_nivcsw);
      
   u_respond(u, "gdata: %d bytes", sizeof(gdata_t));
   
   for (numcountrecent=sizecount=i=0; i<(MEMINFOHASHSIZE * gdata.meminfo_depth); i++)
      if (gdata.meminfo[i].ptr != NULL) {
         sizecount += gdata.meminfo[i].size;
         if (gdata.meminfo[i].alloctime > gdata.curtime-600)
            numcountrecent++;
         }
      
   u_respond(u, "%li bytes allocated for %li arrays (%li created in past 10 min) (depth %d)",
             sizecount,
             gdata.meminfo_count,
             numcountrecent,
             gdata.meminfo_depth);
   
   if (u->arg1 && !strcmp(u->arg1,"list"))
     {
       
       u_respond(u,"iroffer memory usage details:");
       u_respond(u,"     id |    address |    size |     when | where");
       
       for (i=0; i<(MEMINFOHASHSIZE * gdata.meminfo_depth); i++)
         {
           if (gdata.meminfo[i].ptr != NULL)
             {
               u_respond(u, "%3i %3i | 0x%8.8lX | %6iB | %7lis | %s:%d %s()",
                         i / gdata.meminfo_depth,
                         i % gdata.meminfo_depth,
                         (long)gdata.meminfo[i].ptr,
                         gdata.meminfo[i].size,
                         gdata.meminfo[i].alloctime-gdata.startuptime,
                         gdata.meminfo[i].src_file,
                         gdata.meminfo[i].src_line,
                         gdata.meminfo[i].src_func);
             }
         }
     }
   else
     {
       u_respond(u,"for a detailed listing use \"memstat list\"");
     }
   
}


static void u_qsend(const userinput * const u)
{
  
  updatecontext();
  
  if (!gdata.inqueue && !gdata.inslotsmaxqueue)
    {
      u_respond(u,"No Users Queued");
      return;
    }
  
  if (gdata.slotsfull >= ACTUAL_MAXTRANS)
    {
      u_respond(u,"Too many transfers");
      return;
    }
  
  sendaqueue(2);
  return;
}

static void u_shutdown(const userinput * const u) {
   updatecontext();
   
   if (!u->arg1 || (strcmp(caps(u->arg1),"NOW") && strcmp(caps(u->arg1),"DELAYED") && strcmp(u->arg1,"CANCEL")) ) {
      u_respond(u,"Usage: SHUTDOWN <now|delayed|cancel>");
      return;
      }
   
   if (!strcmp(u->arg1,"NOW")) {
      shutdowniroffer();
      }
   else if (!strcmp(u->arg1,"DELAYED")) {
      u_respond(u,"Delayed shutdown activated, iroffer will shutdown once there are no active transfers");
      u_respond(u,"To cancel the delayed shutdown, issue \"SHUTDOWN CANCEL\"");
      gdata.delayedshutdown=1;
      }
   else if (!strcmp(u->arg1,"CANCEL")) {
      u_respond(u,"Delayed shutdown canceled");
      gdata.delayedshutdown=0;
      }
   
   }

static void u_debug(const userinput * const u) {
   updatecontext();
   
   if (!u->arg1) return;
   
   gdata.debug = atoi(u->arg1);
   }

static void u_servqc(const userinput * const u)
{
  int i;
  
  updatecontext();
  
  for (i=0; i<MAXSENDQ && gdata.serverq[i]; i++)
    {
      mydelete(gdata.serverq[i]);
    }
  
  u_respond(u,"Cleared server queue of %d lines",i);
  return;
}

static void u_jump(const userinput * const u) {
  int num = 0;
  
   updatecontext();
   
  if (u->arg1) num = atoi(u->arg1);
  if (num < 0) num = 0;
  
  if (num > 0 && (num > MAXSRVS || !gdata.server[num-1]))
    u_respond(u,"Try specifying a valid server number, use \"servers\" for a list");
  else
    switchserver(num-1);
  
}

static void u_servers(const userinput * const u) {
  int i;
  
  updatecontext();
   
  u_respond(u,"Server List:");
  u_respond(u,"  Num  Server/Port");
  
  for (i=0; i<MAXSRVS; i++)
    if (gdata.server[i]) {
      u_respond(u,"   %2i  %s",i+1,gdata.server[i]);
    }
  
  u_respond(u,"Current Server: %s:%s (%s)",
            gdata.curserverip,
            gdata.curserverport,
            gdata.curserveractualname ? gdata.curserveractualname : "<unknown>");
  
}

static void u_trinfo(const userinput * const u) {
   int num = -1;
   char *tempstr2, *tempstr3;
   const char *y;
   int left,started,lcontact;
   
   updatecontext();
   
   if (u->arg1) num = atoi(u->arg1);
   if (num < 0 || num > MAXTRANS || gdata.trans[num] == NULL) {
      u_respond(u,"Try Specifying a Valid Transfer Number");
      return;
      }
   
   tempstr2 = mycalloc(maxtextlength);
   tempstr3 = mycalloc(maxtextlengthshort);
   
   u_respond(u,"Transfer Info for ID %i:",num);
   
   switch (gdata.trans[num]->tr_status)
     {
     case TRANSFER_STATUS_LISTENING:
       y = "Listening";
       break;
       
     case TRANSFER_STATUS_SENDING:
       y = "Sending";
       break;
       
     case TRANSFER_STATUS_WAITING:
       y = "Finishing";
       break;
       
     case TRANSFER_STATUS_DONE:
       y = "Closing";
       break;
       
     default:
       y = "Unknown!";
       break;
     }
         
   u_respond(u,"User %s, Hostname %s, Status %s",
             gdata.trans[num]->nick,gdata.trans[num]->hostname,y);
   
   u_respond(u,"File: %s",getfilename(gdata.trans[num]->xpack->file));
   
   u_respond(u,"Start %lliK, Current %lliK, End %lliK (%2.0f%% File, %2.0f%% Xfer)",
             (long long)((gdata.trans[num]->startresume)/1024),
             (long long)((gdata.trans[num]->bytessent)/1024),
             (long long)((gdata.trans[num]->xpack->st_size)/1024),
             ((float)gdata.trans[num]->bytessent)*100.0/((float)gdata.trans[num]->xpack->st_size),
             ((float)(gdata.trans[num]->bytessent-gdata.trans[num]->startresume))*100.0/((float)max2(1,(gdata.trans[num]->xpack->st_size-gdata.trans[num]->startresume)))
             );
   
   snprintf(tempstr2,maxtextlength-2,"%1.1fK/s",gdata.trans[num]->xpack->minspeed);
   snprintf(tempstr3,maxtextlengthshort-2,"%1.1fK/s",gdata.trans[num]->xpack->maxspeed);
   
   u_respond(u,"Min %s, Current %1.1fK/s, Max %s, In Transit %lliK",
             (gdata.trans[num]->nomin || (gdata.trans[num]->xpack->minspeed == 0.0)) ? "no" : tempstr2 ,
             gdata.trans[num]->lastspeed,
             (gdata.trans[num]->nomax || (gdata.trans[num]->xpack->maxspeed == 0.0)) ? "no" : tempstr3 ,
             (long long)(gdata.trans[num]->bytessent-gdata.trans[num]->lastack)/1024
             );
   
   left     = min2(359999,(gdata.trans[num]->xpack->st_size-gdata.trans[num]->bytessent)/((int)(max2(gdata.trans[num]->lastspeed,0.001)*1024)));
   started  = min2(359999,gdata.curtime-gdata.trans[num]->connecttime);
   lcontact = min2(359999,gdata.curtime-gdata.trans[num]->lastcontact);
   u_respond(u,"Transfer started %i%c %i%c ago, Finish in %i%c %i%c, Last contact %i%c %i%c ago.",
             started < 3600 ? started/60 : started/60/60 ,
             started < 3600 ? 'm' : 'h',
             started < 3600 ? started%60 : (started/60)%60 ,
             started < 3600 ? 's' : 'm',
             left < 3600 ? left/60 : left/60/60 ,
             left < 3600 ? 'm' : 'h',
             left < 3600 ? left%60 : (left/60)%60 ,
             left < 3600 ? 's' : 'm',
             lcontact < 3600 ? lcontact/60 : lcontact/60/60 ,
             lcontact < 3600 ? 'm' : 'h',
             lcontact < 3600 ? lcontact%60 : (lcontact/60)%60 ,
             lcontact < 3600 ? 's' : 'm'
             );
   
   u_respond(u,"Local: %ld.%ld.%ld.%ld:%d, Remote: %ld.%ld.%ld.%ld:%d",
             gdata.trans[num]->localip>>24, (gdata.trans[num]->localip>>16) & 0xFF, (gdata.trans[num]->localip>>8) & 0xFF, gdata.trans[num]->localip & 0xFF, gdata.trans[num]->listenport,
             gdata.trans[num]->remoteip>>24, (gdata.trans[num]->remoteip>>16) & 0xFF, (gdata.trans[num]->remoteip>>8) & 0xFF, gdata.trans[num]->remoteip & 0xFF, gdata.trans[num]->remoteport
             );
   
   u_respond(u,"Sockets: Listen %i, Transfer %i, File %i",
             (gdata.trans[num]->listensocket == FD_UNUSED) ? 0 : gdata.trans[num]->listensocket,
             (gdata.trans[num]->clientsocket == FD_UNUSED) ? 0 : gdata.trans[num]->clientsocket,
             (gdata.trans[num]->filedescriptor == FD_UNUSED) ? 0 : gdata.trans[num]->filedescriptor);
   
   
   mydelete(tempstr2);
   mydelete(tempstr3);
   }


static void u_listul(const userinput * const u) {
   int count=0;
   DIR *d;
   struct dirent *f;
   struct stat s;
   char *tempstr;
   
   updatecontext();
   
   if (!gdata.uploadallowed || !gdata.uploaddir) {
      u_respond(u,"Upload not Allowed or no Uploaddir defined.");
      return;
      }
   
   if (!(d = opendir(gdata.uploaddir))) {
      u_respond(u,"Cant Access Upload Directory: %s",strerror(errno));
      return;
      }
   
   tempstr = mycalloc(maxtextlength);
   
   u_respond(u,"Contents of %s",gdata.uploaddir);
   
   while ((f = readdir(d))) {
      if (strcmp(".",f->d_name) && strcmp("..",f->d_name)) {
         count++;
         snprintf(tempstr,maxtextlength-2,"%s/%s",gdata.uploaddir,f->d_name);
         stat(tempstr,&s);
         u_respond(u,"   %s (%liK)",
                f->d_name,(long)s.st_size/1024);
         }
      }
   
   u_respond(u,"%i Total Files",count);
   
   mydelete(tempstr);
   
   closedir(d);
   
   }

static void u_clearrecords(const userinput * const u) {
   updatecontext();
   
   gdata.record = 0;
   gdata.sentrecord = 0;
   gdata.totalsent = 0;
   gdata.totaluptime = 0;
   
   u_respond(u,"Cleared transfer record, bandwidth record, total sent, and total uptime");
   
   }

static void u_rmul(const userinput * const u) {
   char *tempstr;
   
   updatecontext();
   
   if (!gdata.uploadallowed || !gdata.uploaddir) {
      u_respond(u,"Upload not Allowed or no Uploaddir defined.");
      return;
      }
   
   if (!u->arg1e || !strlen(u->arg1e)) {
      u_respond(u,"Try Specifying a Filename");
      return;
      }
   
   if (strstr(u->arg1e,"/")) {
      u_respond(u,"Filename contains invalid characters");
      return;
      }
   
   tempstr = mycalloc(maxtextlength);
   snprintf(tempstr,maxtextlength-2,"%s/%s",gdata.uploaddir,u->arg1e);
   
   if (doesfileexist(tempstr)) {
      if (unlink(tempstr) < 0)
         u_respond(u,"Unable to remove the file");
      else
         u_respond(u,"Deleted");
      }
   else
      u_respond(u,"That filename doesn't exist");
   
   mydelete(tempstr);
   
   }

static void u_crash(const userinput * const u) {
   
   updatecontext();
   
   *((int*)(0)) = 0;
   
}

#define USERS_PER_CHAN_LINE  6

static void u_chanl(const userinput * const u)
{
  int i,j;
  channel_member_t *p;
  char *tempstr = mycalloc(maxtextlength);
  
  updatecontext();
  
  u_respond(u,"Channel Members:");
  
  for (i=0; i<MAXCHNLS; i++)
    if (gdata.channels[i])
      {
	j = 0;
	p = gdata.channels[i]->members;
	
	if (p)
	  {
	    
	    while ( p )
	      {
		nocaps(p->nick);
		if (!(j%USERS_PER_CHAN_LINE))
		  snprintf(tempstr,maxtextlength-1,"%s: ",gdata.channels[i]->name);
		strncat(tempstr,p->nick,maxtextlength-strlen(tempstr)-1);
		strncat(tempstr," ",maxtextlength-strlen(tempstr)-1);
		if (!((j+1)%USERS_PER_CHAN_LINE))
		  {
		    u_respond(u,"%s",tempstr);
		    tempstr[0] = '\0';
		  }
		caps(p->nick);
		
		p = p->next;
		j++;
	      }
	    
	    if (j%USERS_PER_CHAN_LINE)
	      u_respond(u,"%s",tempstr);

	  }
	
	u_respond(u,"%s: %i user%s",gdata.channels[i]->name,j,j!=1?"s":"");
	
      }

  mydelete(tempstr);
  
}
/* End of File */
