use Irssi;
use strict;
use vars qw($VERSION %IRSSI);

$VERSION = "1.0";
%IRSSI = (
    authors     => 'Toni Salomki',
    name        => 'autoopper',
    contact     => 'Toni@IRCNet',
    description => 'Auto-op script with random delay. Based on the autoop script by Timo "cras" Sirainen',
    license     => 'GNU GPLv2 or later'
);

my (%oplist);
srand();

sub cmd_save_autoop {
	# Load data from file
	my $file = Irssi::get_irssi_dir."/autoop";
	open FILE, "> $file" or return;
	
	foreach my $key (keys %oplist) {
		print FILE "$key\t$oplist{$key}\n";
	}
	close FILE;
	
	Irssi::print("Auto-op list saved to $file");
}

sub cmd_load_autoop {
	# Save data to file
	my $file = Irssi::get_irssi_dir."/autoop";
	open FILE, "< $file" or return;

	while (<FILE>) {
		chomp;
		my ($chan, $mask) = split (/\t/, $_, 2);
		$oplist{$chan} = $mask;
	}
	close FILE;

	Irssi::print("Auto-op list reloaded from $file");
}

sub cmd_show_autoop {
	# Show who's being auto-opped
	my ($channel) = @_;
	my @chans = $channel ? $channel : sort (keys %oplist);

	foreach $channel (@chans) {
		my $masks = $oplist{$channel} ? $oplist{$channel} : "<no-one>";
		if ($channel eq "*") {
			Irssi::print("All channels: $masks");
		} else {
			Irssi::print("$channel: $masks");
		}
	}
}

sub cmd_add_autoop {
	# Add new auto-op
	my ($data) = @_;
	my ($channel, $mask, undef) = split(" ", $data, 3);

	if ($channel eq "" || $mask eq "") {
		Irssi::print("Usage: /autoop_add <*|#channel> <hostmask>");
		return;
	}

	if ($oplist{$channel}) {
		$oplist{$channel} .= " $mask";
	} else {
		$oplist{$channel} = "$mask";
	}
	Irssi::print("Added auto-op: $channel: $mask");
}

sub cmd_del_autoop {
	# Remove autoop
	my ($data) = @_;
	my ($channel, $mask, undef) = split(" ", $data, 3);

	if ($channel eq "" || $mask eq "") {
		Irssi::print("Usage: /autoop_del <*|#channel> <hostmask>");
		return;
	}

	my $newlist;
	foreach my $item (split(" ", $oplist{$channel})) {
		if ($item eq $mask) {
			Irssi::print("Deleted: $channel: $item");
			next;
		}
		if ($newlist) {
			$newlist .= " $item";
		} else {
			$newlist = "$item";
		}
	}
	$oplist{$channel} = $newlist;
    delete $oplist{$channel} if (!$newlist);
}

sub do_autoop {
	# Do the actual opping
	my $target = shift;
	my $nick = $target->{chan}->nick_find($target->{nick});
	Irssi::timeout_remove($target->{tag});
	$target->{chan}->command("/op " . $nick->{nick}) if (!$nick->{op}); 
	undef $target;
}

sub check_autoop {
	# Check if opping should be done
	my ($channel, $masks, $nickrec) = @_;

	if ($channel->{server}->masks_match($masks, $nickrec->{nick}, $nickrec->{host})) {
		my $min_delay = Irssi::settings_get_int("autoop_min_delay");
		my $max_delay = Irssi::settings_get_int("autoop_max_delay") - $min_delay;
		my $delay = int(rand($max_delay)) + $min_delay;

		my $target = {
			nick => $nickrec->{nick},
			chan => $channel,
			tag => undef
		};

		$target->{tag} = Irssi::timeout_add($delay, 'do_autoop', $target);
		return 1;
	}
	return 0;
}

sub event_massjoin {
	# Someone joined, might be multiple persons
	my ($channel, $nicks_list) = @_;
	my @nicks = @{$nicks_list};

	return if (!$channel->{chanop});

	foreach my $nickrec (@nicks) {
		my $masks = $oplist{"*"};
		my $opped = 0;
		check_autoop($channel, $masks, $nickrec) if ($masks);

		if (!$opped) {
			$masks = $oplist{$channel->{name}};
			check_autoop($channel, $masks, $nickrec) if ($masks);
		}
	}
}

cmd_load_autoop;

Irssi::command_bind('autoop_show', 'cmd_show_autoop');
Irssi::command_bind('autoop_add', 'cmd_add_autoop');
Irssi::command_bind('autoop_del', 'cmd_del_autoop');
Irssi::command_bind('autoop_save', 'cmd_save_autoop');
Irssi::command_bind('autoop_load', 'cmd_load_autoop');
Irssi::signal_add_last('massjoin', 'event_massjoin');
Irssi::signal_add_last('setup saved', 'cmd_save_autoop');
Irssi::signal_add_last('setup reread', 'cmd_load_autoop');
Irssi::settings_add_int('autoop', 'autoop_max_delay', 15000);
Irssi::settings_add_int('autoop', 'autoop_min_delay', 1000);
