/*
 * client.cpp - code for client-windows, which are displayed in several instances in the main-window of iTALC
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

 
#include <qpainter.h>
#include <qdir.h>
#include <qpopupmenu.h>
#include <qdatetime.h>
#include <qmessagebox.h>
#include <qfile.h>
#include <qimage.h>
#include <qapplication.h>
#include <qfiledialog.h>
#include <qtooltip.h>
#include <qspinbox.h>
#include <qworkspace.h>


#include "italc.h"
#include "client.h"
#include "rfb_connection.h"
#include "client_manager.h"
#include "text_input_dialog.h"
#include "cmd_input_dialog.h"
#include "msg_input_dialog.h"
#include "paths.h"
#include "embed.h"
#include "qimage_manips.h"
#include "system_environment.h"
#include "screenshot_list.h"
#include "qt3_compat.h"
//#include "client_viewer_widget.h"

#include "client.moc"


#ifdef OLD_ROOT_WAKE_CODE

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>

#include <sys/socket.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/if.h>

#include <features.h>
#include <netpacket/packet.h>
#include <net/ethernet.h>
#include <netdb.h>
#include <netinet/ether.h>

#endif



//#define HARDCORE_DEBUG


const QSize DEFAULT_CLIENT_SIZE( 256, 192 );



const client::clientCommand client::s_commands[CMD_COUNT] =
{

	{ NONE,		&client::doNothing,	client::tr(""),			"",			FALSE	},
	{ RELOAD,	&client::reload,	client::tr(""),			"", 			TRUE	},

	{ START_FULLSCREEN_DEMO,&client::startFullScreenDemo,client::tr("Start fullscreen-demo"),"client_start_fullscreen_demo",	FALSE	},
	{ START_DEMO,	&client::startDemo,	client::tr("Start window-demo"),"client_start_demo",	FALSE	},
	{ STOP_DEMO,	&client::stopDemo,	client::tr("Stop demo"),	"client_stop_demo",	FALSE	},
	{ CLIENT_DEMO,	&client::clientDemo,	client::tr("Let client show demo"),"client_demo",	FALSE	},
	{ WORK_ON_CLIENT,&client::workOnClient,	client::tr("Help person"),	"support",	FALSE	},
	{ SEND_MESSAGE,	&client::sendMessage,	client::tr("Send message"),	"client_msg",		TRUE	},
	{ DISTRIBUTE_FILE,&client::distributeFile,client::tr("Distribute file"),"distribute_file",	FALSE	},
	{ COLLECT_FILES,&client::collectFiles,	client::tr("Collect files"),	"collect_files",	TRUE	},
	
	{ LOCK_X,	&client::lockX,		client::tr("Lock screen"),	"client_lock_x",	FALSE	},
	{ UNLOCK_X,	&client::unlockX,	client::tr("Unlock screen"),	"client_unlock_x",	FALSE	},
#ifdef COMPILE_LINUX
	{ RESTART_X,	&client::restartX,	client::tr("Restart X"),	"client_restart_x",	TRUE	},
#endif
	{ KILL_GAMES,	&client::killGames,	client::tr("Kill games"),	"client_kill_games",	FALSE	},
	{ KILL_BROWSERS,&client::killBrowsers,	client::tr("Kill browsers"),	"client_kill_browsers", TRUE },
	{ SCREENSHOT,	&client::screenShot,	client::tr("Make screenshot"),	"client_screenshot", TRUE },

	{ EXEC_CMDS,	&client::execCmds,	client::tr("Execute commands/SSH"),"client_exec_cmds", 	FALSE	},
	{ EXEC_CMDS_IRFB,&client::execCmdsIRFB,	client::tr("Execute commands/IRFB"), "client_exec_cmds", FALSE },
#ifdef COMPILE_LINUX
	{ RUN_X_APP,	&client::runXApp,	client::tr("Run X-application"),"client_run_x_app",	FALSE	},
#endif
	{ SSH_LOGIN,	&client::SSHLogin,	client::tr("Login with SSH"),	"client_ssh_login",	TRUE	},
	{ POWER_ON,	&client::powerOn,	client::tr("Power on"),		"client_power_on",	FALSE	},
	{ REBOOT,	&client::reboot,	client::tr("Reboot"),		"client_reboot",	FALSE	},
	{ POWER_OFF,	&client::powerOff,	client::tr("Power off"),	"client_power_off",	FALSE	}

} ;


// resolve static symbols...
QMap<int, client *> client::s_idMap;

QImage * client::s_computerPowerOffImg = NULL;
QImage * client::s_noUserLoggedInImg = NULL;
QImage * client::s_demoRunningImg = NULL;

bool client::s_reloadScreenshotList = FALSE;



client::client( const QString & _ip, const QString & _mac, const QString & _name, classRoom * _class_room, int _id ) :
	QWidget( italc::inst()->workspace(), _name, /*Qt::WNoAutoErase | */Qt::WStyle_Title ),
	m_ip( _ip ),
	m_mac( _mac ),
	m_name( _name ),
	m_user( "" ),
	m_demoRunning( FALSE ),
	m_msgImg(),
	m_makeScreenshot( FALSE ),
	m_syncMutex(),
	m_msgImgMutex()
{
	if( _id <= 0 )
	{
		_id = freeID();
	}
	s_idMap[_id] = this;

	if (s_computerPowerOffImg == NULL)
	{
		s_computerPowerOffImg = new QImage( embed::getIconPixmap( "computer_power_off" ).convertToImage() );
	}
	if( s_noUserLoggedInImg == NULL )
	{
		s_noUserLoggedInImg = new QImage( embed::getIconPixmap( "no_user_logged_in" ).convertToImage() );
	}
	if( s_demoRunningImg == NULL )
	{
		s_demoRunningImg = new QImage( embed::getIconPixmap( "demo_running" ).convertToImage() );
	}

	m_classRoomItem = new classRoomItem( this, _class_room, m_name );


	m_connection = new rfbConnection( m_ip );

	setBackgroundMode( Qt::NoBackground );
	setIcon( embed::getIconPixmap( "client_observed" ) );

	QWhatsThis::add( this, tr( "This is a client-window. It either displays the screen of the according client or a message "
					"about the state of this client (no user logged in/powered off) is shown. You can click with "
					"the right mouse-button and an action-menu for this client will appear. You can also close this "
					"client-window. To open it again, open the client-manager-workspace and search this client and "
					"double-click it.\nYou can change the size of this (and all other visible) client-windows by "
					"using the functions for increasing, decreasing or optimizing the client-window-size." ) );

	setFixedSize( DEFAULT_CLIENT_SIZE/*clientManager::inst()->standardSize()*/ );
}




client::~client()
{
	delete m_connection;
	m_connection = NULL;

	delete m_classRoomItem;
}




void client::resetConnection( void )
{
	m_syncMutex.lock();
	m_connection->resetConnection();
	m_syncMutex.unlock();
}




bool client::userLoggedIn( void )
{
	QMutexLocker ml( &m_syncMutex );

	return( m_connection->connected() || m_connection->resetConnection( m_ip )/* || (int) system("if [ -z \"`ssh root@"+m_ip+" ps aux | grep kdeinit | grep -v grep`\" ] ; then\nexit 0\nelse\nexit 1\nfi") != 0*/ );
}




void client::processCmd( clientCmds _cmd, const QString & _u_data )
{
	if( _cmd < 0 || _cmd >= CMD_COUNT )
	{
		return;
	}
#ifdef HARDCORE_DEBUG
	printf( "client<%s,%s>::processCmd(...) -> %d\n", m_ip.ascii(), m_name.ascii(), (int) _cmd );

#endif

	( this->*( client::s_commands[_cmd].m_exec ) )( _u_data );

#ifdef HARDCORE_DEBUG
	printf( "client::processCmd(...) done\n" );
#endif

}




void client::update( void )
{
	if( m_user == "" )
	{
		m_user = "none";
	}
	setCaption( m_user + "@" + fullName() );
	QWidget::update();
}




void client::reload( const QString & _update )
{
#ifdef HARDCORE_DEBUG
	static int no = 0;
	++no;
	printf ("Reloading client %s (%d) -> ", m_ip.ascii(), no);
#endif
	bool available = FALSE;

	m_syncMutex.lock();


	if( m_connection->connected() || m_connection->resetConnection( m_ip ) == TRUE	)
	{
#ifdef HARDCORE_DEBUG
		printf( "Connected -> Sending getUserRequest...\n" );
#endif
		if( m_connection->sendGetUserRequest() )
		{
#ifdef HARDCORE_DEBUG
			printf ("sucessful -> ");
#endif
			// we only send a framebuffer-update-request if no demo is running...
			m_connection->handleServerMessages( m_demoRunning == FALSE );
#ifdef HARDCORE_DEBUG
			printf( "handling done\n" );
#endif
		}
		else
		{
#ifdef HARDCORE_DEBUG
			printf( "not sucessful -> " );
#endif
		}
		m_user = m_connection->user();
#ifdef HARDCORE_DEBUG
		printf( "user: %s\n", m_user.ascii() );
#endif
		if( m_user != "" && m_demoRunning == FALSE )
		{
			available = TRUE;

			QString real_name = systemEnvironment::realUserName( m_user );

			if( QToolTip::textFor( this ) != real_name )
			{
				QToolTip::remove( this );
				QToolTip::add( this, real_name );
			}
		}
		else if( m_demoRunning == FALSE )
		{
			QImage temporary_image = QImage( size(), 32 );
			// now scale the image and store the output to temporary_image
			fastScaleImage( *s_noUserLoggedInImg, temporary_image );
			m_msgImgMutex.lock();
			// now copy temporary_image to msg-img
			m_msgImg = temporary_image;
			m_msgImgMutex.unlock();
		}
		else
		{
			QImage temporary_image = QImage( size(), 32 );
			// now scale the image and store the output to temporary_image
			fastScaleImage( *s_demoRunningImg, temporary_image );
			m_msgImgMutex.lock();
			// now copy temporary_image to msg-img
			m_msgImg = temporary_image;
			m_msgImgMutex.unlock();
		}
	}
	else
	{
		// we got no m_connection to client... either the computer is not powered on or there's no m_user logged in...
		// so check whether client is reachable
		const QImage * i = NULL;
#ifdef HARDCORE_DEBUG
		printf( "Not connected -> Sending PING...\n" );
#endif
		if( systemEnvironment::hostAvailable( m_ip ) == FALSE )
		{
#ifdef HARDCORE_DEBUG
			printf( "failed -> computer powered off\n" );
#endif
			i = s_computerPowerOffImg;
		}
		else
		{
#ifdef HARDCORE_DEBUG
			printf( "sucessful -> no m_user logged in\n" );
#endif
			// no problems with ping, so ivs is not running -> no m_user logged in
			i = s_noUserLoggedInImg;
		}
		if( i != NULL )
		{
			m_msgImgMutex.lock();
			//qApp->lock();
			if( i->width() > width() || i->height() > height() )
			{
				QImage temporary_image = QImage( size(), 32 );
#ifdef HARDCORE_DEBUG
				printf( "Scaling msg\n" );
#endif
				// now scale the image and store the output to temporary_image
				fastScaleImage( *i, temporary_image );
#ifdef HARDCORE_DEBUG
				printf( "Scaling msg done\n" );
#endif
				// now copy temporary_image to msg-img
				m_msgImg = temporary_image.copy();
			}
			else
			{
				m_msgImg = *i;
			}
			m_msgImgMutex.unlock();
			//qApp->unlock();
		}
	}

	if( available == FALSE )
	{
		QToolTip::remove( this );
	}

	if( m_makeScreenshot == TRUE )
	{
		m_makeScreenshot = FALSE;

		if( m_user != "" && m_connection->connected() )
		{
			// construct text
			QString txt = m_user + "@" + m_name + " (" + m_ip + ") " + QDate( QDate::currentDate() ).toString( Qt::ISODate ) +
					" " + QTime( QTime::currentTime() ).toString( Qt::ISODate );
			QString path = QDir::home().path()+SCREENSHOT_PATH;
			if( QDir( path ).exists() == FALSE )
			{
				QDir::home().mkdir( SCREENSHOT_PATH, FALSE );
			}
			// construct filename
			QString file_name =  path + m_user + "_" + m_ip + "_" + QDate( QDate::currentDate() ).toString( Qt::ISODate ) +
						"_" + QTime( QTime::currentTime() ).toString( Qt::ISODate ) + ".png";
	
			const int FONT_SIZE = 20;
			const int RECT_MARGIN = 10;
			const int RECT_INNER_MARGIN = 5;

			qApp->lock();
			QPixmap screen( m_connection->unscaledScreen() );

			QPixmap italc_icon( embed::getIconPixmap( "client_observed" ) );

			QPainter p( &screen );
			QFont fnt = p.font();
			fnt.setPointSize( FONT_SIZE );
			fnt.setBold( TRUE );
			p.setFont( fnt );
			QFontMetrics fm( p.font() );

			const int rx = RECT_MARGIN;
			const int ry = screen.height() - RECT_MARGIN - 2*RECT_INNER_MARGIN - FONT_SIZE;
			const int rw = RECT_MARGIN + 4*RECT_INNER_MARGIN + fm.size( SingleLine, txt ).width() + italc_icon.width();
			const int rh = 2*RECT_INNER_MARGIN + FONT_SIZE;
			const int ix = rx + RECT_INNER_MARGIN;
			const int iy = ry + RECT_INNER_MARGIN;
			const int tx = ix + italc_icon.width() + 2*RECT_INNER_MARGIN;
			const int ty = ry + RECT_INNER_MARGIN + FONT_SIZE - 2;

			p.fillRect( rx, ry, rw, rh, QBrush( QColor( 0xFF,0xFF,0xFF ), Dense3Pattern ) );
			p.drawPixmap( ix, iy, italc_icon );
			p.drawText( tx, ty, txt );

			screen.save( file_name, "PNG", 50 );

			s_reloadScreenshotList = TRUE;

			qApp->unlock();
		}
	}


	m_syncMutex.unlock();

	// update icon in client-manager which indicates current state
	if( isShown() )
	{
		m_classRoomItem->setObserved( TRUE );
	}
	else
	{
		m_classRoomItem->setObserved( FALSE );
	}


	// if we are called out of draw-thread, we may update...
	if( _update == CONFIRM_YES )
	{
		update();
	}

#ifdef HARDCORE_DEBUG
	printf( "Reload done\n" );
#endif


}




void client::startFullScreenDemo( const QString & )
{
	stopDemo( "" );

	m_syncMutex.lock();

	if( /*m_demoRunning == FALSE && */( m_connection->connected() || m_connection->resetConnection( m_ip ) == TRUE ) )
	{
		systemEnvironment::demoServer::allowClient( m_ip );
		m_connection->startDemo( MASTER_HOST+"::"+DEMO_PORT, TRUE );
		m_demoRunning = TRUE;
	}

	m_syncMutex.unlock();
}




void client::startDemo( const QString & )
{
	stopDemo( "" );

	m_syncMutex.lock();

	if( /*m_demoRunning == FALSE &&*/ ( m_connection->connected() || m_connection->resetConnection( m_ip ) == TRUE ) )
	{
		systemEnvironment::demoServer::allowClient( m_ip );
		m_connection->startDemo (MASTER_HOST+"::"+DEMO_PORT);
		m_demoRunning = TRUE;
	}

	m_syncMutex.unlock();
}




void client::stopDemo( const QString & )
{
	m_syncMutex.lock();

	if( m_connection->connected() || m_connection->resetConnection(m_ip) == TRUE )
	{
		systemEnvironment::demoServer::denyClient( m_ip );
		m_connection->stopDemo();
	}

	m_demoRunning = FALSE;

	m_syncMutex.unlock();
}




void client::clientDemo( const QString & )
{
	if( !clientManager::inst()->demoRunning() )
	{
		QMessageBox::information( this, tr( "No demo running" ), tr( "Before you allow a client to show a demo, you have to "
						"start the demo-mode by clicking on the corresponding icon in the toolbar." ) );
		return;
	}

	stopDemo( "" );
	workOnClient( "" );
}




void client::workOnClient( const QString & )
{
	if( m_user != "" && m_user != "none" )
	{
		systemEnvironment::startClientViewer( m_ip, "\"" + systemEnvironment::realUserName( m_user ) + " (" + m_user + ")\"" );
	}
//	new clientViewerWidget( m_connection );
}




void client::sendMessage (const QString & _msg) {

	if (_msg == "") {

		QString msg;

		msgInputDialog * msg_input_dialog = new msgInputDialog (msg, this);
		if (msg_input_dialog->exec() == QDialog::Accepted && msg != "")
			sendMessage (msg);

		delete msg_input_dialog;

	} else {

		m_syncMutex.lock ();

		m_connection->sendMessage (_msg);

		m_syncMutex.unlock ();
	}

}




void client::distributeFile( const QString & _file )
{
/*	QMessageBox::information( this, tr( "Function not implemented yet." ), tr( "This function is not completely implemented yet. This is why it is disabled at the moment." ), QMessageBox::Ok );
	return;*/

	if( _file == "" )
	{
		QFileDialog ofd( QDir::home().path(), QString::null, this, "", TRUE );
		ofd.setMode( QFileDialog::ExistingFile );
		ofd.setCaption( tr( "Select file to distribute" ) );
		if( ofd.exec() == QDialog::Accepted )
		{
			distributeFile( ofd.selectedFile() );
		}
	}
	else
	{
		systemEnvironment::distributeFile( _file, this );
		//m_connection->postFile (_file);
		m_syncMutex.unlock();
	}
}




void client::collectFiles( const QString & _files )
{
/*	QMessageBox::information( this, tr( "Function not implemented yet."), tr("This function is not completely implemented yet. This is why it is disabled at the moment."), QMessageBox::Ok);
	return;*/

	if( _files == "" )
	{
		QString fn;
		textInputDialog file_name_input( tr( "Please enter the name of the file to be collected.\nOnly files located in the "
							"PUBLIC-directory are allowed." ), fn, this );
		file_name_input.setCaption( tr( "Collect files" ) );

		if( file_name_input.exec() == QDialog::Accepted && fn != "" )
			collectFiles( fn );
	}
	else
	{
		//m_syncMutex.lock ();
		//m_connection->getFile (_file);
		//m_syncMutex.unlock ();
		if( m_user != "" )
		{
			systemEnvironment::collectFiles( _files, m_user );
		}
	}
}




void client::lockX( const QString & _confirm )
{
/*	if (_confirm == CONFIRM_YES || _confirm == "") {
		if (QMessageBox::question(this, tr("Lock screen"), tr("Are you sure want to lock the screen on selected client?"), QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
			return;
	}*/

	m_syncMutex.lock();

	m_connection->lockDisplay();

	m_syncMutex.unlock();
}




void client::unlockX( const QString & _confirm )
{
/*	if (_confirm == CONFIRM_YES || _confirm == "") {
		if (QMessageBox::question(this, tr("Unlock screen"), tr("Are you sure want to unlock the screen on selected client?"), QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
			return;
	}*/
	m_syncMutex.lock();

	m_connection->unlockDisplay();

	m_syncMutex.unlock();
}



#ifdef COMPILE_LINUX

void client::restartX( const QString & _confirm )
{
	if( userLoggedIn() )
	{
		if( QMessageBox::warning( this, tr( "User logged in" ), tr( "Warning: you're trying to restart X on a client where a "
							"user is logged in! Continue anyway?" ), QMessageBox::Yes, QMessageBox::No )
			== QMessageBox::No )
		{
			return;
		}
	}
	else if( _confirm == CONFIRM_YES || _confirm == "" )
	{
		if( QMessageBox::question( this, tr( "Restart X" ), tr( "Are you sure want to restart X on selected client?" ),
						QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
		{
			return;
		}
	}

	m_syncMutex.lock();

	execCmds( "killall X" );

	m_syncMutex.unlock();
}

#endif



void client::killGames( const QString & _confirm )
{

/*	if (_confirm == CONFIRM_YES || _confirm == "") {
		if (QMessageBox::question(this, tr("Kill games"), tr("Are you sure want to kill all games on selected client?"), QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
			return;
	}*/
	execCmdsIRFB(
		"killall kasteroids\n"
		"killall kfouleggs\n"
		"killall kgoldrunner\n"
		"killall ksirtet\n"
		"killall ksmiletris\n"
		"killall ksnake\n"
		"killall kspaceduel\n"
		"killall ktron\n"
		"killall klickety\n"
		"killall kbackgammon\n"
		"killall kblackbox\n"
		"killall kmahjongg\n"
		"killall kreversi\n"
		"killall kenolaba\n"
		"killall knights\n"
		"killall kbattleship\n"
		"killall kshisen\n"
		"killall kwin4\n"
		"killall kpoker\n"
		"killall lskat\n"
		"killall kpat\n"
		"killall katomic\n"
		"killall kbounce\n"
		"killall kjumpingcube\n"
		"killall kmines\n"
		"killall ksokoban\n"
		"killall klines\n"
		"killall konquest\n"
		"killall pingus\n"
		"killall ksame\n"
		"killall kolf\n"
		"killall ktuberling\n"
		"killall tuxracer\n"
		"killall frozen-bubble\n"
		"killall cube_unix\n"
	);
}




void client::killBrowsers( const QString & _confirm )
{
/*	if (_confirm == CONFIRM_YES || _confirm == "") {
		if (QMessageBox::question(this, tr("Kill Mozilla/FireFox"), tr("Are you sure want to kill Mozilla/FireFox on selected client?"), QMessageBox::Yes, QMessageBox::No) == QMessageBox::No)
			return;
	}*/

	execCmdsIRFB(
			"killall firefox-bin\n"
			"killall firefox\n"
			"killall mozilla-bin\n"
			"killall mozilla\n"
			"killall opera\n"
			// killing Konqueror is a tricky thing, because it's forked by kdeinit so just "killall konqueror" doesn't work
			"ps -A -o pid,cmd | grep konqueror | grep -v 'grep konqueror' | while read i ; do kill `echo $i | cut -d ' ' -f 1` ; done\n"
			// we'll also annoy the freaks ;-)
			"killall lynx\n"
			// ...add more here!
		);
}




void client::screenShot( const QString & )
{
	if( m_user == "" || !m_connection->connected() )
	{
		return;
	}

	m_makeScreenshot = TRUE;
}




void client::execCmds( const QString & _cmds )
{
	if( _cmds == "" )
	{
		QString cmds;

		cmdInputDialog cmd_input_dialog( cmds, this );
		if( cmd_input_dialog.exec() == QDialog::Accepted && cmds != "" )
			execCmds( cmds );
	}
	else
	{
		systemEnvironment::execBg( "ssh root@" + m_ip + " << EOS\n" + _cmds + "\nexit\nEOS" );
	}
}




void client::execCmdsIRFB( const QString & _cmds )
{
	if( _cmds == "" )
	{
		QString cmds;

		cmdInputDialog cmd_input_dialog( cmds, this );
		if( cmd_input_dialog.exec() == QDialog::Accepted && cmds != "" )
			execCmdsIRFB( cmds );
	}
	else
	{
		m_syncMutex.lock();
		m_connection->execCmds( _cmds );
		m_syncMutex.unlock();
	}
}




#ifdef COMPILE_LINUX

void client::runXApp( const QString & _app )
{
	if( _app == "" )
	{
		QString app_to_run;

		textInputDialog app_input( tr( "Please enter the name of the application you want to run and redirect." ),
						app_to_run, this );
		app_input.setCaption( tr( "Run application..." ) );

		if( app_input.exec() == QDialog::Accepted && app_to_run != "" )
			runXApp( app_to_run );
	}
	else
	{
		// no systemEnvironment-abstraction needed, because this function is only compiled under Linux
		system( "xhost +"+m_ip );
		execCmds( "export DISPLAY="+MASTER_HOST+":0.0\n"+_app+"\n" );
	}
}

#endif



void client::SSHLogin( const QString & _user )
{

	if( _user == "" )
	{
		QString client_user = "root";

		textInputDialog user_input( tr( "Please enter the user-name you want to login with." ), client_user, this );
		user_input.setCaption( tr( "SSH-Login" ) );
		if( user_input.exec() == QDialog::Accepted && client_user != "" )
			SSHLogin( client_user );
	}
	else
	{
		systemEnvironment::execInTerminal( "ssh " + _user + "@" + m_ip );
	}
}





void client::powerOn( const QString & )
{
#ifndef OLD_ROOT_WAKE_CODE
	// TODO: replace this code with real c-code 
	// construct some python-code...
	QString python_code;
	python_code += "import struct, socket\n";
	python_code += "def WakeOnLan(ethernet_address):\n";
	python_code += "  addr_byte = ethernet_address.split(':')\n";
	python_code += "  hw_addr = struct.pack('BBBBBB', int(addr_byte[0], 16),\n";
	python_code += "    int(addr_byte[1], 16),\n";
	python_code += "    int(addr_byte[2], 16),\n";
	python_code += "    int(addr_byte[3], 16),\n";
	python_code += "    int(addr_byte[4], 16),\n";
	python_code += "    int(addr_byte[5], 16))\n";
	python_code += "  msg = '\\xff' * 6 + hw_addr * 16\n";
	python_code += "  s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)\n";
	python_code += "  s.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)\n";
	python_code += "  s.sendto(msg, ('<broadcast>', 9))\n";
	python_code += "  s.close()\n";

	python_code += "WakeOnLan('"+m_mac+"')\n";
	systemEnvironment::executePythonCode( python_code );

#else

	const char * ifname = "eth0";

	Q_UINT8 outpack[1000];

	int opt_no_src_addr = 0, opt_broadcast = 0;

#if defined(PF_PACKET)
	struct sockaddr_ll whereto;
#else
	struct sockaddr whereto;	/* who to wake up */
#endif
	struct ether_addr eaddr;


	const int wol_passwd_size = 6;
	Q_UINT8 wol_passwd[wol_passwd_size];


	if (sscanf(m_mac.ascii(), "%2x:%2x:%2x:%2x:%2x:%2x", (unsigned int *) &wol_passwd[0], (unsigned int *) &wol_passwd[1], (unsigned int *) &wol_passwd[2], (unsigned int *) &wol_passwd[3], (unsigned int *) &wol_passwd[4], (unsigned int *) &wol_passwd[5]) != wol_passwd_size) {
		printf ("Invalid MAC-adress\n");
		return;
	}

	//printf(" The Magic packet password is %2.2x %2.2x %2.2x %2.2x (%d).\n", passwd[0], passwd[1], passwd[2], passwd[3], byte_cnt);


	/* Note: PF_INET, SOCK_DGRAM, IPPROTO_UDP would allow SIOCGIFHWADDR to
	   work as non-root, but we need SOCK_PACKET to specify the Ethernet
	   destination address. */

#if defined(PF_PACKET)
	int s = socket(PF_PACKET, SOCK_RAW, 0);
#else
	int s = socket(AF_INET, SOCK_PACKET, SOCK_PACKET);
#endif
	if (s < 0) {
		if (errno == EPERM)
			fprintf(stderr, "ether-wake: This program must be run as root.\n");
		else
			perror("ether-wake: socket");
		return;
	}
	/* Don't revert if debugging allows a normal m_user to get the raw socket. */
//	setuid(getuid());

	/* We look up the station address before reporting failure so that
	   errors may be reported even when run as a normal m_user.
	*/
	struct ether_addr * eap = ether_aton(m_mac.ascii());
	if (eap) {
		eaddr = *eap;
		printf ("The target station address is %s.\n", ether_ntoa(eap));
/*	} else if (ether_hostton(m_mac.ascii(), &eaddr) == 0) {
		fprintf(stderr, "Station address for hostname %s is %s.\n",
					m_mac.ascii(), ether_ntoa(&eaddr));*/
	} else {
		printf ("Invalid MAC-adress\n");
		return;
	}


	if (opt_broadcast)
		memset (outpack+0, 0xff, 6);
	else
		memcpy (outpack, eaddr.ether_addr_octet, 6);

	memcpy (outpack+6, eaddr.ether_addr_octet, 6);
	outpack[12] = 0x08;				/* Or 0x0806 for ARP, 0x8035 for RARP */
	outpack[13] = 0x42;
	int pktsize = 14;

	memset (outpack+pktsize, 0xff, 6);
	pktsize += 6;

	for (int i = 0; i < 16; i++) {
		memcpy (outpack+pktsize, eaddr.ether_addr_octet, 6);
		pktsize += 6;
	}
//	fprintf(stderr, "Packet is ");
//	for (i = 0; i < pktsize; i++)
//		fprintf(stderr, " %2.2x", outpack[i]);
//	fprintf(stderr, ".\n");

	/* Fill in the source address, if possible.
	   The code to retrieve the local station address is Linux specific. */
	if (!opt_no_src_addr) {
		struct ifreq if_hwaddr;
		//unsigned char * hwaddr = if_hwaddr.ifr_hwaddr.sa_data;

		strcpy (if_hwaddr.ifr_name, ifname);
		if (ioctl(s, SIOCGIFHWADDR, &if_hwaddr) < 0) {
			fprintf (stderr, "SIOCGIFHWADDR on %s failed: %s\n", ifname, strerror(errno));
			/* Magic packets still work if our source address is bogus, but
			   we fail just to be anal. */
			return;
		}
		memcpy (outpack+6, if_hwaddr.ifr_hwaddr.sa_data, 6);

		//printf("The hardware address (SIOCGIFHWADDR) of %s is type %d  %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x.\n", ifname, if_hwaddr.ifr_hwaddr.sa_family, hwaddr[0], hwaddr[1], hwaddr[2], hwaddr[3], hwaddr[4], hwaddr[5]);
	}

	memcpy (outpack+pktsize, wol_passwd, wol_passwd_size);
	pktsize += wol_passwd_size;



#if defined(PF_PACKET)
	struct ifreq ifr;
	strncpy (ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
	if (ioctl(s, SIOCGIFINDEX, &ifr) == -1) {
		fprintf(stderr, "SIOCGIFINDEX on %s failed: %s\n", ifname, strerror(errno));
		return;
	}
	memset (&whereto, 0, sizeof(whereto));
	whereto.sll_family = AF_PACKET;
	whereto.sll_ifindex = ifr.ifr_ifindex;
	/* The manual page incorrectly claims the address must be filled.
	   We do so because the code may change to match the docs. */
	whereto.sll_halen = ETH_ALEN;
	memcpy (whereto.sll_addr, outpack, ETH_ALEN);

#else
	whereto.sa_family = 0;
	strcpy(whereto.sa_data, ifname);
#endif

	if (sendto(s, outpack, pktsize, 0, (struct sockaddr *)&whereto, sizeof(whereto)) < 0)
		perror ("sendto");

#ifdef USE_SEND
	if (bind(s, (struct sockaddr *)&whereto, sizeof(whereto)) < 0)
		perror ("bind");
	else if (send(s, outpack, 100, 0) < 0)
		perror ("send");
#endif
#ifdef USE_SENDMSG
	{
		struct msghdr msghdr = { 0,};
		struct iovec iovector[1];
		msghdr.msg_name = &whereto;
		msghdr.msg_namelen = sizeof(whereto);
		msghdr.msg_iov = iovector;
		msghdr.msg_iovlen = 1;
		iovector[0].iov_base = outpack;
		iovector[0].iov_len = pktsize;
		if ((i = sendmsg(s, &msghdr, 0)) < 0)
			perror("sendmsg");
		else if (debug)
			printf("sendmsg worked, %d (%d).\n", i, errno);
	}
#endif

#endif

}




void client::reboot( const QString & _confirm )
{
	if( userLoggedIn() )
	{
		if( QMessageBox::warning( this, tr( "User logged in" ), tr( "Warning: you're trying to reboot a client at which a user "
						"is logged in! Continue anyway?" ), QMessageBox::Yes, QMessageBox::No )
			== QMessageBox::No )
		{
			return;
		}
	}
	else if( _confirm == CONFIRM_YES || _confirm == "" )
	{
		if( QMessageBox::question( this, tr( "Reboot client" ), tr( "Are you sure want to reboot selected client?" ),
			QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
		{
			return;
		}
	}
	execCmds( "reboot" );
}





void client::powerOff( const QString & _confirm )
{
	if( userLoggedIn() )
	{
		if( QMessageBox::warning( this, tr( "User logged in" ), tr( "Warning: you're trying to power off a client at which a "
						"user is logged in! Continue anyway?" ), QMessageBox::Yes, QMessageBox::No )
			== QMessageBox::No )
		{
			return;
		}
	}
	else if( _confirm == CONFIRM_YES || _confirm == "" )
	{
		if( QMessageBox::question( this, tr( "Power off client" ), tr( "Are you sure want to power off selected client?" ),
			QMessageBox::Yes, QMessageBox::No ) == QMessageBox::No )
		{
			return;
		}
	}
	execCmds( "poweroff" );
//	execCmds( "halt" );
}





void client::processCmdSlot( int _cmd )
{
	processCmd( static_cast<clientCmds>( _cmd ) );
}




void client::createActionMenu( QPopupMenu * _m )
{
	QPopupMenu * orig_root = _m;
	QPopupMenu * admin_submenu = new QPopupMenu( _m );

	for( int i = START_FULLSCREEN_DEMO; i < CMD_COUNT; ++i )
	{
		if( i >= EXEC_CMDS )
		{
			_m = admin_submenu;
		}

		int id = _m->insertItem( embed::getIconPixmap( s_commands[i].m_icon ), tr(s_commands[i].m_name ), this,
								SLOT( processCmdSlot( int ) ) );
		_m->setItemParameter( id, i );

		if( s_commands[i].m_insertSep == TRUE )
		{
			_m->insertSeparator();
		}
	}

	orig_root->insertItem( embed::getIconPixmap( "client_settings" ), tr( "Administation" ), admin_submenu );
}




void client::setClassRoom( classRoom * _cr )
{
	delete m_classRoomItem;

	m_classRoomItem = new classRoomItem( this, _cr, m_name );
}




void client::contextMenuEvent( QContextMenuEvent * )
{
	QPopupMenu context_menu( this );

	createActionMenu( &context_menu );

	context_menu.exec( QCursor::pos() );
}




void client::resizeEvent( QResizeEvent * )
{
	m_connection->setImageSize( size() );
	// gives pseudo-transparency-effect, especially with Dense4Pattern -> unusable, because slow as hell!!
/*	QBitmap b( size(), TRUE );
	QPainter p( &b );
	p.fillRect( rect(), QColor( 255, 255, 255 ) );
	p.fillRect( geometry(), QBrush( QColor( 0, 0, 0 ), Qt::CrossPattern ) );
	p.end();
	setMask( b );
	QBitmap b2( parentWidget()->size(), TRUE );
	p.begin( &b2 );
	p.fillRect( parentWidget()->rect(), QColor( 255, 255, 255 ) );
	p.fillRect( geometry(), QColor( 255, 255, 255 ) );
	p.fillRect( geometry(), QBrush( QColor( 0, 0, 0 ), Qt::CrossPattern ) );
	p.end();
	parentWidget()->setMask( b2 );*/
}





void client::paintEvent( QPaintEvent * )
{
	QPixmap draw_pm( rect().size() );
	draw_pm.fill( QColor( 255, 255, 255 ) );

	QPainter p( &draw_pm, this );

	if( m_connection->connected() && m_demoRunning == FALSE )
	{
		p.drawImage( 0, 0, m_connection->clientScreen() );
	}
	else
	{
		if( m_msgImg.isNull() == FALSE )
		{
			m_msgImgMutex.lock();
			p.drawImage( ( width()-m_msgImg.width() ) / 2, ( height()-m_msgImg.height() ) / 2, m_msgImg );
			m_msgImgMutex.unlock();
		}
	}
	// and blit all drawn stuff on the screen...
	bitBlt( this, rect().topLeft(), &draw_pm );
}




void client::closeEvent( QCloseEvent * _ce )
{
	hide();
	_ce->ignore();
}




void client::hideEvent( QHideEvent * )
{
	if( isMinimized() )
	{
		hide();
	}

	if( m_classRoomItem != NULL )
	{
		m_classRoomItem->setObserved( FALSE );
	}
}




void client::showEvent( QShowEvent * )
{
	if( m_classRoomItem != NULL )
	{
		m_classRoomItem->setObserved( TRUE );
	}
}




void client::mouseDoubleClickEvent( QMouseEvent * )
{
	workOnClient( "" );
}




bool client::inviteForSupportingLocalUser( void )
{
	m_syncMutex.lock();

	bool result = m_connection->inviteForSupport( systemEnvironment::realUserName(systemEnvironment::localUser() ) );

	m_syncMutex.unlock();

	return( result );
}




int client::id( void ) const
{
	QMap<int, client *>::const_iterator it;
	for( it = s_idMap.begin(); it != s_idMap.end(); ++it )
	{
		if( it.data() == this )
		{
			return( it.key() );
		}
	}
	return( 0 );
}




client * client::clientFromID( int _id )
{
	if( s_idMap.contains( _id ) )
	{
		return( s_idMap[_id] );
	}
	return( NULL );
}




int client::freeID( void )
{
	for( int i = 1; i < 32000; ++i )
	{
		if( s_idMap.contains( i ) == FALSE )
		{
			return( i );
		}
	}
	return( 0 );
}
