/*
 * italc.cpp - main-file for iTALC-Application
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include <signal.h>


#include <qapplication.h>
#include <qtranslator.h>
#include <qtextcodec.h>
#include <qdir.h>
#if QT_VERSION >= 0x030200
#include <qsplashscreen.h>
#endif
#include <qmessagebox.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qhbox.h>
#include <qpopupmenu.h>
#include <qsplitter.h>
#include <qtoolbutton.h>
#include <qmenubar.h>
#include <qworkspace.h>
#include <qbitmap.h>
#include <qdatetime.h>

#include "italc.h"
#include "client_manager.h"
#include "about_dialog.h"
#include "embed.h"
#include "paths.h"
#include "version.h"
#include "italc_side_bar.h"
#include "overview_widget.h"
#include "user_list.h"
#include "chat_management.h"
#include "screenshot_list.h"
#include "config_widget.h"
#include "help_widget.h"
#include "system_environment.h"
#include "italc_rfb_ext.h"
#include "rfb_connection.h"
#include "config.h"

#include "italc.moc"
#include "italc_side_bar.moc"


#if QT_VERSION >= 0x030200
QSplashScreen * splashScreen = NULL;
#endif

QString MASTER_HOST;


void interrupted( int sig )
{
	fprintf (stderr, "Got signal %d. Ignoring.\n", sig);
}



// good old main-function... initializes qt-app and starts iTALC
int main (int argc, char * * argv) {

	//signal (SIGBUS,  interrupted);
	//signal (SIGSEGV, interrupted);
	//signal (SIGFPE,  interrupted);


	QApplication app( argc, argv );
#if QT_VERSION >= 0x030200
	// set work-directory to
	//QDir::setCurrent( app.applicationDirPath() );
	QPixmap splash = embed::getIconPixmap( "splash" );
	QPixmap actual_splash = QPixmap::grabWindow( QApplication::desktop()->winId(),
							( QApplication::desktop()->availableGeometry().width()-splash.width() ) / 2,
							( QApplication::desktop()->availableGeometry().height()-splash.height() ) / 2,
							splash.width(), splash.height() );
	//bitBlt( &actual_splash, 0, 0, &splash, 0, 0, splash.width(), splash.height(), Qt::XorROP );
	splashScreen = new QSplashScreen( actual_splash );
	splashScreen->setMask( splash.createHeuristicMask() );
	splashScreen->show();

	// ok, let's have a nice fade-in-effect ;-)
	QImage splash_img = splash.convertToImage();
	QImage actual_splash_img = actual_splash.convertToImage();
	QImage dest_img = actual_splash.convertToImage();
	QTime t;
	t.start();
	while( t.elapsed() < 2000 && splashScreen->isShown() )
	{
		short int i = 256*t.elapsed()/1500;
		if( i > 256 )
		{
			i = 256;
		}

		memcpy( dest_img.bits(), actual_splash_img.bits(), actual_splash_img.width()*actual_splash_img.height() * 4 );
		register uchar * rs = splash_img.bits();
		register uchar * gs = rs + 1;
		register uchar * bs = gs + 1;
		register uchar * as = bs + 1;
		register uchar * rd = dest_img.bits();
		register uchar * gd = rd + 1;
		register uchar * bd = gd + 1;
		for( short int y = 0; y < dest_img.height(); ++y )
		{
			for( short int x = 0; x < dest_img.width(); ++x )
			{
				short int opac = static_cast<short int>( (i*(short int)*as)>>8 );
				*rd += ( ( ( *rs - *rd ) * opac ) >> 8 );
				rs += 4; rd += 4;
				*gd += ( ( ( *gs - *gd ) * opac ) >> 8 );
				gs += 4; gd += 4;
				*bd += ( ( ( *bs - *bd ) * opac ) >> 8 );
				bs += 4; bd += 4;
				as += 4;
			}
		}
		// conversion is slow as hell if desktop-depth != 24bpp...
		actual_splash.convertFromImage( dest_img );
		splashScreen->setPixmap( actual_splash );
	}
#endif

	// load translations
	QString loc = QString( QTextCodec::locale() ).section( '_', 0, 0 );
	// load translation for Qt-widgets/-dialogs
	embed::loadTranslation( QString( "qt_" ) + loc );
	// load actual translation for iTALC
	embed::loadTranslation( loc );

	// read private key, create it if it doesn't exist yet
	rfbConnection::initializeAuthentication();

	// now create the main-window
	italc * italc_main_window = italc::inst();
#if QT_VERSION >= 0x030200
	// hide splash-screen as soon as main-window is shown
	splashScreen->finish( italc_main_window );
#endif
	app.setMainWidget( italc_main_window );
	italc_main_window->show();

	// let's rock!!
	return( app.exec() );
}





italc * italc::s_instOfMe = NULL;

bool italc::ensureConfigPathExists( void )
{
	QDir home = QDir::home();
	// does a file/dir named ITALC_CONFIG_PATH exist?
	if( home.exists( ITALC_CONFIG_PATH ) == FALSE )
	{
		// no, then create dir
		return( home.mkdir( ITALC_CONFIG_PATH ) );
	}
	// exists, so test, whether it is a file
	else if( home.cd( ITALC_CONFIG_PATH ) == FALSE )
	{
		// it is, so we remove it
		if( home.remove( ITALC_CONFIG_PATH ) == FALSE )
		{
			// remove failed, we give up
			return( FALSE );
		}
		return( home.mkdir( ITALC_CONFIG_PATH ) );
	}
	return( TRUE );
}




italc::italc() :
	QMainWindow( 0, "", WDestructiveClose ),
	m_openedTabInSideBar( 1 )
{
	s_instOfMe = this;

	setCaption( tr( "iTALC" ) + " " + VER_STRING );
	setIcon( embed::getIconPixmap( "splash" ) );
	setUsesBigPixmaps( TRUE );


	QHBox * hbox = new QHBox( this );

	// create sidebar
	m_sideBar = new italcSideBar( italcSideBar::Vertical, hbox );
	m_sideBar->setStyle( italcSideBar::VSNET/*KDEV3ICON*/ );


	// create splitter, which is used for splitting sidebar-workspaces from main-workspace
	m_splitter = new QSplitter( Qt::Horizontal, hbox );
#if QT_VERSION >= 0x030200
	m_splitter->setChildrenCollapsible( FALSE );
#endif

	// create actual workspace for client-windows
	m_workspace = new QWorkspace( m_splitter );
	m_workspace->setScrollBarsEnabled( TRUE );
	m_workspace->setPaletteBackgroundPixmap( embed::getIconPixmap( "background_artwork" ) );


	// create instance of clientManager -> loads all available classrooms and clients from config-files
	clientManager * cm = clientManager::inst();
	// now create all sidebar-workspaces
	m_overviewWidget = new overviewWidget( m_splitter );
	m_userList = new userList( m_splitter );
	m_chatManagement = new chatManagement( m_splitter );
	m_screenshotList = new screenshotList( m_splitter );
	m_configWidget = new configWidget( m_splitter );
	m_helpWidget = new helpWidget( m_splitter );

	// append sidebar-workspaces to sidebar
	int id = 0;
	m_sideBar->appendTab( m_overviewWidget, ++id );
	m_sideBar->appendTab( cm, ++id );
	m_sideBar->appendTab( m_userList, ++id );
	m_sideBar->appendTab( m_chatManagement, ++id );
	m_sideBar->appendTab( m_screenshotList, ++id );
	m_sideBar->appendTab( m_configWidget, ++id );
	m_sideBar->appendTab( m_helpWidget, ++id );
	m_sideBar->setPosition( italcSideBar::Left );
	m_sideBar->setTab( m_openedTabInSideBar, TRUE );

	m_splitter->moveToLast( m_workspace );
	
	setCentralWidget( hbox );


	// file-popup-menu
	m_filePopupMenu = new QPopupMenu( this );
	menuBar()->insertItem( tr( "&File" ), m_filePopupMenu );

	m_filePopupMenu->insertItem( embed::getIconPixmap( "exit" ), tr( "&Quit" ), qApp, SLOT( closeAllWindows() ), CTRL+Key_Q );



	// clients-popup-menu
	m_actionsPopupMenu = new QPopupMenu( this );
	menuBar()->insertItem( tr( "&Actions" ), m_actionsPopupMenu );

	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_start_fullscreen_demo" ), tr( "Start fullscreen-demo" ), cm,
					SLOT( startFullScreenDemo() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_start_demo" ), tr( "Start window-demo" ), cm, SLOT( startDemo() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_stop_demo" ), tr( "Stop demo" ), cm, SLOT( stopDemo() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_msg" ), tr( "Send message to all clients" ), cm,
					SLOT( sendMessage() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "distribute_file" ), tr( "Distribute file" ), cm, SLOT( distributeFile() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "collect_files" ), tr( "Collect files" ), cm, SLOT( collectFiles() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_lock_x" ), tr( "Lock screens" ), cm, SLOT( lockScreens() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_unlock_x" ), tr( "Unlock screens" ), cm, SLOT( unlockScreens() ) );
	//m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_restart_x" ), tr( "Restart X on all clients" ), cm, SLOT( restartX() ) );

	m_actionsPopupMenu->insertSeparator();

	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_kill_games" ), tr( "Kill games on all clients" ), cm,
					SLOT( killGames() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_kill_browsers" ), tr( "Kill browsers on all clients" ), cm,
					SLOT( killBrowsers() ) );

	m_actionsPopupMenu->insertSeparator();

	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_power_on" ), tr( "Power on all clients" ), cm,
					SLOT( powerOnClients() ) );
	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client_reboot" ), tr( "Reboot all clients" ), cm,
					SLOT( rebootClients() ) );
	//m_actionsPopupMenu->insertSeparator ();
	m_actionsPopupMenu->insertItem (embed::getIconPixmap( "client_power_off" ), tr( "Power off all clients" ),
					cm, SLOT( powerOffClients() ) );
	m_actionsPopupMenu->insertItem (embed::getIconPixmap( "client_exec_cmds" ), tr( "Execute commands on all clients" ),
					cm, SLOT( execCmds() ) );



	m_actionsPopupMenu->insertSeparator();

	QPopupMenu * single_clients_submenu = new QPopupMenu( m_actionsPopupMenu );
	clientManager::inst()->createActionMenu( single_clients_submenu );

	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "client" ), tr( "Action on single client" ), single_clients_submenu );

	QPopupMenu * classrooms_submenu = new QPopupMenu( m_actionsPopupMenu );
	clientManager::inst()->createActionMenuForClassRooms( classrooms_submenu );

	m_actionsPopupMenu->insertItem( embed::getIconPixmap( "classroom_opened" ), tr( "Action for whole classroom" ),
					classrooms_submenu );



	// help-popup-menu
	m_helpPopupMenu = new QPopupMenu( this );
	menuBar()->insertItem( tr( "&Help" ), m_helpPopupMenu );

	m_helpPopupMenu->insertItem( embed::getIconPixmap( "whatsthis" ), tr( "What's this?" ), this, SLOT( enterWhatsThisMode() ) );
	m_helpPopupMenu->insertSeparator();
	m_helpPopupMenu->insertItem( embed::getIconPixmap( "info" ), tr( "About iTALC" ), this, SLOT( aboutITALC() ) );


	// create the action-toolbar
	m_actionToolBar = new QToolBar( this );
	addDockWindow( m_actionToolBar, tr( "Actions" ), DockTop, TRUE );

	QToolButton * sfd = new QToolButton( embed::getIconPixmap( "client_start_fullscreen_demo" ), tr( "Start fullscreen-demo" ),
						QString::null, cm, SLOT( startFullScreenDemo() ), m_actionToolBar );
	QWhatsThis::add( sfd, tr( "With this button you can start a fullscreen-demo on all visible clients. While a fullscreen-demo is "
					"running, the users neither are able to quit the demo nor can they do something else. This is "
					"good if you want to have the user's full attention." ) );

	QToolButton * swd = new QToolButton( embed::getIconPixmap( "client_start_demo" ), tr( "Start window-demo" ), QString::null, cm,
						SLOT( startDemo() ), m_actionToolBar );
	QWhatsThis::add( swd, tr( "With this button you can start a window-demo on all visible clients. The users are able to minimize or "
					"resize the demo-window, so they can continue with their work. But at any time they can restore the "
					"window, so that they can see, what you're showing to them." ) );

	QToolButton * sd = new QToolButton( embed::getIconPixmap( "client_stop_demo" ), tr( "Stop demo" ), QString::null, cm,
						SLOT( stopDemo() ), m_actionToolBar );
	QWhatsThis::add( sd, tr( "Click on this button if you want to stop the demo. It doesn't matter what kind of demo (fullscreen/window) "
					"you started. Either the screen is unlocked and the users can continue their work or the demo-window "
					"ist just closed." ) );

	QToolButton * sm = new QToolButton( embed::getIconPixmap( "client_msg" ), tr( "Send message to all clients" ), QString::null, cm,
					SLOT( sendMessage() ), m_actionToolBar );
	QWhatsThis::add( sm, tr( "To send a message to all users, you can use this button. After clicking on it, a window opens and you can "
					"enter the text, which should be shown on every client. This is for example useful when giving new "
					"tasks for every user." ) );

	QToolButton * df = new QToolButton( embed::getIconPixmap( "distribute_file" ), tr( "Distribute a file to all clients" ),
						QString::null, cm, SLOT( distributeFile() ), m_actionToolBar );
	QWhatsThis::add( df, tr( "If you want to distribute a file to all clients, you can click on this button. You can select the file "
					"you want to distribute and then this file is copied into every users home-directory." ) );

	QToolButton * cf = new QToolButton( embed::getIconPixmap( "collect_files" ), tr( "Collect files from all clients" ), QString::null,
						cm, SLOT( collectFiles() ), m_actionToolBar );
	QWhatsThis::add( cf, tr( "For collecting a specific file from every user/client click on this button. Then you can enter the "
					"filename (wildcards are possible) and the file is copied from each client if it exists." ) );

	QToolButton * ls = new QToolButton( embed::getIconPixmap( "client_lock_x" ), tr( "Lock screens" ), QString::null, cm,
						SLOT( lockScreens() ), m_actionToolBar);
	QWhatsThis::add( ls, tr( "Often it's neccessary to have the user's full attention. To achieve this, click on this button and the "
					"screen of every client is locked. The users can't do something, because their screens got black "
					"and the computer doesn't react to any inputs (mouse, keyboard...)." ) );

	QToolButton * uls = new QToolButton( embed::getIconPixmap( "client_unlock_x" ), tr( "Unlock screens" ), QString::null, cm,
						SLOT( unlockScreens() ), m_actionToolBar );
	QWhatsThis::add( uls, tr( "To unlock the client's screens, click on this button. Then all users will be able to continue their "
					"work." ) );

	m_actionToolBar->addSeparator();

	QToolButton * kg = new QToolButton( embed::getIconPixmap( "client_kill_games" ), tr( "Kill games on all clients" ), QString::null,
						cm, SLOT( killGames() ), m_actionToolBar );
	QWhatsThis::add( kg, tr( "Clicking this button will kill all typical Linux/KDE-Games. Especially when teaching young pupils this "
					"is needed very often..." ) );

	QToolButton * kb = new QToolButton( embed::getIconPixmap( "client_kill_browsers" ), tr( "Kill browsers on all clients" ),
						QString::null, cm, SLOT( killBrowsers() ), m_actionToolBar );
	QWhatsThis::add( kb, tr( "Clicking this button will kill all browser-windows. This can be used, if the use of the internet is not "
					"allowed at the moment." ) );

	m_actionToolBar->addSeparator();

	QToolButton * pon = new QToolButton( embed::getIconPixmap( "client_power_on" ), tr( "Power on all clients" ), QString::null, cm,
						SLOT( powerOnClients() ), m_actionToolBar );
	QWhatsThis::add( pon, tr( "Click on this button if you want to power on all clients. For using this feature, the clients must have "
					"Wake-on-LAN-capability. This function is very useful, because you don't have to power on each "
					"computer by hand." ) );

	QToolButton * rac = new QToolButton( embed::getIconPixmap( "client_reboot" ), tr( "Reboot all clients" ), QString::null, cm,
						SLOT( rebootClients() ), m_actionToolBar );
	QWhatsThis::add( rac, tr( "Sometimes it is neccessary to reboot all clients, for example if the configuration has changed. If users "
					"are logged in on the machines to be rebooted, a confirmation will appear for every client." ) );

	QToolButton * poff = new QToolButton( embed::getIconPixmap( "client_power_off" ), tr( "Power off all clients" ), QString::null, cm,
						SLOT( powerOffClients() ), m_actionToolBar );
	QWhatsThis::add( poff, tr( "If you want to power off all clients (for example at the end of a lesson) you can click on this "
					"button." ) );


	// now create toolbar for view-settings
	m_viewToolBar = new QToolBar( this );
	addDockWindow( m_viewToolBar, tr( "View" ), DockTop, FALSE );

	QToolButton * ocs = new QToolButton( embed::getIconPixmap( "optimize_client_size" ), tr( "Optimize client-size" ), QString::null,
						cm, SLOT( optimizeClientSize() ), m_viewToolBar );
	QWhatsThis::add( ocs, tr( "By clicking this button, the greates possible size for the client-windows is adjusted, so that you "
					"don't have to scroll. Then as much as possible available space is used. The client-windows are "
					"also aligned." ) );

	QToolButton * ics = new QToolButton( embed::getIconPixmap( "inc_client_size" ), tr( "Increase client-size" ), QString::null, cm,
						SLOT( increaseClientSize() ), m_viewToolBar );
	QWhatsThis::add( ics, tr( "When you click on this button, the size of the visible client-windows is increased, if they have not "
					"already the biggest possible size. The size is limited to protect privacy of the users." ) );

	QToolButton * dcs = new QToolButton( embed::getIconPixmap( "dec_client_size" ), tr( "Decrease client-size" ), QString::null, cm,
						SLOT( decreaseClientSize() ), m_viewToolBar );
	QWhatsThis::add( dcs, tr( "When you click on this button, the size of the visible client-windows is decreased, if they have not "
						"already the smallest possible size." ) );

	m_viewToolBar->addSeparator();

	QToolButton * swr = new QToolButton( embed::getIconPixmap( "classroom_switch" ), tr( "Switch classroom" ), QString::null,
						NULL, NULL, m_viewToolBar );
	swr->setPopup( clientManager::inst()->quickSwitchMenu() );
	swr->setPopupDelay( 1 );
	QWhatsThis::add( swr, tr( "Click on this button, to switch between classrooms." ) );

	QString win_cfg = cm->winCfg();
	QTextStream ts( &win_cfg, IO_ReadOnly );
	ts >> *( this );

	// if there's still a demo-server running (for example after iTALC-crash...) quit it...
	systemEnvironment::demoServer::stop();
	// ...and start a new one
	systemEnvironment::demoServer::start();


	QTimer::singleShot( 1000, cm, SLOT( updateClients() ) );


	// start client-update-thread with high priority. otherwise we could loose connections because of timeouts...
#if QT_VERSION >= 0x030200
	start( QThread::HighPriority );
#else
	start();
#endif
}




italc::~italc()
{
}




void italc::run( void )
{
	while( 1 )
	{
		// reload all clients...

		QValueVector<client *> clients = clientManager::inst()->visibleClients();

		// loop through all clients
		for( Q_UINT16 cl = 0; cl < clients.size(); ++cl )
		{
			// reload current client
			clients[cl]->processCmd( client::RELOAD, CONFIRM_NO );
		}

		m_userList->reload();

		if( client::reloadScreenshotList() )
		{
			m_screenshotList->reloadList();
		}
		client::resetReloadOfScreenshotList();

		// now sleep before reloading clients again
		QThread::sleep( clientManager::inst()->updateInterval() );

		// now do cleanup-work
		clientManager::inst()->doCleanupWork();
	}
}




void italc::closeEvent( QCloseEvent * _ce )
{
	// TODO: add check whether screens are locked!
	if( clientManager::inst()->demoRunning() )
	{
		QMessageBox::information( this, tr( "Demo running" ), tr( "You can't exit while a demo is running! Please stop the demo "
										"on all clients and try again!" ), QMessageBox::Ok );
		_ce->ignore();
	}
	else
	{
		terminate();
		wait();

		// kill demo-ivs
		systemEnvironment::demoServer::stop();

		clientManager::inst()->doCleanupWork();
		_ce->accept();
	}
}





void italc::aboutITALC( void )
{
	aboutDialog( this ).exec();
}
