/****************************************************************************
 *                              DFIOValChg.cc
 * Author: Matthew Ballance
 * Desc:   
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *
 ****************************************************************************/
#include "DFIOValChg.h"
#include <stdlib.h>
#include <stdio.h>

/**********************************************************
 * DFIOValChg_Size()
 **********************************************************/
Uint32 DFIOValChg_Size(Uint32 bitwidth, Uint32 count)
{
    Uint32 byteLen = (bitwidth>>3) + ((bitwidth&7)?1:0);

    byteLen--;

    Uint32 size_per_elem = sizeof(DFIOValChg)+byteLen*sizeof(DFIOVal);

    /**** If the size of a single value-change isn't 4-byte aligned,
     **** add some pad to the size
     ****/
    Uint32 mod;
    if ((mod = (size_per_elem & 0x3))) {
        size_per_elem += (4-mod);
    }

    return (size_per_elem * count);
}

/**********************************************************
 * DFIOValChg_Alloc()
 *
 * allocates an array of DFIOValChg structures
 **********************************************************/
DFIOValChg *DFIOValChg_Alloc(Uint32 bitwidth, Uint32 count)
{
    DFIOValChg *arr;

    arr = (DFIOValChg *)malloc(DFIOValChg_Size(bitwidth, count));

    return arr;
}

/**********************************************************
 * DFIOValChg_Init()
 **********************************************************/
void DFIOValChg_Init(DFIOValChg *changes, Uint32 bitwidth, Uint32 count)
{
    Uint32 i, x, vlen;

    vlen = bitwidth/8;
    if (bitwidth%8) {
        vlen++;
    }

    for (i=0; i<count; i++) {
        changes[i].changeTime = 0;
        for (x=0; x<vlen; x++) {
            changes[i].bitVal[x].aval = 0;
            changes[i].bitVal[x].bval = 0;
        }
    }
}

/**********************************************************
 * DFIOValChg_SetBit()
 **********************************************************/
void DFIOValChg_SetBit(DFIOValChg *chg, Uint32 bit, DFIOBitVal val)
{
    Uint32 idx   = (bit >> 3);
    Uint32 ibit  =  bit & 7;
    Uint32 mask  = ~(1 << ibit);

    chg->bitVal[idx].aval &= mask;
    chg->bitVal[idx].bval &= mask;
    chg->bitVal[idx].aval |= ((val&1) << ibit);
    chg->bitVal[idx].bval |= (((val&2) >> 1) << ibit);
}

/**********************************************************
 * DFIOValChg_GetBit()
 **********************************************************/
DFIOBitVal DFIOValChg_GetBit(const DFIOVal *bitVal, Uint32 bit)
{
    Uint32       idx  = (bit >> 3);
    Uint32       ibit  =  bit & 7;
    Uint32       mask = (1 << ibit);
    Uint32       a, b;
    DFIOBitVal   ret = DFIOBitVal_0;

    a = ((bitVal[idx].aval & mask) >> ibit);
    b = (((bitVal[idx].bval & mask) >> ibit) << 1);

    return ((DFIOBitVal)(a|b));
}

/**********************************************************
 * DFIOValChg_Idx()
 **********************************************************/
DFIOValChg *DFIOValChg_Idx(DFIOValChg *chgs, Uint32 width, Uint32 idx)
{
    DFIOValChg *chg;
    char       *ptr;
    Uint32      byteLen = (width>>3) + ((width&7)?1:0);
    byteLen--;

    Uint32 size_per_elem = sizeof(DFIOValChg)+byteLen*sizeof(DFIOVal);

    /**** If the size of a single value-change isn't 4-byte aligned,
     **** add some pad to the size
     ****/
    Uint32 mod;
    if ((mod = (size_per_elem & 0x3))) {
        size_per_elem += (4-mod);
    }

    ptr = &((char *)chgs)[idx*size_per_elem];

    return (DFIOValChg *)ptr;
}

/**********************************************************
 * DFIOValChg_Dump()
 **********************************************************/
void DFIOValChg_Dump(const DFIOVal *val, Uint32 len)
{
    fprintf(stderr, "DFIOValChg_Dump(): ");
    for (Uint32 i=0; i<len; i++) {
        fprintf(stderr, "%c ", "01zx"[(Uint32)DFIOValChg_GetBit(val, i)]);
    }
    fprintf(stderr, "\n");

    fprintf(stderr, "\t");
    for (Uint32 i=0; i<(len/8); i++) {
        fprintf(stderr, "(%d,%d) ", val[i].aval, val[i].bval);
    }
    fprintf(stderr, "\n");
}

#if 0
/**********************************************************
 * DFIOValChg()
 **********************************************************/
DFIOValChg::DFIOValChg(void) :
    bitVal((Uint32)0)
{
/*    value.realVal = 0.0; */
}

/**********************************************************
 * DFIOValChg(len)
 **********************************************************/
DFIOValChg::DFIOValChg(Uint32 len) :
    bitVal(len)
{
    changeTime = 0;
}
#endif


