/****************************************************************************
 *                           ShmDFIOTrace.cc
 *
 * Author: Matthew Ballance
 * Desc:   
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "ShmDFIOTrace.h"
#include "ShmValChgVector.h"
#include "ShmDFIO.h"


/********************************************************************
 * ShmDFIOTrace()
 *
 * Constructor for a reading trace
 ********************************************************************/
ShmDFIOTrace::ShmDFIOTrace(
        ShmDFIO            *dfio,
        ShmDFIOTraceDesc   *desc) : 
        DFIOTrace(desc->name, desc->rows, desc->msb, desc->lsb, 
                desc->traceType), d_dfio(dfio)
{
    d_dfio       = dfio;
    d_traceDesc  = dfio->Ptr2ShmPtr((char *)desc);
    d_numChgBlks = 0;
    d_initial    = 1;

    d_traceId = d_traceCount;

    d_traceCount++;
}

/********************************************************************
 * ShmDFIOTrace()
 *
 * Constructor for a writing trace 
 ********************************************************************/
ShmDFIOTrace::ShmDFIOTrace(
        ShmDFIO            *writer,
        Char               *name,
        Uint32              rows,
        Uint32              msb,
        Uint32              lsb,
        Uint32              traceType) : 
        DFIOTrace(name, rows, msb, lsb, traceType), d_dfio(writer)
{
    ShmDFIOTraceDesc *desc;
/*    char             *base = writer->getMapBase(); */

    d_traceId = d_traceCount;
    d_traceCount++;

    desc = (ShmDFIOTraceDesc *)writer->Malloc(
            sizeof(ShmDFIOTraceDesc)+strlen(name));


    strcpy(desc->name, name);
    desc->rows       = rows;
    desc->msb        = msb;
    desc->lsb        = lsb;
    desc->traceType  = traceType;
    desc->nextTrace  = 0;
    d_traceDesc      = writer->Ptr2ShmPtr((char *)desc);

    d_currValChgBlock = AllocChgBlock();

    d_traceTime = 0;
    d_traceTime64.low = d_traceTime64.high = 0;
    d_initial    = 1;

}

/********************************************************************
 * ~ShmDFIOTrace()
 ********************************************************************/
ShmDFIOTrace::~ShmDFIOTrace()
{
}

/********************************************************************
 * setValueBitStr()
 ********************************************************************/
Int32 ShmDFIOTrace::setValueBitStr(Uint32 row, Char *bitStr)
{
    Uint32               i, slen = strlen(bitStr);
    ShmDFIOChgBlkData   *blkData;
    DFIOValChg          *chg;

    /**** Must get the current location of the change block ****/
    blkData = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(
            d_currValChgBlock->d_blkStorePtr);

    if (!d_currValChgBlock->d_currChange) {
        d_currValChgBlock->d_startTime = parent->currentTime64;
        blkData->d_startTime = parent->currentTime64;
    }

    chg = DFIOValChg_Idx(blkData->d_changes, len, blkData->d_currChange);

    if ((chg->changeTime != parent->currentTime) && !d_initial) {
        d_currValChgBlock->d_currChange++;
        blkData->d_currChange++;
        if (d_currValChgBlock->d_currChange >= 1024) {
            d_currValChgBlock->d_currChange--;
            blkData->d_currChange--;

            /**** Set start/end times ****/
            d_currValChgBlock = AllocChgBlock();
        } else {
            d_currValChgBlock->d_endTime = parent->currentTime64;
            blkData->d_endTime           = parent->currentTime64;
        }

        blkData = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(
                d_currValChgBlock->d_blkStorePtr);
        chg = DFIOValChg_Idx(blkData->d_changes, len, blkData->d_currChange);

        if (!d_currValChgBlock->d_currChange) {
            d_currValChgBlock->d_startTime = parent->currentTime64;
            blkData->d_startTime = parent->currentTime64;

            d_currValChgBlock->d_endTime = parent->currentTime64;
            blkData->d_endTime           = parent->currentTime64;
        }
    }

    if (len > slen) {
        /**** High bits are trimmed off.. Therefore, fill in 
         **** these with the appropriate sign-extended value
         ****/
        DFIOBitVal   fillv;

        if (bitStr[0] == '0' || bitStr[0] == '1') {
            fillv = DFIOBitVal_0;
        } else if (bitStr[0] == 'z' || bitStr[0] == 'Z') {
            fillv = DFIOBitVal_Z;
        } else {
            fillv = DFIOBitVal_X;
        }
            
        for (i=0; i<(len-slen); i++) {
            DFIOValChg_SetBit(chg, slen+i, fillv);
        }

        for (i=0; i<slen; i++) {
            char bv = bitStr[slen-i-1];
            DFIOValChg_SetBit(chg, i,
                    (bv=='0')?DFIOBitVal_0:
                    (bv=='1')?DFIOBitVal_1:
                    (bv=='Z'||bv=='z')?DFIOBitVal_Z:DFIOBitVal_X);
        }
    } else {
        for (i=0; i<len; i++) {
            char bv = bitStr[len-i-1];
            DFIOValChg_SetBit(chg, i,
                    (bv=='0')?DFIOBitVal_0:
                    (bv=='1')?DFIOBitVal_1:
                    (bv=='Z'||bv=='z')?DFIOBitVal_Z:DFIOBitVal_X);
        }
    }

    chg->changeTime = parent->currentTime;
    d_currValChgBlock->d_endTime = parent->currentTime64;

    d_initial = 0;

    return 1;
}

/********************************************************************
 * setValueABVal8()
 ********************************************************************/
Int32 ShmDFIOTrace::setValueABVal8(Uint32 row, DFIOVal8 *value)
{

}

/********************************************************************
 * setValueABVal32()
 ********************************************************************/
Int32 ShmDFIOTrace::setValueABVal32(Uint32 row, DFIOVal32 *value)
{

}

/********************************************************************
 * clearTrace()
 ********************************************************************/
void ShmDFIOTrace::clearTrace()
{

}

/**********************************************************
 * findBlock()
 * Find the smallest block that has a timestamp larger
 * (or equal to) the starting time.
 **********************************************************/
Uint32 ShmDFIOTrace::findBlock(Uint32   reqTime)
{
    Int32           ub, lb, len;
    Int32           checkPt, i;
    Int32           tmp;
    ShmDFIOChgBlk  *pt, *pt2;

    lb = 0;
    len = d_valChgBlocks.length();
    ub  = len-1;
    checkPt = (ub-lb)/2;


    do {
        if (!(pt = d_valChgBlocks.idx(checkPt))) {
            break;
        }

        if (reqTime > pt->d_endTime.low) {

            /**** First, see if the next block has a start-time >= the
             **** reqTime...
             ****/
            if ((pt2 = d_valChgBlocks.idx(checkPt+1))) {
                if (pt2->d_startTime.low >= reqTime) {
                    return checkPt+1;
                }
            }

            lb = checkPt;
            tmp = (lb+(ub-lb)/2);

            if (tmp == checkPt) {
                checkPt++;
            } else {
                checkPt = (lb+(ub-lb)/2);
            }
        } else if (reqTime < pt->d_startTime.low) {
            ub = checkPt;
            tmp = lb+(ub-lb)/2;

            if (checkPt == tmp) {
                if (checkPt) {
                    checkPt--;
                } else {
                    break;
                }
            } else {
                checkPt = tmp;
            }
        } else {
            return checkPt;
        }
    } while ((lb >= 0) && (lb <= ub) && (ub >= 0) && (checkPt < len));

    /*** If we under- or over- flowed, fixup the return... ***/
    if (checkPt >= len) {
        checkPt = len-1;
    } else if (checkPt < 0) {
        checkPt = 0;
    }

    return checkPt;
}

/********************************************************************
 * getValue()
 *
 * Returns the first value with time <= begin. Vector 
 * extends to the last value <= end
 ********************************************************************/
Vector<DFIOValChg> *ShmDFIOTrace::getValue( 
        Uint32 begin, Uint32 end, Uint32 flags)
{
    ShmValChgVector      *ret;
    Uint32                done = 0, startTime = begin, idx, x;
    ShmDFIOChgBlk        *blkS, *tmp_blk;
    ShmDFIOChgBlkData    *blkD;
    Int32                 i;

    Update();

    ret = new ShmValChgVector(len);

    idx = findBlock(startTime);

    if (!(blkS = d_valChgBlocks.idx(idx))) {
        return ret;
    }

    blkD = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(blkS->d_blkStorePtr);

    if ((blkD->d_changes[0].changeTime > startTime)) {
        if (idx) {
            idx--;
        } else {
            /**** See if the end-time of this block somehow falls into
             **** the begin/end range...
             ****/
            if (blkD->d_changes[0].changeTime > end) {
                return ret;
            } else {
                startTime = blkD->d_changes[0].changeTime;
            }
        }
    }

   /**** Find the actual beginning of the data-of-interest
    **** within this block
    **** - Work from the end. Find the first val_chg entry
    ****   that has a timestamp <= the starting time
    ****/
    if (!(blkS = d_valChgBlocks.idx(idx))) {
        goto srw_exit;
    }

    blkD = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(blkS->d_blkStorePtr);

    /**** NOTE: Need a much more efficient way of doing
     **** this... Probably should do a bin-search 
     **** here...
     ****/
    for (i=(blkD->d_currChange); i>=0; i--) {
        if (DFIOValChg_Idx(blkD->d_changes, len, i)->changeTime 
                <= startTime) {
            if ((flags & GVF_OneMinus) && (i > 0)) { 
                i--;
            }
            ret->set_init_idx(i);
            break;
        }
    }
    ret->append(blkD);
    idx++;

    /**** Now, append blocks until the time-limit is reached
     ****/
    do {
        if (!(blkS = d_valChgBlocks.idx(idx))) {
            blkS = d_valChgBlocks.idx(idx-1);
            goto srw_exit;
        }

        if (blkS->d_startTime.low <= end) {
            blkD = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(blkS->d_blkStorePtr);
            ret->append(blkD); 
            idx++;
        } else if (blkS->d_endTime.low > end) {
            done = 1;
            if (idx) {
                blkS = d_valChgBlocks.idx(--idx);
            }
        }
    } while (!done);

srw_exit:
    /*** Find the value-change that is <= the requested end-time
     ***/
    if (blkS) {
        blkD = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(blkS->d_blkStorePtr);
        for (i=(blkD->d_currChange); i>=0; i--) {
            if (DFIOValChg_Idx(blkD->d_changes, len, i)->changeTime <= end) {
                if (flags & GVF_OnePlus && (i < (blkD->d_currChange))) {
                    i++;
                }
                break;
            }
        }

        if (i >= 0) {
            ret->decLen(blkD->d_currChange-i);
        } else if (!blkD->d_currChange) {
            ret->incLen(1);
        }
    }
    return ret;
}

/********************************************************************
 * Update()
 ********************************************************************/
int ShmDFIOTrace::Update()
{
    int ret = 0;
    ShmDFIOTraceDesc    *desc;
    ShmDFIOPtr           chgBlkPtr;
    ShmDFIOChgBlkData   *chgBlk;
    ShmDFIOChgBlk       *newBlk;

    d_dfio->UpdateMapping();
    
    desc = (ShmDFIOTraceDesc *)d_dfio->ShmPtr2Ptr(d_traceDesc);

    /**** Rescan each block... ****/
    if (desc->numChgBlks != d_numChgBlks) {
        if (d_numChgBlks > desc->numChgBlks) {
            fprintf(stderr, "ERROR: Number of blocks shrank!!\n");
        } else {
            /**** Okay, scan through the blocks that we do know about... ****/
            chgBlkPtr = desc->chgBlkList;
            for (Uint32 i=0; i<d_numChgBlks; i++) {
                if (!chgBlkPtr) {
                    fprintf(stderr, "ERROR: Reached the end of the ChgBlk "
                            "list too soon in ShmDFIOTrace::Update(1)\n");
                    fprintf(stderr, "\texpecting %d pre-existing blocks ; "
                            "found %d\n", d_numChgBlks, i);
                    return 0;
                }

                chgBlk = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(chgBlkPtr);
                chgBlkPtr = chgBlk->d_nextBlock;
            }

            /**** Now, add the blocks that are new ****/
            for (Uint32 i=0; i<(desc->numChgBlks-d_numChgBlks); i++) {
                if (!chgBlkPtr) {
                    fprintf(stderr, "ERROR: Reached the end of the ChgBlk "
                            "list too soon in ShmDFIOTrace::Update(2)\n");
                    fprintf(stderr, "\texpecting %d new blocks ; found %d\n",
                            (desc->numChgBlks-d_numChgBlks), i);
                    fprintf(stderr, "\trace is %s\n", desc->name);
                    return 0;
                }

                chgBlk = (ShmDFIOChgBlkData *)d_dfio->ShmPtr2Ptr(chgBlkPtr);

                newBlk = new ShmDFIOChgBlk(chgBlkPtr);
                newBlk->d_currChange = chgBlk->d_currChange;
                newBlk->d_startTime  = chgBlk->d_startTime;
                newBlk->d_endTime    = chgBlk->d_endTime;

                d_valChgBlocks.append(newBlk);

                chgBlkPtr = chgBlk->d_nextBlock;
            }
        }
    }

    d_numChgBlks = desc->numChgBlks;

    return ret;
}

/********************************************************************
 * AllocChgBlock()
 ********************************************************************/
ShmDFIOChgBlk *ShmDFIOTrace::AllocChgBlock()
{
    char             *mem;
    ShmDFIOPtr        memPtr;
    ShmDFIOChgBlk    *newBlk, *lastBlk;
    ShmDFIOTraceDesc *desc;

    mem = d_dfio->Malloc(sizeof(ShmDFIOChgBlkData)+
            DFIOValChg_Size(len, 1024));
    memPtr = d_dfio->Ptr2ShmPtr(mem);

    memset(mem, 0, DFIOValChg_Size(len, 1024));

    desc = (ShmDFIOTraceDesc *)d_dfio->ShmPtr2Ptr(d_traceDesc);

    newBlk = new ShmDFIOChgBlk(memPtr);
    memset(mem, 0, sizeof(ShmDFIOChgBlkData));

    if (!desc->chgBlkList) {
        desc->chgBlkList = newBlk->d_blkStorePtr;
    } else {
        ShmDFIOChgBlkData  *lastData;
        lastBlk = d_valChgBlocks.idx(d_valChgBlocks.length()-1);

        lastData = (ShmDFIOChgBlkData *)d_dfio->
            ShmPtr2Ptr(lastBlk->d_blkStorePtr);

        lastData->d_nextBlock = newBlk->d_blkStorePtr;
    }

    d_valChgBlocks.append(newBlk);

    desc->numChgBlks++;

    return newBlk;
}

Uint32 ShmDFIOTrace::d_traceCount = 0;

