/****************************************************************************
 *                            StubFilter.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements the import/export filter for Stub file format...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "StubFilter.h"
#include "DFIO.h"

#define FILTER_DESC \
    "The Stub Filter processes primitive data-dump files... This (as its " \
    "name implies), is a Stub - meant to get stuff working"

    /* S D D F => 53 44 44 46 */
static UChar    sdd_magic[] = { 0x53^0xFF, 0x44^0xFF, 0x44^0xFF, 0x46^0xFF};

#define SF_SECT_TRACES        0x5e334500
#define SF_SECT_DATA          0x5e334502
#define SF_SECT_TRACE_DATA    0x5e334504

#define     FP    stderr
#undef      DEBUG_IMPORT_EXPORT

#ifdef DEBUG_IMPORT_EXPORT
#define DBG_MSG(x)    fprintf x
#else
#define DBG_MSG(x)
#endif

/********************************************************************
 * StubFilter()
 ********************************************************************/
StubFilter::StubFilter(Tcl_Interp *interp) :
 DFIOFilter(interp, "DFIOFilter::StubFilter", "stub", FILTER_DESC, ".sdd")
{
    tmp_buf_len = 0;
    tmp_buf     = 0;
    str_buf     = 0;
    str_buf_len = 0;
    trace_vector = new Vector<DFIOTrace>();
}


/********************************************************************
 * isValidFile()
 *
 * Expect that the file is rewound to the beginning... Look for the
 * magic first 4 bytes:
 *
 ********************************************************************/
Int32 StubFilter::isValidFile(FileAccObj &file)
{
    Uint32 i;
    UChar  ch;

    for (i=0; i<4; i++) {
        file.read_uint8(&ch);
        if (ch != sdd_magic[i]) {
            break;
        }
    }

    if (i == 4) {
        return 1;
    } else {
        return 0;
    }
}

/********************************************************************
 * Import()
 ********************************************************************/
Int32 StubFilter::Import(DFIO **dfio, FileAccObj &file)
{
    data = *dfio;
    this->file = file;

    DBG_MSG((FP, "----> Import()\n"));

    trace_vector->empty();

    /**** Reads the magic_num... ****/
    if (!isValidFile(this->file)) {
        DBG_MSG((FP, "\tInvalid file\n"));
        return -1;
    }

    /**** Read in the traces...  ****/
    if (read_traces() < 0) {
        DBG_MSG((FP, "\tproblem reading traces\n"));
        return -1;
    }

    if (read_trace_data() < 0) {
        DBG_MSG((FP, "\tproblem reading data\n"));
        return -1;
    }

    DBG_MSG((FP, "<---- Import()\n"));

    return 0;
}

/********************************************************************
 * read_traces()
 ********************************************************************/
int StubFilter::read_traces()
{
    Uint32       num_traces, i, msb, lsb, sect;
    DFIOTrace   *trace;
    String       name(256);

    file.read_uint32(&sect);
    if (sect != SF_SECT_TRACES) {
        DBG_MSG((FP, "\tExpecting 0x%08x ; got 0x%08x\n",SF_SECT_TRACES,sect));
        return -1;
    }

    file.read_uint32(&num_traces);

    for (i=0; i<num_traces; i++) {
        /**** Read in name... ****/
        read_string(name);

        /**** Read in msb, lsb ****/
        file.read_uint32(&msb);
        file.read_uint32(&lsb);

        trace = data->newTrace(name.value(), 0, msb, lsb, 0);
        data->addTrace(trace);

        trace_vector->append(trace);
    }
    return 0;
}

/********************************************************************
 * read_trace_data()
 ********************************************************************/
int StubFilter::read_trace_data()
{
    Uint32            i, x, y, num_changes, b_idx, sect;
    Uint32            num_bytes, change_time, trace_width;
    DFIOTrace        *trace;
    String            val_tmp(256);

    file.read_uint32(&sect);
    if (sect != SF_SECT_DATA) {
        return -1;
    }

    for (i=0; i<trace_vector->length(); i++) {
        int z = 0;
        file.read_uint32(&sect);
        if (sect != SF_SECT_TRACE_DATA) {
            return -1;
        }

        file.read_uint32(&num_changes);

        trace = trace_vector->idx(i);

        DBG_MSG((FP, "Reading trace \"%s\"\n", trace->get_name()));
        DBG_MSG((FP, "\t%d changes\n", num_changes));

        trace_width = trace->get_len();
        val_tmp.chkSize(trace_width+1);

        num_bytes   = trace_width/4;

        if (trace_width%4) {
            num_bytes++;
        }
        
        if (num_bytes > tmp_buf_len) {
            if (tmp_buf) {
                delete[] tmp_buf;
            }
        
            tmp_buf = new UChar[num_bytes+1];
            tmp_buf_len = num_bytes+1;
        }


        for (x=0; x<num_changes; x++) {
            file.read_uint32(&change_time);

            file.read_bytes(tmp_buf, num_bytes);
            DBG_MSG((FP, "\t%d bytes read\n", num_bytes));

            DBG_MSG((FP, "\t\t"));
            for (y=0; y<num_bytes; y++) {
                DBG_MSG((FP, "%02x ", tmp_buf[y]));
            }
            DBG_MSG((FP, "\n"));

            b_idx = 0;
            for (y=0; y<trace_width; y++) {
                if (!(y%4) && y) {
                    b_idx++;
                }
                val_tmp[y] = bval_to_char(tmp_buf[b_idx] >> ((y%4)*2));
            }

            DBG_MSG((FP, "\t\t"));
            for (y=0; y<trace_width; y++) {
                DBG_MSG((FP, "%c", val_tmp[y]));
            }
            DBG_MSG((FP, "\n"));

            val_tmp[trace_width] = 0;

            trace->setTime(change_time);
            trace->setValueBitStr(0, val_tmp.value());
        }
    }

    return 0;
}

/********************************************************************
 * Export()
 ********************************************************************/
Int32 StubFilter::Export(DFIO **dfio, FileAccObj &file)
{
    this->file   = file;
    data = *dfio;

    max_trace_width = 0;

    DBG_MSG((FP, "----> Export()\n"));

    /**** Write out the magic number... ****/
    file.write_bytes(sdd_magic, 4);

    dump_traces();

    dump_trace_data();

    DBG_MSG((FP, "<---- Export()\n"));

    return 100;
}

/********************************************************************
 * dump_traces()
 *
 * Section is:
 * SF_SECT_TRACES
 * <NumTraces>
 * Foreach trace:
 *     <TraceName>
 *     <Msb>
 *     <Lsb>
 ********************************************************************/
void StubFilter::dump_traces()
{
    Uint32        i, num_traces, msb, lsb, len;
    DFIOTrace    *trace;

    file.write_uint32(SF_SECT_TRACES);

    num_traces = data->traces->length();
    file.write_uint32(num_traces);

    for (i=0; i<num_traces; i++) {
        trace = data->traces->idx(i);

        dump_string(
                trace->get_name_str().value(),
                trace->get_name_str().length());

        msb = trace->get_msb();
        lsb = trace->get_lsb();


        file.write_uint32(msb);
        file.write_uint32(lsb);
        len = (msb>lsb)?(msb-lsb):(lsb-msb);
        if (len > max_trace_width) {
            max_trace_width = len;
        }
    }
}

/********************************************************************
 * dump_trace_data()
 *
 * SF_SECT_DATA
 * Foreach Trace
 *     SF_SECT_TRACE_DATA
 *     <UINT32 NumChanges>
 *         Changes...
 *         <UINT32 ChangeTime>
 *         <ChangeData>
 ********************************************************************/
void StubFilter::dump_trace_data()
{
    Uint32                 i, num_traces, x, chg_len, width;
    Uint32                 b_idx, trace_len, y;
    DFIOTrace             *trace;
    Vector<DFIOValChg>    *chgs;
    DFIOValChg            *chg;
    UChar                  v1, v2;

    width = max_trace_width/4;
    if (max_trace_width%4 || !width) {
        width++;
    }

    if (width > tmp_buf_len) {
        if (tmp_buf) {
            delete[] tmp_buf;
        }
        
        tmp_buf = new UChar[width];
        tmp_buf_len = width;
    }

    file.write_uint32(SF_SECT_DATA);

    num_traces = data->traces->length();

    for (i=0; i<num_traces; i++) {
        trace = data->traces->idx(i);

        DBG_MSG((FP, "Writing trace \"%s\"\n", trace->get_name()));

        chgs = trace->getValue(0, 0xFFFFFFFF, 0);
        chg_len = chgs->length();

        file.write_uint32(SF_SECT_TRACE_DATA);
        file.write_uint32(chg_len);

        trace_len = trace->get_len();
        DBG_MSG((FP, "\tlength = %d\n", trace_len));

        for (x=0; x<chg_len; x++) {
            chg   = chgs->idx(x);
            b_idx = 0;

            file.write_uint32(chg->changeTime);

            DBG_MSG((FP, "\t\t"));
            for (y=0; y<trace_len; y++) {
                DBG_MSG((FP, "%c", chg->bitVal[y]));
            }
            DBG_MSG((FP, "\n"));


            for (y=0; y<trace_len; y++) {
                if (!(y%4)) {
                    if (y) {
                        b_idx++;
                    }
                    tmp_buf[b_idx] = 0;
                }

                v1 = tmp_buf[b_idx];
                tmp_buf[b_idx] |= 
                 (DFIOValChg_GetBit(chg->bitVal, (trace_len-y-1)) << ((y%4)*2));
                v2 = tmp_buf[b_idx];
                DBG_MSG((FP, "\t\tv1 = %02x ; v2 = %02x ; y = %d ; %d ; %d\n",
                            v1, v2, y, char_to_bval(chg->bitVal[y]),
                            (char_to_bval(chg->bitVal[y]) << ((y%4)*2))));
            }

            b_idx++;

            DBG_MSG((FP, "\t\t"));
            for (y=0; y<b_idx; y++) {
                DBG_MSG((FP, "%02x ", tmp_buf[y]));
            }
            DBG_MSG((FP, "\n"));

            /**** Dump out data for this change... ****/
            file.write_bytes(tmp_buf, b_idx);

            DBG_MSG((FP, "\t%d bytes written\n", b_idx));

        }
        delete chgs;
    }
}


