/*
 * JasperReports - Free Java Reporting Library.
 * Copyright (C) 2001 - 2009 Jaspersoft Corporation. All rights reserved.
 * http://www.jaspersoft.com
 *
 * Unless you have purchased a commercial license agreement from Jaspersoft,
 * the following license terms apply:
 *
 * This program is part of JasperReports.
 *
 * JasperReports is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * JasperReports is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with JasperReports. If not, see <http://www.gnu.org/licenses/>.
 */
package net.sf.jasperreports.charts.base;

import java.awt.Color;
import java.io.Serializable;

import net.sf.jasperreports.charts.JRValueDisplay;
import net.sf.jasperreports.engine.JRChart;
import net.sf.jasperreports.engine.JRConstants;
import net.sf.jasperreports.engine.JRExpressionCollector;
import net.sf.jasperreports.engine.JRFont;
import net.sf.jasperreports.engine.JRRuntimeException;
import net.sf.jasperreports.engine.base.JRBaseFont;
import net.sf.jasperreports.engine.base.JRBaseObjectFactory;

/**
 * An immutable representation of the formatting options for showing the
 * value of a value dataset.  Used by charts that display a single value,
 * such as a Meter or Thermometer.
 *
 * @author Barry Klawans (bklawans@users.sourceforge.net)
 * @version $Id: JRBaseValueDisplay.java 3032 2009-08-27 11:32:28Z teodord $
 */
public class JRBaseValueDisplay implements JRValueDisplay, Serializable
{


	/**
	 *
	 */
	private static final long serialVersionUID = JRConstants.SERIAL_VERSION_UID;

	protected JRChart chart = null;

	/**
	 * The color to use when writing the value.
	 */
	protected Color color = null;

	/**
	 * The formatting mask to use when writing the value.  Must a pattern
	 * that is accepted by a code>java.text.DecimalFormat</code> object.
	 */
	protected String mask = null;

	/**
	 * The font to use when writing the value.
	 */
	protected JRFont font = null;


	/**
	 * Constructs a copy of an existing value format specification.
	 *
	 * @param valueDisplay the value formatting object to copy
	 * @param chart the parent chart
	 */
	public JRBaseValueDisplay(JRValueDisplay valueDisplay, JRChart chart)
	{
		this.chart = chart;
		
		if (valueDisplay == null)
		{
			font = new JRBaseFont(chart, null);
		}
		else
		{
			color = valueDisplay.getColor();
			mask = valueDisplay.getMask();
			font = new JRBaseFont(valueDisplay.getChart(), valueDisplay.getFont());
		}
	}

	/**
	 * Constructs a copy of an existing value format specification and registers
	 * any expression in the new copy with the specified factory.
	 *
	 * @param valueDisplay the value formatting object to copy
	 * @param factory the factory object to register expressions with
	 */
	public JRBaseValueDisplay(JRValueDisplay valueDisplay, JRBaseObjectFactory factory)
	{
		factory.put(valueDisplay, this);

		chart = (JRChart)factory.getVisitResult(valueDisplay.getChart());

		color = valueDisplay.getColor();
		mask = valueDisplay.getMask();
		font = new JRBaseFont(valueDisplay.getChart(), valueDisplay.getFont());
	}


	/**
	 *
	 */
	public JRChart getChart()
	{
		return chart;
	}

	/**
	 *
	 */
	public Color getColor()
	{
		return color;
	}
	/**
	 *
	 */
	public String getMask()
	{
		return mask;
	}
	/**
	 *
	 */
	public JRFont getFont()
	{
		return font;
	}

	/**
	 * Adds all the expression used by this plot with the specified collector.
	 * All collected expression that are also registered with a factory will
	 * be included with the report is compiled.
	 *
	 * @param collector the expression collector to use
	 */
	public void collectExpressions(JRExpressionCollector collector)
	{
	}

	/**
	 * 
	 */
	public Object clone() 
	{
		try
		{
			return super.clone();
		}
		catch (CloneNotSupportedException e)
		{
			throw new JRRuntimeException(e);
		}
	}
}
