/*
 * Java-Gnome Bindings Library
 *
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 *
 * The Java-Gnome bindings library is free software distributed under
 * the terms of the GNU Library General Public License version 2.
 */
package org.gnu.gnomevfs;

import org.gnu.glib.GObject;

/**
 * An object that describes a URI element
 */
public class VFSURI extends GObject {
	
	public VFSURI(String uri) {
		super(gnome_vfs_uri_new(uri));
	}
	
	VFSURI(int handle) {
		super(handle);
	}
	
	/**
	 * Create a new VFSURI obtained by appending uriFragment to the
	 * current VFSURI.
	 * 
	 * @param uriFragment
	 * @return
	 */
	public VFSURI appendString(String uriFragment) {
		return new VFSURI(gnome_vfs_uri_append_string(getHandle(), uriFragment));
	}
	
	/**
	 * Create a new VFSURI obtained by appending path to the
	 * current VFSURI.
	 * 
	 * @param path
	 * @return
	 */
	public VFSURI appendPath(String path) {
		return new VFSURI(gnome_vfs_uri_append_path(getHandle(), path));
	}
	
	/**
	 * Create a new VFSURI obtained by appending filename to the
	 * current VFSURI.
	 * 
	 * @param filename
	 * @return
	 */
	public VFSURI appendFilename(String filename) {
		return new VFSURI(gnome_vfs_uri_append_file_name(getHandle(), filename));
	}
	
	/**
	 * Translate the VFSURI into a printable string.
	 * 
	 * @param hideOptions
	 * @return
	 */
	public String toString(VFSURIHideOptions hideOptions) {
		return gnome_vfs_uri_to_string(getHandle(), hideOptions.getValue());
	}
	
	/**
	 * Check if uri is local (native) file system.
	 * 
	 * @return
	 */
	public boolean isLocal() {
		return gnome_vfs_uri_is_local(getHandle());
	}
	
	/**
	 * Check if the uri has a parent or not.
	 * 
	 * @return
	 */
	public boolean hasParent() {
		return gnome_vfs_uri_has_parent(getHandle());
	}
	
	/**
	 * Retrieve the uri's parent.
	 * 
	 * @return
	 */
	public VFSURI getParent() {
		return new VFSURI(gnome_vfs_uri_get_parent(getHandle()));
	}
	
	/**
	 * Retrieve the toplevel VFSURI in the uri.
	 * 
	 * @return
	 */
	public VFSURI getToplevel() {
		return new VFSURI(gnome_vfs_uri_get_toplevel(getHandle()));
	}
	
	/**
	 * Retrieve the host name for uri.
	 * 
	 * @return
	 */
	public String getHostName() {
		return gnome_vfs_uri_get_host_name(getHandle());
	}
	
	/**
	 * Retrieve the scheme for uri.
	 * 
	 * @return
	 */
	public String getScheme() {
		return gnome_vfs_uri_get_scheme(getHandle());
	}
	
	/**
	 * Retrieve the host port number in uri.
	 * @return
	 */
	public int getHostPort() {
		return gnome_vfs_uri_get_host_port(getHandle());
	}
	
	/**
	 * Retrieve the user name in uri.
	 * 
	 * @return
	 */
	public String getUserName() {
		return gnome_vfs_uri_get_user_name(getHandle());
	}
	
	/**
	 * Retrieve the password in uri.
	 * 
	 * @return
	 */
	public String getPassword() {
		return gnome_vfs_uri_get_password(getHandle());
	}
	
	/**
	 * Set the host name accessed by uri.
	 * 
	 * @param hostname
	 */
	public void setHostName(String hostname) {
		gnome_vfs_uri_set_host_name(getHandle(), hostname);
	}
	
	/**
	 * Set the host port number in uri.  If the port number
	 * is zero, the default port for the uri's toplevel access
	 * method is used.
	 * 
	 * @param portNumber
	 */
	public void setHostPort(int portNumber) {
		gnome_vfs_uri_set_host_port(getHandle(), portNumber);
	}
	
	/**
	 * Set the user name for the uri.
	 * 
	 * @param username
	 */
	public void setUserName(String username) {
		gnome_vfs_uri_set_user_name(getHandle(), username);
	}
	
	/**
	 * Set the password for the uri.
	 * 
	 * @param password
	 */
	public void setPassword(String password) {
		gnome_vfs_uri_set_password(getHandle(), password);
	}
	
	/**
	 * Returns true if the provided uri is equal to this object.
	 * 
	 * @param uri
	 * @return
	 */
	public boolean isEqual(VFSURI uri) {
		return gnome_vfs_uri_equal(getHandle(), uri.getHandle());
	}
	
	/**
	 * Check if child is contained in this uri.
	 * 
	 * @param child
	 * @param recursive
	 * @return
	 */
	public boolean isParent(VFSURI child, boolean recursive) {
		return gnome_vfs_uri_is_parent(getHandle(), child.getHandle(), recursive);
	}
	
	/**
	 * Retrieve the full path name for the uri.
	 * 
	 * @return
	 */
	public String getPath() {
		return gnome_vfs_uri_get_path(getHandle());
	}

    /****************************************
     * BEGINNING OF JNI CODE
     ****************************************/
    native static final protected int gnome_vfs_uri_new(String uri);
    native static final protected int gnome_vfs_uri_resolve_relative(int base, String relatedReference);
    native static final protected int gnome_vfs_uri_append_string(int uri, String uriFragment);
    native static final protected int gnome_vfs_uri_append_path(int uri, String path);
    native static final protected int gnome_vfs_uri_append_file_name(int uri, String filename);
    native static final protected String gnome_vfs_uri_to_string(int uri, int hideOptions);
    native static final protected int gnome_vfs_uri_dup(int uri);
    native static final protected boolean gnome_vfs_uri_is_local(int uri);
    native static final protected boolean gnome_vfs_uri_has_parent(int uri);
    native static final protected int gnome_vfs_uri_get_parent(int uri);
    native static final protected int gnome_vfs_uri_get_toplevel(int uri);
    native static final protected String gnome_vfs_uri_get_host_name(int uri);
    native static final protected String gnome_vfs_uri_get_scheme(int uri);
    native static final protected int gnome_vfs_uri_get_host_port(int uri);
    native static final protected String gnome_vfs_uri_get_user_name(int uri);
    native static final protected String gnome_vfs_uri_get_password(int uri);
    native static final protected void gnome_vfs_uri_set_host_name(int uri, String host);
    native static final protected void gnome_vfs_uri_set_host_port(int uri, int port);
    native static final protected void gnome_vfs_uri_set_user_name(int uri, String username);
    native static final protected void gnome_vfs_uri_set_password(int uri, String password);
    native static final protected boolean gnome_vfs_uri_equal(int a, int b);
    native static final protected boolean gnome_vfs_uri_is_parent(int uri, int child, boolean recursive);
    native static final protected String gnome_vfs_uri_get_path(int uri);
    native static final protected String gnome_vfs_uri_get_fragment_identifier(int uri);
    native static final protected String gnome_vfs_uri_extract_dirname(int uri);
    native static final protected String gnome_vfs_uri_extract_short_name(int uri);
    native static final protected String gnome_vfs_uri_extract_short_path_name(int uri);
    /****************************************
     * END OF JNI CODE
     ****************************************/
}
