<?php
    /**
     * Daemon poller agent
     * Controls clients via shared memory
     */
    declare(ticks=1)

    $jffnms_functions_include="engine";
    include_once("../conf/config.php");

    $my_sat_id = satellite_my_id();

    @set_time_limit(0);
	
    pcntl_signal(SIGHUP, "sig_handler");
    pcntl_signal(SIGINT, "sig_handler");
    pcntl_signal(SIGTERM, "sig_handler");	


    $nr_clients = 30;
    $max_pollings = 30;
    $max_command_time = 300;
    
    $child_name = "exp_poller_child.php";
    $clients;
    $server_shm;

    function main($argc,$argv) {
	global $clients,$busy_clients,$my_sat_id;
 	/**
	  * Initialize the shared memory and star the clients 
	  */

	/**
	 * Check if a flag (-F) is provided,
	 * In that case, run in foreground
	 */
	if ($argv["1"]!="-f") { 
	    /**
	     * Fork first
	    */
	 
	    $forkpid = pcntl_fork();
	    if ($forkpid == -1) {
    		die("could not fork");
	    } else if ($forkpid) {
    		// we are the parent
	        //pcntl_waitpid(-1,$status); // this waits on the child ****
		exit();
	    } else {
    		// we are the child
    	        // code block for child process to run
	    }
	
	    if (!posix_setsid())
		die("could not detach from terminal");
	}

	init();
	CheckStatus();

	$total_interfaces_polled = 0;
	while ( 1 ) { //main loop
	    $now = time()-30;
	    
	    $query_hosts="
		SELECT 
		    hosts.id as host_id, interfaces.id as interface_id, 
		    (interfaces.last_poll_date + it.rrd_structure_step) as next_poll
    		FROM 
		    hosts, interfaces ,interface_types as it 
		WHERE 
		    interfaces.host = hosts.id and interfaces.type = it.id and interfaces.id > 1 and 
		    hosts.poll > 0 and interfaces.poll > 0 and hosts.satellite = $my_sat_id and
		    $now > (interfaces.last_poll_date + it.rrd_structure_step)
		ORDER BY
		    next_poll asc
	    ";
            
	    if (!$result_hosts = db_query ($query_hosts)) {
	        logmsg( "Query failed - R2 - ".db_error()."\n");
		break;
	    }
	
	    $num = db_num_rows($result_hosts);

	    if ( $num > 0 ) { //if there's something to poll
	
		$poll_time = time_usec ($poll_time);
		$polled_interfaces = 0;
		
		while ($host = db_fetch_array($result_hosts)) {

		    logmsg("Polling Status: $polled_interfaces/$num\n");

		    $host_id = $host["host_id"];
		    $interface_id = $host["interface_id"];
		    $cmd = "$host_id $interface_id";
		
		    /**
		     * Get a free child, wait otherwise
		     */
   		
		    while (($key = GetFreeClient()) === false) {
		        //logmsg("Waiting for Available Client...\n");
		        logmsg(".");
		        sleep(1);
		    }
		
		    //logmsg("Sending $cmd to $key\n");
		    
		    SendCmd($clients[$key]["shm"],$cmd);
		    
		    $clients[$key]["usage"]++;
		    $clients[$key]["last_command"]=$cmd;
		    $clients[$key]["last_command_time"]=time();
		    $polled_interfaces++;    
		}
		
	        db_free($result_hosts);
		//logmsg("Freed DB Result\n");
		//usleep(20);

		$poll_time = time_usec_diff($poll_time);
		logmsg("Polled $num interfaces, Total Poll time was $poll_time msec, ".round($poll_time/$num)." msec average per interface.\n");
	    }
	    
	    $cleaned_something = cleanup_clients(&$clients);

	    if (!$cleaned_something) {
		logmsg("Waiting for next round...\n");
	        sleep(5);
	    }
	    
	    $total_interfaces_polled += $num;
	    
	    logmsg("Total interfaces polled since start: $total_interfaces_polled\n");
	}
	
	logmsg("Shutting down ALL Clients...\n");
	shutdown();
    }

    function cleanup_clients ($clients) {
	global $max_pollings,$max_command_time;
	
	$result = false;
	$now = time();
		
	foreach ($clients as $number=>$client) {
	    $shut_it_down = 0;
	
	    if ($client["usage"] >= $max_pollings) {
		logmsg("Client $number has been used ".$client["usage"]." times, restarting it to prevent memory leaks\n");
		$shut_it_down = 1;
	    }
	
	    $time_processing = $now - $client["last_command_time"];
	    if (($client["status"]==false) && ($time_processing >= $max_command_time)) {
		logmsg("Client $number has been processing for $time_processing seconds, I think its hanged, restarting it\n");
		$shut_it_down = 1;
	    }
	    
	    if ($shut_it_down == 1) {
		usleep(20);
		shutdown_client($number,&$clients[$number]);
		init_client($number,&$clients[$number]);
		sleep(2);
		CheckClientStatus($number,&$clients[$number]);
	
		$result = true;
	    }
	}
	return $result;
    }

    function logmsg($text) {
	global $server_shm;
	/**
	 * Function to logmsg data to a shared memory
	 */
	logger ($text);
	SendCmd($server_shm,$text);
	usleep(10);
    }
    
    function sig_handler($signo) {
	switch($signo) {
	    case SIGTERM:
	    case SIGINT:
            // handle shutdown tasks
			logmsg("$signo Caught!, Shuting down");
    			shutdown();
			exit();
            		break;
         case SIGHUP:
             // handle restart tasks
			logmsg("$signo Caught!, Restarting");
            		shutdown();
			init();
			CheckStatus();
			break;
         default:
             // handle all other signals
			logmsg ("Received Signal $signo\n");
			break;
        }
    }

    function make_seed() {
	list($usec, $sec) = explode(' ', microtime());
        return (float) $sec + ((float) $usec * 100000);
    }

    function init_client($number,$client) {
	global $child_name;

	if ($number > 0) {	
	    /**
	    * Ok, generate the IDs and create the shared memory pointers
	     */
	    logmsg("Lauching Client $number...\n");
	    $shm_key = $number;
	    $shm_id = shmop_open($shm_key, "c", 0644, 1024);
	    logmsg("Client $number SHMid is:".$shm_id." SHMkey: ".$shm_key."\n");
	    usleep(10);
        
	    /**
             * Empty the shared memory, just in case
             */
            SendCmd($shm_id,"");

	    $php = get_config_option("php_executable");
	    $process = popen("$php -q $child_name ".str_pad("$shm_key - - Child Number $number",30),'r');

	    $client["shm"]=$shm_id;
	    $client["process"]=$process;
	    $client["status"]=TRUE;
	    $client["usage"]=0;
	    $client["last_command_time"]=time();
	    usleep(10);
	}
    }

    function init() {
	global $server_shm,$clients,$nr_clients;
	/**
	 * Function to generate shared memory pointers and clients
	 */
	/**
	 * Init the random engine first
	 */
	srand(make_seed());
	
	/**
	 * Setup a shared memory for daemon diagnostic / controll
	 * Use the pid as indetifier,
	 */
	$pid = getmypid();
	$server_shm = shmop_open($pid,"c",0644,1024);

	//init all clients;	
	for ( $i = 1; $i <= $nr_clients; $i++ ) 
	    init_client($i,&$clients[$i]);
    }

    function SendCmd($shm,$cmd) {
	/**
	 * Function that writes something to a shared memory, that a
	 * "Client" ( Child, or other process) is reading from
	 */
	
	/**
	 * Ok, lets put some data into the shared memory
	 */
	$cmd = str_pad($cmd,@shmop_size($shm)-1, " ", STR_PAD_RIGHT);

	$shm_bytes_written = @shmop_write($shm, $cmd, 0);
	
	return $shm_bytes_written;
    }

    function ReadCmd($shm) {
	/**
	 * Read what was put in the shared memory
	 */
	$data = @shmop_read($shm,0,shmop_size($shm));
	return trim($data);
    }

    function CheckClientStatus($number,$client) {
 
	if ($number > 0) {
 	    /**
	     * Check status of the client 
	     */
	    
	    $data = ReadCmd($client["shm"]);
	    logmsg(str_pad("STATUS ".$number,12)." ".str_pad("Usage: ".$client["usage"],15)." ".str_pad("Time: ".(time()-$client["last_command_time"]),10)." = $data\n");
	    //usleep(10);

	    list ($data) = explode (" ",$data);

	    if (trim($data) == "READY")
		$client["status"] = true;
	    else
		$client["status"] = false;
	}
	return $client["status"];
    }

    function CheckStatus() {
	global $clients;

	foreach ($clients as $number=>$client)
	    CheckClientStatus($number,&$clients[$number]);
	
	cleanup_clients(&$clients);
    }

    function close_shm($shm_id) {
 	/**
	 * Closes the shared memory
	 */
	
	shmop_delete($shm_id);
        shmop_close($shm_id);
    }

    // Not Used
    function shutdown_process($process) {
	@pclose($process);
    }

    function shutdown_client($number,$client) {
	$wait = 5;
    
	if ($number > 0) {
	    $starttime = time();

	    logmsg( "Shutting Down Client $number...\n");
	    
	    //Wait until the client is READY
	    while (CheckClientStatus($number,&$client) === false) {

		$timediff = time() - $starttime;
		if ($timediff > $wait) {
		    logmsg("More than $wait seconds waiting for Client $number to finish... Shutting it down.\n");
		    break;
		}
		
		logmsg("Waiting Client $number to finish...\n");
		sleep(1);
	    }


	    $starttime = time();
	    //wait until Status is Exiting, that means the client has been shutdown
	    while (($return = ReadCmd($client["shm"])) != "Exiting") {
		
		if ($return=="READY") {
		    logmsg("Sending EXIT to Client $number\n");
		    SendCmd($client["shm"],"EXIT");
		}

		$timediff = time() - $starttime;
		if ($timediff > $wait) {
		    logmsg("More than $wait seconds waiting for Client $number to shutdown... Killing it.\n");
		    break;
		}
		
		logmsg("Waiting Client $number to shutdown: $return\n");
		sleep(1);
	    }
	    
	    logmsg ("Last Status from Client $number is $return\n");
	    
	    logmsg("Closing down Shared memory resource ".$client["shm"]."\n");
	    close_shm($client["shm"]);
	
	    //logmsg("Closing process handler for Client $number\n");
	    //shutdown_process($client["process"]);
	    
	    logmsg("Shutdown of Client $number, done.\n");
	
	    unset ($client);
	}
    }

    function shutdown() {
	global $clients,$server_shm;

	$waittime = 5;
	logmsg("Shutting down!\n");
	logmsg("Waiting $waittime Seconds for clients to cleanup\n");
	sleep($waittime);

	foreach ($clients as $number=>$client)
	    shutdown_client($number,$clients[$number]);
	
	logmsg("Closing daemon shared memory resource at $server_shm...\n");
	close_shm($server_shm);
	unset ($clients);
    }

    
    function GetFreeClient() {
	global $clients;
	
	CheckStatus();

	foreach ($clients as $key=>$client)
	    if ($client["status"] === true) //get the first free client;
		return $key;

	/**
	 * If we are here, it means that we have no free childs
	 * Return false in that case
	 */
	return false;
    }

    main($argc,$argv);
?>
