#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

static void write_buf (char* buf, char* str) {
    while ((*buf++ = *str++))
	;
}

/** 
 * @param class_name the field type to translate.
 * @param buf the output buffer.
 * @return the length of the converted string.
 */
int translate_field_type (const char* cls, char* buf) {
    *buf = '\0';
    if (*cls == '[') {
	int l;
	l = translate_field_type (cls + 1, buf);
	buf += l;
	*buf++ = '[';
	*buf++ = ']';
	*buf++ = '\0';
	return l + 2;
    } else {
	int l = strlen (cls);
	int r = 0;
	if (l == 1) {
	    switch (*cls) {
	    case 'L':
		write_buf (buf, "java.lang.Object");
		r = 16;
		break;
	    case 'Z':
		write_buf (buf, "boolean");
		r = 7;
		break;
	    case 'B':
		write_buf (buf, "byte");
		r = 4;
		break;
	    case 'C':
		write_buf (buf, "char");
		r = 4;		
		break;
	    case 'S':
		write_buf (buf, "short");
		r = 5;
		break;
	    case 'I':
		write_buf (buf, "int");
		r = 3;
		break;
	    case 'J':
		write_buf (buf, "long");
		r = 4;
		break;
	    case 'F':
		write_buf (buf, "float");
		r = 5;
		break;
	    case 'D':
		write_buf (buf, "double");
		r = 6;
		break;
	    }
	} else {
	    memcpy (buf, cls, l + 1);
	    for (r = 0; r < l + 1; r++) {
		if (buf[r] == '/')
		    buf[r] = '.';
	    }
	    r = l;
	}
	return r;
    }
}

typedef struct {
    size_t used_in;
    size_t used_out;
} mf;

/** Translate one field from the nane buffer.
 * @return how many chars were read from name and how many bytes were used in buf.
 */
static mf translate_method_field (const char* name, char* buf) {
    mf out;
    mf t;
    switch (*name) {
    case 'L':
	name++;
	out.used_in = 1;
	out.used_out = 0;
	while (*name != ';') {
	    *buf = *name++;
	    if (*buf == '/')
		*buf = '.';
	    buf++;
	    out.used_in++;
	    out.used_out++;
	}
	out.used_in++;   /* the ';' is also used. */
	break;
    case 'Z':
	write_buf (buf, "boolean");
	out.used_out = 7;
	out.used_in = 1;
	break;
    case 'B':
	write_buf (buf, "byte");
	out.used_out = 4;
	out.used_in = 1;
	break;
    case 'C':
	write_buf (buf, "char");
	out.used_out = 4;		
	out.used_in = 1;
	break;
    case 'S':
	write_buf (buf, "short");
	out.used_out = 5;
	out.used_in = 1;
	break;
    case 'I':
	write_buf (buf, "int");
	out.used_out = 3;
	out.used_in = 1;
	break;
    case 'J':
	write_buf (buf, "long");
	out.used_out = 4;
	out.used_in = 1;
	break;
    case 'F':
	write_buf (buf, "float");
	out.used_out = 5;
	out.used_in = 1;
	break;
    case 'D':
	write_buf (buf, "double");
	out.used_out = 6;
	out.used_in = 1;
	break;
    case 'V':
	write_buf (buf, "void");
	out.used_out = 4;
	out.used_in = 1;
	break;
    case '[':
	t = translate_method_field (name + 1, buf);
	write_buf (buf + t.used_out, "[]");
	out.used_out = t.used_out + 2;
	out.used_in = t.used_in + 1;
	break;
    default:
	/* ok, this should not happen, but if the 
	 * jvm is evil (or a bad simulator..) 
	 */
	out.used_out = 0;
	out.used_in = 0;
    }
    
    return out;
}

/** 
 * @param name the name of the method to translate
 * @param signature the signature of the method we want to translate.
 * @param buf the output buffer.
 * @return the length of the converted string.
 */
int translate_method (const char* name, const char* signature, char* buf) {
    mf p;
    size_t l;
    const char* c = strchr (signature, ')');
    int first = 1;
    *buf = '\0';
    if (c == NULL) {
	fprintf (stderr, "could not find ')' in method signature...\n");
	return 0;
    }
    
    p = translate_method_field (++c, buf);
    buf += p.used_out;
    *buf++ = ' ';
    l = strlen (name);
    memcpy (buf, name, l + 1);    
    buf += l;
    *buf++ = ' ';
    *buf++ = '(';
    
    c = signature + 1; 
    while (*c != ')') {
	if (!first) {
	    *buf++ = ',';
	    *buf++ = ' ';
	}
	first = 0;
	p = translate_method_field (c, buf);
	c += p.used_in;
	buf += p.used_out;
    }
    
    *buf++ = ')';
    *buf = '\0';
    return 0;
}



/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
