#!/usr/bin/env python

from datetime import *
from dateutil.tz import *
from dateutil.relativedelta import *

# Some tests for creating timezones given the information in the Palm timezone database

class palm_dst:
    def __init__(self, dayOfWeek, weekOfMonth, month):
        '''
        dayOfWeek - 0 = Sunday
        weekOfMonth - 0 = first, 1 = second, 2 = third, 3 = fourth, 4 = last
        month - 1 = january, ..., 12 = december
        '''
        self.dayOfWeek = dayOfWeek
        self.weekOfMonth = weekOfMonth
        self.month = month
        
class palm_timezone:
    def __init__(self, offset, dstStart, dstEnd, dstObserved, name):
        '''
        offset - offset East GMT in minutes
        dstStart - palm_dst object for when dst starts, may be None
        dstEnd - palm_dst object for when dst ends, may be None
        dstObserved - boolean if dst is observed (dstStart and dstEnd cannot be None is this is True)
        name - the name of the timezone, may be None
        '''
        self.offset = offset
        self.dstStart = dstStart
        self.dstEnd = dstEnd
        self.dstObserved = dstObserved
        self.name = name

def to_tzinfo(palm_tz):
    '''
    Convert a palm_timezone object into a tzinfo object
    '''

    # need to convert offset to HH:mm
    local_offset = palm_tz.offset
    # note that when reading the TZ environment variable, we need to
    # specify the amount to be ADDED to the time to get to UTC
    negative = "-"
    if palm_tz.offset < 0:
        negative = "+"
        local_offset = -1 * palm_tz.offset
    offset_str = ("%s%d:%d" %
                  (negative,
                  int(local_offset/60),
                  local_offset % 60))
    
    if palm_tz.dstObserved:
        dstStart_str = "M%d.%d.%d/2" % (palm_tz.dstStart.month,
                                      palm_tz.dstStart.weekOfMonth+1,
                                      palm_tz.dstStart.dayOfWeek)
        dstEnd_str = "M%d.%d.%d/2" % (palm_tz.dstEnd.month,
                                      palm_tz.dstEnd.weekOfMonth+1,
                                      palm_tz.dstEnd.dayOfWeek)

        
        #return tzstr("%s%s%s,%s%s" %
        #             (palm_tz.name,
        #              offset_str,
        #              palm_tz.name + "DST",
        #              dstStart_str,
        #              dstEnd_str));
        start_day = 0
        if palm_tz.dstStart.dayOfWeek == 0:
            start_day = SU(+(palm_tz.dstStart.weekOfMonth+2))
        elif palm_tz.dstStart.dayOfWeek == 1:
            start_day = MO(+(palm_tz.dstStart.weekOfMonth+2))
        elif palm_tz.dstStart.dayOfWeek == 2:
            start_day = TU(+(palm_tz.dstStart.weekOfMonth+2))
        elif palm_tz.dstStart.dayOfWeek == 3:
            start_day = WE(+(palm_tz.dstStart.weekOfMonth+2))
        elif palm_tz.dstStart.dayOfWeek == 4:
            start_day = TH(+(palm_tz.dstStart.weekOfMonth+2))
        elif palm_tz.dstStart.dayOfWeek == 5:
            start_day = FR(+(palm_tz.dstStart.weekOfMonth+2))
        elif palm_tz.dstStart.dayOfWeek == 6:
            start_day = SA(+(palm_tz.dstStart.weekOfMonth+2))
        else:
            print("Error!")

        end_day = 0
        if palm_tz.dstEnd.dayOfWeek == 0:
            end_day = SU(+(palm_tz.dstEnd.weekOfMonth+2))
        elif palm_tz.dstEnd.dayOfWeek == 1:
            end_day = MO(+(palm_tz.dstEnd.weekOfMonth+2))
        elif palm_tz.dstEnd.dayOfWeek == 2:
            end_day = TU(+(palm_tz.dstEnd.weekOfMonth+2))
        elif palm_tz.dstEnd.dayOfWeek == 3:
            end_day = WE(+(palm_tz.dstEnd.weekOfMonth+2))
        elif palm_tz.dstEnd.dayOfWeek == 4:
            end_day = TH(+(palm_tz.dstEnd.weekOfMonth+2))
        elif palm_tz.dstEnd.dayOfWeek == 5:
            end_day = FR(+(palm_tz.dstEnd.weekOfMonth+2))
        elif palm_tz.dstEnd.dayOfWeek == 6:
            end_day = SA(+(palm_tz.dstEnd.weekOfMonth+2))
        else:
            print("Error!")
        
        return tzrange(palm_tz.name,
                       palm_tz.offset*60,
                       palm_tz.name+"DST",
                       (palm_tz.offset+60)*60,
                       relativedelta(hours=+2,
                                     month=palm_tz.dstStart.month,
                                     day=1,
                                     weekday=start_day),
                       relativedelta(hours=+1,
                                     month=palm_tz.dstEnd.month,
                                     day=1,
                                     weekday=end_day))


    else:
        #return tzstr("%s%s" %
        #             (palm_tz.name,
        #             offset_str))
        return tzrange(palm_tz.name, palm_tz.offset*60)

def datetimeToUTC(dt, palm_timezone):
    '''
    @brief Convert a datetime object to UTC.

    If the datetime object has a timezone, use that reference to convert to
    UTC. If the tzinfo is not specified, then use the specified palm
    timezone. If the palm timezone is NULL, use tzlocal() to get the timezone
    for the specified datetime.

    @param pydatetime a python datetime object that needs to be converted to UTC

    @param palm_tz the palm timezone, may be NULL

    @return a python datetime object in UTC with tzinfo set to tzutc(), NULL if
    pydatetime is NULL
    '''

    if dt == None:
        return None
    
    if dt.tzinfo != None:
        # use timezone in datetime object
        return dt.astimezone(tzutc())
    elif palm_timezone == None:
        # use local timezone
        local_dt = dt.replace(tzinfo=tzlocal())
        return local_dt.astimezone(tzutc())
    else:
        # use palm timezone
        local_dt = dt.replace(tzinfo=to_tzinfo(palm_timezone))
        return local_dt.astimezone(tzutc())
    

def main():
    # Chicago
    tz_chicago = palm_timezone(-360, palm_dst(0, 1, 3), palm_dst(0, 0, 11), True, "Chicago")
    tz_san_fran = palm_timezone(-480, palm_dst(0, 0, 4), palm_dst(0, 4, 10), True, "San Francisco")

    tzinfo_chicago = to_tzinfo(tz_chicago)
    print tzinfo_chicago
    tzinfo_san_fran = to_tzinfo(tz_san_fran)

    local_now = datetime.now(tzlocal())
    naive_now = datetime.now()
    utc_now = datetime.now(tzutc())
    #utc_now = datetime(2008, 11, 2, 20, 56, tzinfo=tzutc())
    
    print ("local: %s, naive: %s, utc: %s, chicago: %s, san fran: %s"
           % (local_now.strftime('%X %x %Z'),
              naive_now.strftime('%X %x %Z'),
              utc_now.strftime('%X %x %Z'),
              utc_now.astimezone(tzinfo_chicago).strftime('%X %x %Z'),
              utc_now.astimezone(tzinfo_san_fran).strftime('%X %x %Z')))
    

    etz = gettz("Europe/London")
    localized = naive_now.replace(tzinfo=etz)
    print "naive: %s, localized: %s" % (naive_now.strftime('%X %x %Z'),
                                         localized.strftime('%X %x %Z'))

    print "Without palm timezone" 
    print "naive->utc: %s" % (datetimeToUTC(naive_now, None).strftime('%X %x %Z'))
    print "local->utc: %s" % (datetimeToUTC(local_now, None).strftime('%X %x %Z'))
    print "utc->utc: %s" % (datetimeToUTC(utc_now, None).strftime('%X %x %Z'))

    print "With palm timezone (San Fran)" 
    print "naive->utc: %s" % (datetimeToUTC(naive_now, tz_san_fran).strftime('%X %x %Z'))
    print "local->utc: %s" % (datetimeToUTC(local_now, tz_san_fran).strftime('%X %x %Z'))
    print "utc->utc: %s" % (datetimeToUTC(utc_now, tz_san_fran).strftime('%X %x %Z'))

if __name__ == "__main__":
    main()
    
