/* jslaunch 2.0: a joystick reset daemon
   Copyright (C) Sander Pronk, 1998
   
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>

#include <unistd.h>

#include <asm/io.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <time.h>

#ifdef DEBUG
#include <stdio.h>
#endif  

#include "jslaunch.h"

int  ioperm(unsigned  long,  unsigned  long, int);
/* I've supplied the prototype manually, because different
   distributions seem to have this at different places */

int main(int argc, char *argv[])
{
    int i;
    
    char *msk[NMASK]; /* the strings with executables */
    
    int joymask=MASKALL; /* bit mask for the joystick port */
    
    int port; /* data read in from port */
    char *cmd; /* command to be executed */

    int polltime=DEFAULTPOLL; /* polling time */
    int bounce=BOUNCETIME; /* wait time for actual read */

    int dmn=FALSE; /* whether to be a daemon */

    char *lockfile=NULL; /* path for lock file */

    int ret; /* return code for readargs */
    pid_t pid;  /* pid as returned by fork */

#ifdef DEBUG
    printf("%s\n%d\n",EXPN,strlen(EXPN));
    /* I did not use printf in the standard program because
       it introduces a memory overhead of a couple of pages. 
       That's why the strings include string length etc. */
#endif /* DEBUG */

    for(i=0;i<NMASK;i++) /* reset the masks */
	msk[i]=NULL;

    ret=readargs(argc, argv, msk, &polltime, &bounce, &joymask, &dmn, 
		 &lockfile);
    if (ret) /* if everything did not go as planned while reading
	      the arguments */
    {
	if (ret==2)
	    write(STDERR_FILENO,EXPN,EXPNLEN);
	exit(EXIT_FAILURE);
    }

    if (dmn) /* if we want to become a daemon */
    {
	/* become a daemon */
	pid=fork(); 
	if (pid>0)
	    exit(EXIT_SUCCESS); /* fork was succesful */
	if (pid<0)
	{
	    write(STDERR_FILENO,NOFORK,NOFORKLEN); 
	    exit(EXIT_FAILURE);
	}
	/* if pid==0 this is the child process */
    }

    if (ioperm(PORT,PORTLEN,PERMLEVEL)) 
	/* get permission for joystick port */
    {
	write(STDERR_FILENO,NOPERM,NOPERMLEN);
	exit(EXIT_FAILURE);
    }

    if ( setuid(getuid()) || setgid(getgid()) )
	/* lose our privileges as fast as we can. 
	   It's best NOT to use sudo with jslaunch, but to make
	   it a suid executable, because with sudo there may be 
	   a way to execute arbitrary 
	   commands as root for ordinary users */
    {
	write(STDERR_FILENO,UIDFAIL,UIDFAILLEN);
	exit(EXIT_FAILURE);
    }

    port=(~inb(PORT)&joymask) >> SHIFTVAL;
    if ( msk[port] )
	 /* test whether current mask triggers something */
    {
	write(STDERR_FILENO,ACTPRESSED,ACTPRESSEDLEN);
	exit(EXIT_FAILURE);
    }

    if (dmn)
    {
	setsid(); /* lose our term */
	chdir("/"); /* lose dependency on mounted fs */
    }

    if (lockfile) /* after we've given up root perms,
		     we can now safely set up a lock file */
    {
	setlock(lockfile);
    }

    if (dmn)
    {
	close(STDIN_FILENO);
	close(STDOUT_FILENO);
	close(STDERR_FILENO);	/* close all term dependecies,
				 the last error messages.
				 After this no more error messages
				 can be printed */
    }


    while(1) /* the actual loop */
    {
	usleep(polltime); /* wait... */

	port=inb(PORT)&joymask; /* get input from joystick interface */

	if ( (port)!=joymask ) /* whether one of the buttons is
					  pressed */
	{
	    usleep(bounce);
	    /* wait for delta E*delta t>= h/2 
	     (I'm writing this just before a quantum physics exam :-) */
	    port= ((~inb(PORT))&joymask) >> SHIFTVAL;
	    /* we do the negation because the joystick gives the
	       inverse value of a button press (ie. button NOT pressed = 1)*/

	    cmd=msk[port];
	    if (cmd) /* whether we've hit one of the 
			masks */
	    {
		system(cmd); /* do the thing... */
		while((inb(PORT) & MASKALL)!=MASKALL) /* wait for release */
		    usleep(polltime);  /* we don't want 100% cpu time
					  consumed by this program */
	    }
	}
    }

    return EXIT_SUCCESS;
}

int readargs(int argc, char *argv[], char *msk[], int *polltime,
	     int *bounce, int *joymask, int *dmn, char **lockfile)
{
    int i=1; /* arg counter, we start at argv[1]; */
    int cmds=0; /* number of -r options */
    int newmask; /* a new joystick mask, for -r etc. */

    if (argc==1) /* no arguments, return immediately */
	return 2;

    while ( i<argc )
    {
	if (argv[i][0]!='-') /* check whether all arguments start 
				with '-' */
	    return 2;

	switch (argv[i][1])
	{
	    case 'p': /* poll time */
		if (!argv[++i])
		    goto fwopt;
		if ( (*polltime=readtime(argv[i])) < 0 )
		    return 1;
		break;
	    case 'w': /* wait time */
		if (!argv[++i])
		    goto fwopt;
		if ( (*bounce=readtime(argv[i])) < 0 )
		    return 1;
		break;
	    case 'r': /* run */
		if (!argv[i+2])
		    goto fwopt;
		newmask=readmask(argv[++i]);
		if (!newmask)
		    return 1;
		if (msk[newmask]) /* whether action is already 
				     occupied */
		{
		    write(STDERR_FILENO,ACTTWICE,ACTTWICELEN);
		    return 1;
		}
		msk[newmask]=argv[++i]; /* set new command */
		cmds++;
		break;
	    case 'i': /* ignore */
		if (argc <= i+1)
		    goto fwopt;
		newmask=readmask(argv[++i]);
		if (!newmask)
		    return 1;
		*joymask= ((~newmask)&MASKMASK) << SHIFTVAL;
		break;
	    case 'd': /* daemon */
		*dmn=TRUE;
		break;
	    case 'l': /* lock */
		if (!argv[++i])
		    goto fwopt;
		*lockfile=argv[i];
		break;
	    default: /* wrong arg type */
		return 2;
	}
	i++;
    }

    if (!cmds)
    {
	write(STDERR_FILENO,NOCOMMAND,NOCOMMANDLEN);
	return 1;
    }

    return 0;

 fwopt:
    write(STDERR_FILENO,FEWOPTS,FEWOPTSLEN);
    return 1;
}


int readmask(const char *str)
{
    int curbut; /* current new button  */
    int curmask=0; /* current mask */

    while (*str)
    {
	curbut= (*str)-'1';
	if ( (curbut<0) || (curbut>3) )
	{
	    write(STDERR_FILENO,WRONGBUT,WRONGBUTLEN);
	    return 0;
	}
	if ( (curmask >> curbut) & 1 )
	{
	    write(STDERR_FILENO,BUTTWICE,BUTTWICELEN);
	    return 0;
	}
	curmask |= 1 << curbut;
	str++;
    }

    curmask &= MASKMASK;
    return curmask;
}

int readtime(const char *st)

{
    int t_read;

    t_read=atoi(st);
    
    if  ( (t_read<=0) || (t_read>POLLTIME_MAX) )
    {
	write(STDERR_FILENO,TIMEERR,TIMEERRLEN);
	return -1;
    }
    return t_read*MICRO_IN_MILLI;
}

void setlock(char *lockfile)
{
    int lock_fd;
    struct flock lck;

    lock_fd=open(lockfile, O_RDWR | O_CREAT, 
		 S_IRUSR | S_IWUSR | S_IRGRP |
		 S_IROTH); /* open file */
    if (lock_fd<0)
    {
	write(STDERR_FILENO,OPENFAIL,OPENFAILLEN);
	exit(EXIT_FAILURE);
    }
    lck.l_type=F_WRLCK; /* exclusive write lock */
    lck.l_start=0;
    lck.l_whence=SEEK_SET;
    lck.l_len=0; /* lock the whole file */
    lck.l_pid=getpid(); /* set our pid, so if another proc is reading 
			   the lock status, it gets our pid */
    if (fcntl(lock_fd,F_SETLK,&lck)) /* set lock */
    {
	write(STDERR_FILENO,LOCKFAIL,LOCKFAILLEN);
	exit(EXIT_FAILURE);
    }
}


