import logging

import zookeeper

from twisted.internet.defer import (
    Deferred, inlineCallbacks, fail, returnValue, succeed)
from twisted.internet import reactor

from txzookeeper.client import ZOO_OPEN_ACL_UNSAFE

from juju.agents.provision import ProvisioningAgent
from juju.environment.environment import Environment
from juju.environment.config import EnvironmentsConfig
from juju.environment.errors import EnvironmentsConfigError
from juju.environment.tests.test_config import SAMPLE_ENV

from juju.errors import ProviderInteractionError
from juju.lib.mocker import MATCH
from juju.providers.dummy import DummyMachine
from juju.state.errors import StopWatcher
from juju.state.machine import MachineStateManager
from juju.state.tests.test_service import ServiceStateManagerTestBase

from .common import AgentTestBase

MATCH_MACHINE = MATCH(lambda x: isinstance(x, DummyMachine))


class ProvisioningTestBase(AgentTestBase):

    agent_class = ProvisioningAgent

    def get_serialized_environment(self):
        config = EnvironmentsConfig()
        config.parse(SAMPLE_ENV)
        return config.serialize("myfirstenv")


class ProvisioningAgentStartupTest(ProvisioningTestBase):

    @inlineCallbacks
    def setUp(self):
        yield super(ProvisioningAgentStartupTest, self).setUp()
        yield self.agent.connect()

    def test_agent_waits_for_environment(self):
        """
        When the agent starts it waits for the /environment node to exist.
        As soon as it does, the agent will fetch the environment, and
        deserialize it into an environment object.
        """
        env_loaded_deferred = self.agent.configure_environment()

        def verify_environment(result):
            self.assertTrue(isinstance(result, Environment))
            self.assertEqual(result.name, "myfirstenv")

        env_loaded_deferred.addCallback(verify_environment)

        def create_environment_node():
            self.assertFalse(env_loaded_deferred.called)
            return self.client.create(
                "/environment", self.get_serialized_environment())

        reactor.callLater(0.3, create_environment_node)
        return env_loaded_deferred

    @inlineCallbacks
    def test_agent_with_existing_environment(self):
        """An agent should load an existing environment to configure itself."""

        yield self.client.create(
            "/environment", self.get_serialized_environment())

        def verify_environment(result):
            self.assertTrue(isinstance(result, Environment))
            self.assertEqual(result.name, "myfirstenv")

        d = self.agent.configure_environment()
        d.addCallback(verify_environment)
        yield d

    @inlineCallbacks
    def test_agent_with_invalid_environment(self):
        yield self.client.create("/environment", "WAHOO!")
        d = self.agent.configure_environment()
        yield self.assertFailure(d, EnvironmentsConfigError)

    def test_agent_with_nonexistent_environment_created_concurrently(self):
        """
        If the environment node does not initially exist but it is created
        while the agent is processing the NoNodeException, it should detect
        this and configure normally.
        """
        data = self.get_serialized_environment()
        exists_and_watch = self.agent.client.exists_and_watch

        mock_client = self.mocker.patch(self.agent.client)
        mock_client.exists_and_watch("/environment")

        def inject_creation(path):
            zookeeper.create(
                self.agent.client.handle, path, data, [ZOO_OPEN_ACL_UNSAFE])
            return exists_and_watch(path)

        self.mocker.call(inject_creation)
        self.mocker.replay()

        def verify_configured(result):
            self.assertTrue(isinstance(result, Environment))
            self.assertEqual(result.type, "dummy")
        # mocker magic test
        d = self.agent.configure_environment()
        d.addCallback(verify_configured)
        return d


class ProvisioningAgentTest(ProvisioningTestBase):

    @inlineCallbacks
    def setUp(self):
        yield super(ProvisioningAgentTest, self).setUp()
        yield self.client.create(
            "/environment", self.get_serialized_environment())
        self.agent.set_watch_enabled(False)
        yield self.agent.startService()
        self.output = self.capture_logging("juju.agents.provision",
                                           logging.DEBUG)

    def test_get_agent_name(self):
        self.assertEqual(self.agent.get_agent_name(), "provision:dummy")

    @inlineCallbacks
    def test_watch_machine_changes_processes_new_machine_id(self):
        """The agent should process a new machine id by creating it"""
        manager = MachineStateManager(self.client)
        machine_state0 = yield manager.add_machine_state()
        machine_state1 = yield manager.add_machine_state()

        yield self.agent.watch_machine_changes(
            None, [machine_state0.id, machine_state1.id])

        self.assertIn(
            "Machines changed old:None new:[0, 1]", self.output.getvalue())
        self.assertIn("Starting machine id:0", self.output.getvalue())

        machines = yield self.agent.provider.get_machines()
        self.assertEquals(len(machines), 2)

        instance_id = yield machine_state0.get_instance_id()
        self.assertEqual(instance_id, 0)

        instance_id = yield machine_state1.get_instance_id()
        self.assertEqual(instance_id, 1)

    @inlineCallbacks
    def test_watch_machine_changes_ignores_running_machine(self):
        """
        If there is an existing machine instance and state, when a
        new machine state is added, the existing instance is preserved,
        and a new instance is created.
        """
        manager = MachineStateManager(self.client)
        machine_state0 = yield manager.add_machine_state()
        machines = yield self.agent.provider.start_machine(
            {"machine-id": machine_state0.id})
        machine = machines.pop()
        yield machine_state0.set_instance_id(machine.instance_id)

        machine_state1 = yield manager.add_machine_state()

        machines = yield self.agent.provider.get_machines()
        self.assertEquals(len(machines), 1)

        yield self.agent.watch_machine_changes(
            None, [machine_state0.id, machine_state1.id])

        machines = yield self.agent.provider.get_machines()
        self.assertEquals(len(machines), 2)

        instance_id = yield machine_state1.get_instance_id()
        self.assertEqual(instance_id, 1)

    @inlineCallbacks
    def test_watch_machine_changes_terminates_unused(self):
        """
        Any running provider machine instances without corresponding
        machine states are terminated.
        """
        # start an unused machine within the dummy provider instance
        yield self.agent.provider.start_machine({"machine-id": "machine-1"})
        yield self.agent.watch_machine_changes(None, [])
        self.assertIn("Shutting down machine id:0", self.output.getvalue())
        machines = yield self.agent.provider.get_machines()
        self.assertFalse(machines)

    @inlineCallbacks
    def test_watch_machine_changes_stop_watches(self):
        """Verify that the watches stops once the agent stops."""
        yield self.agent.start()
        yield self.agent.stop()
        yield self.assertFailure(
            self.agent.watch_machine_changes(None, []),
            StopWatcher)

    @inlineCallbacks
    def test_new_machine_state_removed_while_processing(self):
        """
        If the machine state is removed while the event is processing the
        state, the watch function should process it normally.
        """
        yield self.agent.watch_machine_changes(
            None, [0])
        machines = yield self.agent.provider.get_machines()
        self.assertEquals(len(machines), 0)

    @inlineCallbacks
    def test_process_machines_non_concurrency(self):
        """
        Process machines should only be executed serially by an
        agent.
        """
        manager = MachineStateManager(self.client)
        machine_state0 = yield manager.add_machine_state()
        machine_state1 = yield manager.add_machine_state()

        call_1 = self.agent.process_machines([machine_state0.id])

        # The second call should return immediately due to the
        # instance attribute guard.
        call_2 = self.agent.process_machines([machine_state1.id])
        self.assertEqual(call_2.called, True)
        self.assertEqual(call_2.result, False)

        # The first call should have started a provider machine
        yield call_1

        machines = yield self.agent.provider.get_machines()
        self.assertEquals(len(machines), 1)

        instance_id_0 = yield machine_state0.get_instance_id()
        self.assertEqual(instance_id_0, 0)

        instance_id_1 = yield machine_state1.get_instance_id()
        self.assertEqual(instance_id_1, None)

    def test_new_machine_state_removed_while_processing_get_provider_id(self):
        """
        If the machine state is removed while the event is processing the
        state, the watch function should process it normally.
        """
        yield self.agent.watch_machine_changes(
            None, [0])
        machines = yield self.agent.provider.get_machines()
        self.assertEquals(len(machines), 0)

    @inlineCallbacks
    def test_on_environment_change_agent_reconfigures(self):
        """
        If the environment changes the agent reconfigures itself
        """
        provider = self.agent.provider
        data = self.get_serialized_environment()
        yield self.client.set("/environment", data)
        yield self.sleep(0.2)
        self.assertNotIdentical(provider, self.agent.provider)

    @inlineCallbacks
    def test_machine_state_reflects_invalid_provider_state(self):
        """
        If a machine state has an invalid instance_id, it should be detected,
        and a new machine started and the machine state updated with the
        new instance_id.
        """
        machine_manager = MachineStateManager(self.client)
        m1 = yield machine_manager.add_machine_state()
        yield m1.set_instance_id("zebra")

        m2 = yield machine_manager.add_machine_state()
        yield self.agent.watch_machine_changes(None, [m1.id, m2.id])

        m1_instance_id = yield m1.get_instance_id()
        self.assertEqual(m1_instance_id, 0)

        m2_instance_id = yield m2.get_instance_id()
        self.assertEqual(m2_instance_id, 1)

    def test_periodic_task(self):
        """
        The agent schedules period checks that execute the process machines
        call.
        """
        mock_reactor = self.mocker.patch(reactor)
        mock_reactor.callLater(self.agent.machine_check_period,
                               self.agent.periodic_machine_check)
        mock_agent = self.mocker.patch(self.agent)
        mock_agent.process_machines(())
        self.mocker.result(succeed(None))
        self.mocker.replay()

        # mocker magic test
        self.agent.periodic_machine_check()

    @inlineCallbacks
    def test_transient_provider_error_on_start_machine(self):
        """
        If there's an error when processing changes, the agent should log
        the error and continue.
        """
        manager = MachineStateManager(self.client)
        machine_state0 = yield manager.add_machine_state()
        machine_state1 = yield manager.add_machine_state()

        mock_provider = self.mocker.patch(self.agent.provider)
        mock_provider.start_machine({"machine-id": 0})
        self.mocker.result(fail(ProviderInteractionError()))

        mock_provider.start_machine({"machine-id": 1})
        self.mocker.passthrough()
        self.mocker.replay()

        yield self.agent.watch_machine_changes(
            [], [machine_state0.id, machine_state1.id])

        machine1_instance_id = yield machine_state1.get_instance_id()
        self.assertEqual(machine1_instance_id, 0)
        self.assertIn(
            "Cannot process machine 0",
            self.output.getvalue())

    @inlineCallbacks
    def test_transient_provider_error_on_shutdown_machine(self):
        """
        A transient provider error on shutdown will be ignored
        and the shutdown will be reattempted (assuming similiar
        state conditions) on the next execution of process machines.
        """
        yield self.agent.provider.start_machine({"machine-id": 1})
        mock_provider = self.mocker.patch(self.agent.provider)

        mock_provider.shutdown_machine(MATCH_MACHINE)
        self.mocker.result(fail(ProviderInteractionError()))

        mock_provider.shutdown_machine(MATCH_MACHINE)
        self.mocker.passthrough()

        self.mocker.replay()
        try:
            yield self.agent.process_machines([])
        except:
            self.fail("Should not raise")

        machines = yield self.agent.provider.get_machines()
        self.assertTrue(machines)

        yield self.agent.process_machines([])
        machines = yield self.agent.provider.get_machines()
        self.assertFalse(machines)

        self.assertIn(
            "Cannot shutdown machine 0",
            self.output.getvalue())

    @inlineCallbacks
    def test_transient_provider_error_on_get_machines(self):
        manager = MachineStateManager(self.client)
        machine_state0 = yield manager.add_machine_state()

        mock_provider = self.mocker.patch(self.agent.provider)
        mock_provider.get_machines()
        self.mocker.result(fail(ProviderInteractionError()))

        mock_provider.get_machines()
        self.mocker.passthrough()

        self.mocker.replay()
        try:
            yield self.agent.process_machines([machine_state0.id])
        except:
            self.fail("Should not raise")

        instance_id = yield machine_state0.get_instance_id()
        self.assertEqual(instance_id, None)

        yield self.agent.process_machines(
            [machine_state0.id])

        instance_id = yield machine_state0.get_instance_id()
        self.assertEqual(instance_id, 0)
        self.assertIn(
            "Cannot get machine list",
            self.output.getvalue())

    @inlineCallbacks
    def test_start_agent_with_watch(self):
        mock_reactor = self.mocker.patch(reactor)
        mock_reactor.callLater(
            self.agent.machine_check_period,
            self.agent.periodic_machine_check)
        self.mocker.replay()

        self.agent.set_watch_enabled(True)
        yield self.agent.start()

        manager = MachineStateManager(self.client)
        machine_state0 = yield manager.add_machine_state()

        # the watch invocation happens out of band, sleep for
        # a moment so it has a chance to run.
        yield self.sleep(0.1)
        instance_id = yield machine_state0.get_instance_id()
        self.assertEqual(instance_id, 0)


class ExposeTestBase(
        ProvisioningTestBase, ServiceStateManagerTestBase):

    @inlineCallbacks
    def setUp(self):
        yield super(ExposeTestBase, self).setUp()
        yield self.client.create(
            "/environment", self.get_serialized_environment())
        self.agent.set_watch_enabled(False)
        yield self.agent.startService()
        self.output = self.capture_logging(level=logging.DEBUG)

    @inlineCallbacks
    def start_agent_with_watch_support(self, enabled=True):
        if enabled:
            mock_reactor = self.mocker.patch(reactor)
            mock_reactor.callLater(
                self.agent.machine_check_period,
                self.agent.periodic_machine_check)
            self.mocker.replay()
        self.agent.set_watch_enabled(enabled)
        yield self.agent.start()

    def wait_on_expected_units(self, expected):
        """Returns deferred for waiting on `expected` unit names.

        These unit names may require the firewall to have ports opened
        and/or closed.
        """
        condition_met = Deferred()
        seen = set()

        def observer(unit_state):
            unit_name = unit_state.unit_name
            seen.add(unit_name)
            if seen >= expected:
                # Call the callback just once, since it is possible
                # for this condition to be satisfied multiple times in
                # using tests because of background activity
                if not condition_met.called:
                    condition_met.callback(True)
            return succeed(True)

        self.agent.set_open_close_ports_observer(observer)
        return condition_met

    def wait_on_expected_machines(self, expected):
        """Returns deferred for waiting on `expected` machine IDs.

        These machines may require the firewall to have ports opened
        and/or closed.
        """
        condition_met = Deferred()
        seen = set()

        def observer(machine_id):
            seen.add(machine_id)
            if seen >= expected:
                # Call the callback just once, since it is possible
                # for this condition to be satisfied multiple times in
                # using tests because of background activity
                if not condition_met.called:
                    condition_met.callback(True)
            return succeed(True)

        self.agent.set_open_close_ports_on_machine_observer(observer)
        return condition_met


class ExposeServiceTest(ExposeTestBase):

    @inlineCallbacks
    def test_service_exposed_flag_changes(self):
        """Verify that a service unit is checked whenever a change
        occurs such that ports may need to be opened and/or closed
        for the machine corresponding to a given service unit.
        """
        # Start the provisioning agent such that watches will run
        yield self.start_agent_with_watch_support()

        # Ensure that an existing service unit for a newly exposed
        # service triggers firewall mgmt in the provisioing agent
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0"]))
        wordpress = yield self.add_service("wordpress")
        yield wordpress.add_unit_state()
        yield wordpress.set_exposed_flag()
        self.assertTrue((yield expected_units))

        # Then clear the flag, see that it triggers on the expected units
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0"]))
        yield wordpress.clear_exposed_flag()
        self.assertTrue((yield expected_units))

        # Re-expose wordpress: set the flag again, verify that it
        # triggers on the expected units
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0"]))
        yield wordpress.set_exposed_flag()
        self.assertTrue((yield expected_units))
        yield self.agent.stop()

    @inlineCallbacks
    def test_add_remove_service_units_for_exposed_service(self):
        """Verify that adding/removing service units for an exposed
        service triggers the appropriate firewall management of
        opening/closing ports on the machines for the corresponding
        service units.
        """
        yield self.start_agent_with_watch_support()
        wordpress = yield self.add_service("wordpress")
        yield wordpress.set_exposed_flag()

        # Adding service units to this exposed service will trigger
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0", "wordpress/1"]))
        wordpress_0 = yield wordpress.add_unit_state()
        yield wordpress.add_unit_state()
        self.assertTrue((yield expected_units))

        # Removing service units will also trigger
        expected_units = self.wait_on_expected_units(
            set(["wordpress/2"]))
        yield wordpress.remove_unit_state(wordpress_0)
        yield wordpress.add_unit_state()
        self.assertTrue((yield expected_units))
        yield self.agent.stop()

    @inlineCallbacks
    def test_open_close_ports(self):
        """Verify that opening/closing ports triggers the appropriate
        firewall management for the corresponding service units.
        """
        yield self.start_agent_with_watch_support()
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0"]))
        wordpress = yield self.add_service("wordpress")
        yield wordpress.set_exposed_flag()
        wordpress_0 = yield wordpress.add_unit_state()
        wordpress_1 = yield wordpress.add_unit_state()
        yield wordpress.add_unit_state()
        yield wordpress_0.open_port(443, "tcp")
        yield wordpress_0.open_port(80, "tcp")
        yield wordpress_0.close_port(443, "tcp")
        self.assertTrue((yield expected_units))

        expected_units = self.wait_on_expected_units(
            set(["wordpress/1", "wordpress/3"]))
        wordpress_3 = yield wordpress.add_unit_state()
        yield wordpress_1.open_port(53, "udp")
        yield wordpress_3.open_port(80, "tcp")
        self.assertTrue((yield expected_units))

        expected_units = self.wait_on_expected_units(
            set(["wordpress/0", "wordpress/1", "wordpress/3"]))
        yield wordpress.clear_exposed_flag()
        self.assertTrue((yield expected_units))
        yield self.agent.stop()

    @inlineCallbacks
    def test_remove_service_state(self):
        """Verify that firewall mgmt for corresponding service units
        is triggered upon the service's removal.
        """
        yield self.start_agent_with_watch_support()
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0", "wordpress/1"]))
        wordpress = yield self.add_service("wordpress")
        yield wordpress.add_unit_state()
        yield wordpress.add_unit_state()
        yield wordpress.set_exposed_flag()
        self.assertTrue((yield expected_units))

        # Do not clear the exposed flag prior to removal, triggering
        # should still occur as expected
        yield self.service_state_manager.remove_service_state(wordpress)
        yield self.agent.stop()

    @inlineCallbacks
    def test_port_mgmt_for_unexposed_service_is_a_nop(self):
        """Verify that activity on an unexposed service does NOT
        trigger firewall mgmt for the corresponding service unit."""
        expected_units = self.wait_on_expected_units(
            set(["not-called"]))
        wordpress = yield self.add_service("wordpress")
        wordpress_0 = yield wordpress.add_unit_state()
        yield wordpress_0.open_port(53, "tcp")
        # The observer should not be called in this case
        self.assertFalse(expected_units.called)
        yield self.agent.stop()

    @inlineCallbacks
    def test_provisioning_agent_restart(self):
        """Verify that firewall management is correct if the agent restarts.

        In particular, this test verifies that all state relevant for
        firewall management is stored in ZK and not in the agent
        itself.
        """
        # Store into ZK relevant state, this might have been observed
        # in a scenario in which the agent has previously been
        # running.
        wordpress = yield self.add_service("wordpress")
        wordpress_0 = yield wordpress.add_unit_state()
        wordpress_1 = yield wordpress.add_unit_state()
        yield wordpress_1.open_port(443, "tcp")
        yield wordpress_1.open_port(80, "tcp")
        yield wordpress.set_exposed_flag()

        # Now simulate agent start
        yield self.start_agent_with_watch_support()

        # Verify the expected service units are observed as needing
        # firewall mgmt
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0", "wordpress/1"]))
        yield wordpress_0.open_port(53, "udp")
        yield wordpress_1.close_port(443, "tcp")
        self.assertTrue((yield expected_units))

        # Also verify that opening/closing ports work as expected
        expected_units = self.wait_on_expected_units(
            set(["wordpress/1"]))
        yield wordpress_1.close_port(80, "tcp")

        expected_units = self.wait_on_expected_units(
            set(["wordpress/0", "wordpress/1"]))
        yield wordpress.clear_exposed_flag()
        self.assertTrue((yield expected_units))
        yield self.agent.stop()


class ExposeMachineTest(ExposeTestBase):

    @inlineCallbacks
    def get_provider_ports(self, machine):
        instance_id = yield machine.get_instance_id()
        machine_provider = yield self.agent.provider.get_machine(instance_id)
        provider_ports = yield self.agent.provider.get_opened_ports(
            machine_provider, machine.id)
        returnValue(provider_ports)

    def test_open_close_ports_on_machine(self):
        """Verify opening/closing ports on a machine works properly.

        In particular this is done without watch support."""
        yield self.start_agent_with_watch_support(enabled=False)
        manager = MachineStateManager(self.client)
        machine = yield manager.add_machine_state()
        yield self.agent.watch_machine_changes(None, [machine.id])

        # Expose a service
        wordpress = yield self.add_service("wordpress")
        yield wordpress.set_exposed_flag()
        wordpress_0 = yield wordpress.add_unit_state()
        yield wordpress_0.open_port(80, "tcp")
        yield wordpress_0.open_port(443, "tcp")
        yield wordpress_0.assign_to_machine(machine)
        yield self.agent.open_close_ports_on_machine(machine.id)
        self.assertEqual((yield self.get_provider_ports(machine)),
                         set([(80, "tcp"), (443, "tcp")]))
        self.assertIn("Opened 80/tcp on provider machine 0",
                      self.output.getvalue())
        self.assertIn("Opened 443/tcp on provider machine 0",
                      self.output.getvalue())

        # Now change port setup
        yield wordpress_0.open_port(8080, "tcp")
        yield wordpress_0.close_port(443, "tcp")
        yield self.agent.open_close_ports_on_machine(machine.id)
        self.assertEqual((yield self.get_provider_ports(machine)),
                         set([(80, "tcp"), (8080, "tcp")]))
        self.assertIn("Opened 8080/tcp on provider machine 0",
                      self.output.getvalue())
        self.assertIn("Closed 443/tcp on provider machine 0",
                      self.output.getvalue())

    @inlineCallbacks
    def test_open_close_ports_on_unassigned_machine(self):
        """Verify corner case that nothing happens on an unassigned machine."""
        yield self.start_agent_with_watch_support(enabled=False)
        manager = MachineStateManager(self.client)
        machine = yield manager.add_machine_state()
        yield self.agent.watch_machine_changes(None, [machine.id])
        yield self.agent.open_close_ports_on_machine(machine.id)
        self.assertEqual((yield self.get_provider_ports(machine)),
                         set())

    @inlineCallbacks
    def test_open_close_ports_on_machine_unexposed_service(self):
        """Verify opening/closing ports on a machine works properly.

        In particular this is done without watch support."""
        yield self.start_agent_with_watch_support(enabled=False)
        manager = MachineStateManager(self.client)
        machine = yield manager.add_machine_state()
        yield self.agent.watch_machine_changes(None, [machine.id])
        wordpress = yield self.add_service("wordpress")
        wordpress_0 = yield wordpress.add_unit_state()

        # Port activity, but service is not exposed
        yield wordpress_0.open_port(80, "tcp")
        yield wordpress_0.open_port(443, "tcp")
        yield wordpress_0.assign_to_machine(machine)
        yield self.agent.open_close_ports_on_machine(machine.id)
        self.assertEqual((yield self.get_provider_ports(machine)),
                         set())

        # Now expose it
        yield wordpress.set_exposed_flag()
        yield self.agent.open_close_ports_on_machine(machine.id)
        self.assertEqual((yield self.get_provider_ports(machine)),
                         set([(80, "tcp"), (443, "tcp")]))

    @inlineCallbacks
    def test_open_close_ports_on_machine_not_yet_provided(self):
        """Verify that opening/closing ports will eventually succeed
        once a machine is provided.
        """
        yield self.start_agent_with_watch_support(enabled=False)
        manager = MachineStateManager(self.client)
        machine = yield manager.add_machine_state()
        wordpress = yield self.add_service("wordpress")
        yield wordpress.set_exposed_flag()
        wordpress_0 = yield wordpress.add_unit_state()
        yield wordpress_0.open_port(80, "tcp")
        yield wordpress_0.open_port(443, "tcp")
        yield wordpress_0.assign_to_machine(machine)

        # First attempt to open ports quietly fails (except for
        # logging) because the machine has not yet been provisioned
        yield self.agent.open_close_ports_on_machine(machine.id)
        self.assertIn("No provisioned machine for machine 0",
                      self.output.getvalue())

        # Machine is now provisioned (normally visible in the
        # provisioning agent through periodic rescan and corresponding
        # watches)
        yield self.agent.watch_machine_changes(None, [machine.id])
        yield self.agent.open_close_ports_on_machine(machine.id)
        self.assertEqual((yield self.get_provider_ports(machine)),
                         set([(80, "tcp"), (443, "tcp")]))

    @inlineCallbacks
    def test_open_close_ports_in_stopped_agent_stops_watch(self):
        """Verify code called by watches properly stops when agent stops."""
        yield self.start_agent_with_watch_support(enabled=False)
        yield self.agent.stop()
        yield self.assertFailure(
            self.agent.open_close_ports_on_machine(0),
            StopWatcher)

    @inlineCallbacks
    def test_watches_trigger_port_mgmt(self):
        """Verify that watches properly trigger firewall management
        for the corresponding service units on the corresponding
        machines.
        """
        yield self.start_agent_with_watch_support()
        manager = MachineStateManager(self.client)

        # Immediately expose
        drupal = yield self.add_service("drupal")
        wordpress = yield self.add_service("wordpress")
        yield drupal.set_exposed_flag()
        yield wordpress.set_exposed_flag()

        # Then add these units
        drupal_0 = yield drupal.add_unit_state()
        wordpress_0 = yield wordpress.add_unit_state()
        wordpress_1 = yield wordpress.add_unit_state()
        wordpress_2 = yield wordpress.add_unit_state()

        # Assign some machines; in particular verify that multiple
        # service units on one machine works properly with opening
        # firewall
        expected_machines = self.wait_on_expected_machines(set([0, 1, 2]))
        machine_0 = yield manager.add_machine_state()
        machine_1 = yield manager.add_machine_state()
        machine_2 = yield manager.add_machine_state()
        yield drupal_0.assign_to_machine(machine_0)
        yield wordpress_0.assign_to_machine(machine_0)
        yield wordpress_1.assign_to_machine(machine_1)
        yield wordpress_2.assign_to_machine(machine_2)
        self.assertTrue((yield expected_machines))

        # Simulate service units opening ports
        expected_machines = self.wait_on_expected_machines(set([0, 1]))
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0", "wordpress/1", "drupal/0"]))
        yield drupal_0.open_port(8080, "tcp")
        yield drupal_0.open_port(443, "tcp")
        yield wordpress_0.open_port(80, "tcp")
        yield wordpress_1.open_port(80, "tcp")
        self.assertTrue((yield expected_units))
        self.assertTrue((yield expected_machines))
        self.assertEqual((yield self.get_provider_ports(machine_0)),
                         set([(80, "tcp"), (443, "tcp"), (8080, "tcp")]))
        self.assertEqual((yield self.get_provider_ports(machine_1)),
                         set([(80, "tcp")]))

        # Simulate service units close port
        expected_machines = self.wait_on_expected_machines(set([1, 2]))
        yield wordpress_1.close_port(80, "tcp")
        yield wordpress_2.open_port(80, "tcp")
        self.assertTrue((yield expected_machines))
        self.assertEqual((yield self.get_provider_ports(machine_1)), set())

        # Simulate service units open port
        expected_machines = self.wait_on_expected_machines(set([0]))
        yield wordpress_0.open_port(53, "udp")
        self.assertTrue((yield expected_machines))
        self.assertEqual((yield self.get_provider_ports(machine_0)),
                         set([(53, "udp"), (80, "tcp"),
                              (443, "tcp"), (8080, "tcp")]))
        yield self.agent.stop()

    @inlineCallbacks
    def test_late_expose_properly_triggers(self):
        """Verify that an expose flag properly cascades the
        corresponding watches to perform the desired firewall mgmt.
        """
        yield self.start_agent_with_watch_support()
        manager = MachineStateManager(self.client)
        drupal = yield self.add_service("drupal")
        wordpress = yield self.add_service("wordpress")

        # Then add these units
        drupal_0 = yield drupal.add_unit_state()
        wordpress_0 = yield wordpress.add_unit_state()
        wordpress_1 = yield wordpress.add_unit_state()

        expected_machines = self.wait_on_expected_machines(set([0, 1]))
        machine_0 = yield manager.add_machine_state()
        machine_1 = yield manager.add_machine_state()
        yield drupal_0.assign_to_machine(machine_0)
        yield wordpress_0.assign_to_machine(machine_0)
        yield wordpress_1.assign_to_machine(machine_1)
        self.assertTrue((yield expected_machines))

        # Simulate service units opening ports
        expected_machines = self.wait_on_expected_machines(set([0, 1]))
        expected_units = self.wait_on_expected_units(
            set(["wordpress/0", "wordpress/1"]))
        yield drupal_0.open_port(8080, "tcp")
        yield drupal_0.open_port(443, "tcp")
        yield wordpress_0.open_port(80, "tcp")
        yield wordpress_1.open_port(80, "tcp")
        yield wordpress.set_exposed_flag()
        self.assertTrue((yield expected_machines))
        self.assertTrue((yield expected_units))
        self.assertEqual((yield self.get_provider_ports(machine_0)),
                         set([(80, "tcp")]))
        self.assertEqual((yield self.get_provider_ports(machine_1)),
                         set([(80, "tcp")]))

        # Expose drupal service, verify ports are opened on provider
        expected_machines = self.wait_on_expected_machines(set([0]))
        expected_units = self.wait_on_expected_units(set(["drupal/0"]))
        yield drupal.set_exposed_flag()
        self.assertTrue((yield expected_machines))
        self.assertTrue((yield expected_units))
        self.assertEqual((yield self.get_provider_ports(machine_0)),
                         set([(80, "tcp"), (443, "tcp"), (8080, "tcp")]))

        # Unexpose drupal service, verify only wordpress ports are now opened
        expected_machines = self.wait_on_expected_machines(set([0]))
        expected_units = self.wait_on_expected_units(set(["drupal/0"]))
        yield drupal.clear_exposed_flag()
        self.assertTrue((yield expected_machines))
        self.assertTrue((yield expected_units))
        self.assertEqual((yield self.get_provider_ports(machine_0)),
                         set([(80, "tcp")]))

        # Re-expose drupal service, verify ports are once again opened
        expected_machines = self.wait_on_expected_machines(set([0]))
        expected_units = self.wait_on_expected_units(set(["drupal/0"]))
        yield drupal.set_exposed_flag()
        self.assertTrue((yield expected_machines))
        self.assertTrue((yield expected_units))
        self.assertEqual((yield self.get_provider_ports(machine_0)),
                         set([(80, "tcp"), (443, "tcp"), (8080, "tcp")]))
        yield self.agent.stop()
