/****************************************************************************
 * The main entry point and related JWM functions.
 * Copyright (C) 2004 Joe Wingbermuehle
 ****************************************************************************/

#include "jwm.h"

static const char *CONFIG_FILE = "/.jwmrc";

static void Initialize();
static void Startup();
static void Shutdown();
static void Destroy();

static void StartupConnection();
static void ShutdownConnection();
static void EventLoop();
static void HandleExit();
static void DoExit(int code);

static char *configPath = NULL;
static char *displayString = NULL;

/****************************************************************************
 ****************************************************************************/
int main(int argc, char *argv[]) {
	char *temp;
	int x;

	StartDebug();

	temp = getenv("HOME");
	if(temp) {
		configPath = Allocate(strlen(temp) + strlen(CONFIG_FILE) + 1);
		strcpy(configPath, temp);
		strcat(configPath, CONFIG_FILE);
	} else {
		configPath = Allocate(strlen(CONFIG_FILE) + 1);
		strcpy(configPath, CONFIG_FILE);
	}

	for(x = 1; x < argc; x++) {
		if(!strcmp(argv[x], "-v")) {
			DisplayAbout();
			DoExit(0);
		} else if(!strcmp(argv[x], "-h")) {
			DisplayHelp();
			DoExit(0);
		} else if(!strcmp(argv[x], "-p")) {
			Initialize();
			ParseConfig(configPath);
			Destroy();
			DoExit(0);
		} else if(!strcmp(argv[x], "-display") && x + 1 < argc) {
			displayString = argv[++x];
		} else {
			DisplayUsage();
			DoExit(1);
		}
	}

	do {
		shouldExit = 0;
		shouldRestart = 0;
		Initialize();
		ParseConfig(configPath);
		Startup();
		EventLoop();
		Shutdown();
		Destroy();
	} while(shouldRestart);

	Release(configPath);

	if(exitCommand) {
		execl(SHELL_NAME, SHELL_NAME, "-c", exitCommand, NULL);
		Warning("exec failed: (%s) %s", SHELL_NAME, exitCommand);
		Release(exitCommand);
		exitCommand = NULL;
	}

	StopDebug();

	return 0;
}

/****************************************************************************
 ****************************************************************************/
void DoExit(int code) {
	if(configPath) {
		Release(configPath);
	}
	StopDebug();
	exit(code);
}

/****************************************************************************
 ****************************************************************************/
void EventLoop() {
	XEvent event;

	while(!shouldExit) {
		WaitForEvent(&event);
		ProcessEvent(&event);
	}
}

/****************************************************************************
 ****************************************************************************/
void StartupConnection() {
	XSetWindowAttributes attr;
	int temp;

	initializing = 1;
	display = JXOpenDisplay(displayString);
	if(!display) {
		if(displayString) {
			printf("error: could not open display %s\n", displayString);
		} else {
			printf("error: could not open display\n");
		}
		DoExit(1);
	}

#if 0
	XSynchronize(display, True);
#endif

	JXSetErrorHandler(ErrorHandler);

	clientContext = XUniqueContext();
	frameContext = XUniqueContext();

	rootScreen = DefaultScreen(display);
	rootWindow = RootWindow(display, rootScreen);
	rootWidth = DisplayWidth(display, rootScreen);
	rootHeight = DisplayHeight(display, rootScreen);
	rootDepth = DefaultDepth(display, rootScreen);
	rootColormap = DefaultColormap(display, rootScreen);
	rootVisual = DefaultVisual(display, rootScreen);
	colormapCount = MaxCmapsOfScreen(ScreenOfDisplay(display, rootScreen));

	attr.event_mask = SubstructureRedirectMask | SubstructureNotifyMask
		| PropertyChangeMask | ColormapChangeMask | ButtonPressMask
		| ButtonReleaseMask;
	JXChangeWindowAttributes(display, rootWindow, CWEventMask, &attr);

	signal(SIGTERM, HandleExit);
	signal(SIGINT, HandleExit);
	signal(SIGHUP, HandleExit);

#ifdef USE_SHAPE
	haveShape = JXShapeQueryExtension(display, &shapeEvent, &temp);
	if(!haveShape) {
		Debug("No shape extension.");
	}
#endif

	initializing = 0;

}

/****************************************************************************
 ****************************************************************************/
void ShutdownConnection() {
	JXCloseDisplay(display);
}

/****************************************************************************
 ****************************************************************************/
void HandleExit() {
	signal(SIGTERM, HandleExit);
	signal(SIGINT, HandleExit);
	signal(SIGHUP, HandleExit);
	shouldExit = 1;
}

/****************************************************************************
 * This is called before the X connection is opened.
 ****************************************************************************/
void Initialize() {
	InitializeBorders();
	InitializeClients();
	InitializeColors();
	InitializeCursors();
	#ifndef DISABLE_CONFIRM
		InitializeDialogs();
	#endif
	InitializeFonts();
	InitializeGroups();
	InitializeHints();
	InitializeIcons();
	InitializeKeys();
	InitializeLoadDisplay();
	InitializeOutline();
	InitializeOSDependent();
	InitializePager();
	InitializePopup();
	InitializeRootMenu();
	InitializeTiming();
	InitializeTray();
}

/****************************************************************************
 * This is called after the X connection is opened.
 ****************************************************************************/
void Startup() {

	/* This order is important. */

	StartupConnection();

	StartupGroups();
	StartupColors();
	StartupFonts();
	StartupCursors();
	StartupOutline();

	StartupLoadDisplay();
	StartupPager();
	StartupTray();
	StartupKeys();
	StartupHints();
	StartupBorders();
	StartupClients();

	StartupIcons();
	StartupTiming();
	#ifndef DISABLE_CONFIRM
		StartupDialogs();
	#endif
	StartupPopup();

	StartupRootMenu();

	SetDefaultCursor(rootWindow);
	ReadCurrentDesktop();
	JXFlush(display);

}

/****************************************************************************
 * This is called before the X connection is closed.
 ****************************************************************************/
void Shutdown() {

	/* This order is important. */

	ShutdownOutline();
	#ifndef DISABLE_CONFIRM
		ShutdownDialogs();
	#endif
	ShutdownPopup();
	ShutdownKeys();
	ShutdownPager();
	ShutdownRootMenu();
	ShutdownLoadDisplay();
	ShutdownTray();
	ShutdownBorders();
	ShutdownClients();
	ShutdownIcons();
	ShutdownCursors();
	ShutdownFonts();
	ShutdownColors();
	ShutdownGroups();

	ShutdownHints();
	ShutdownTiming();

	ShutdownConnection();

}

/****************************************************************************
 * This is called after the X connection is closed.
 ****************************************************************************/
void Destroy() {
	DestroyBorders();
	DestroyClients();
	DestroyColors();
	DestroyCursors();
	#ifndef DISABLE_CONFIRM
		DestroyDialogs();
	#endif
	DestroyFonts();
	DestroyGroups();
	DestroyHints();
	DestroyIcons();
	DestroyKeys();
	DestroyLoadDisplay();
	DestroyOutline();
	DestroyOSDependent();
	DestroyPager();
	DestroyPopup();
	DestroyRootMenu();
	DestroyTiming();
	DestroyTray();
}


