// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

#include "basic_bitmap_modifier.h"

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_matte_colordiff

class bitmap_matte_colordiff :
	public basic_bitmap_modifier
{
	typedef basic_bitmap_modifier base;

public:
	bitmap_matte_colordiff(k3d::idocument& Document) :
		base(Document),
		m_value(init_owner(*this) + init_name("value") + init_label(_("Threshold Value")) + init_description(_("Set alpha channel using color difference")) + init_value(0.0))
	{
		m_value.changed_signal().connect(sigc::mem_fun(*this, &bitmap_matte_colordiff::on_value_change));
	}

	void on_value_change(k3d::iunknown* const Hint)
	{
		m_value_cache = m_value.value();

		on_reset_bitmap(Hint);
	}

        void on_filter(const k3d::pixel& I, k3d::pixel& O)
	{
		O.red   = I.red;
		O.green = I.green;
		O.blue  = std::min(I.green, I.blue);
		O.alpha = I.blue > m_value_cache ? 1 - (I.blue - std::max(I.red, I.green)) : 1;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_matte_colordiff>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0xeefaccf2, 0x65bc4c78, 0xbd46cbdb, 0x5ca5d3e2),
				"BitmapMatteColorDiff",
				_("Create alpha channel using color difference"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_value;

	double m_value_cache;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_matte_colordiff_factory

k3d::iplugin_factory& bitmap_matte_colordiff_factory()
{
	return bitmap_matte_colordiff::get_factory();
}

} // namespace libk3dbitmap


