// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
	\author Tim Shead (tshead@k-3d.com)
*/

#include "application_state.h"
#include "black_box_recorder.h"
#include "messages.h"

#include <k3dsdk/classes.h>
#include <k3dsdk/command_tree.h>
#include <k3dsdk/create_plugins.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/icommand_node.h>
#include <k3dsdk/iscript_engine.h>
#include <k3dsdk/fstream.h>
#include <k3dsdk/system.h>
#include <k3dsdk/version.h>

#include <boost/date_time/gregorian/gregorian.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <boost/regex.hpp>

namespace libk3dngui
{

namespace detail
{
	
const std::string started_token = "Black Box Recorder Started:";
const std::string stopped_token = "Black Box Recorder Stopped:";

} // namespace detail

////////////////////////////////////////////////////////////////////////////////
// black_box_recorder::implementation

class black_box_recorder::implementation
{
public:
	implementation() :
		script_engine(0)
	{
		k3d::command_tree().command_signal().connect(sigc::mem_fun(*this, &implementation::on_command));
	}

	void validate_old_logs()
	{
		const boost::filesystem::path log_path =
			k3d::system::get_home_directory() / boost::filesystem::path(".k3d/blackbox.log", boost::filesystem::native);
		if(!boost::filesystem::exists(log_path))
			return;

		std::stringstream buffer_stream;
		k3d::filesystem::ifstream stream(log_path);
		buffer_stream << stream.rdbuf();
		stream.close();

		const std::string buffer = buffer_stream.str();
		const bool normal_exit = buffer.find(detail::started_token) != std::string::npos &&
			buffer.rfind(detail::stopped_token) != std::string::npos;

		boost::match_results<std::string::const_iterator> match;
		typedef std::vector<std::string> files;

		files started_files;
		boost::regex start_exp("<startfile>([^<]*)</startfile>");
		for(std::string::const_iterator begin = buffer.begin(); boost::regex_search(begin, buffer.end(), match, start_exp); begin = match[0].second)
			started_files.push_back(match[1].str());

		files completed_files;
		boost::regex complete_exp("<completefile>([^<]*)</completefile>");
		for(std::string::const_iterator begin = buffer.begin(); boost::regex_search(begin, buffer.end(), match, complete_exp); begin = match[0].second)
			completed_files.push_back(match[1].str());
		
		if(normal_exit && (started_files == completed_files))
			return;

		unsigned long i = 1;
		boost::filesystem::path storage = log_path;
		while(boost::filesystem::exists(storage))
			storage = k3d::system::get_home_directory() / boost::filesystem::path(".k3d/blackbox.log." + k3d::string_cast(i++), boost::filesystem::native);

		try
		{
			boost::filesystem::rename(log_path, storage);
		}
		catch(std::exception& e)
		{
			k3d::log() << error << "Error archiving blackbox log: " << log_path.native_file_string() << ": " << e.what() << std::endl;
		}

		if(application_state::instance().batch_mode())
			return;

		std::string command_line;
		command_line += "k3d-bug-buddy";
		command_line += " \"" + storage.native_file_string() + "\"";
		for(files::const_iterator file = started_files.begin(); file != started_files.end(); ++file)
			command_line += " \"" + *file + "\"";
		
		if(!k3d::system::run_process(command_line))
		{
			// We fall-back on our old nag message if bug-buddy can't be run for any reason ...
			boost::format warning_message(_("It appears that a previous K-3D session may have terminated unexpectedly.  A logfile has been saved as\n\n%1%\n\nPlease send this file to the K-3D development team for analysis."));
			warning_message % storage.native_file_string();
			message(warning_message.str(), _("Black Box Recorder:"));
		}
	}
	
	void on_command(k3d::icommand_node& CommandNode, const k3d::icommand_node::type Type, const std::string& Name, const std::string& Arguments)
	{
		// We only want user interface commands ... !
		if(Type != k3d::icommand_node::COMMAND_INTERACTIVE)
			return;

		// And only if recording is enabled ...
		if(!log_stream.is_open())
			return;

		// Sanity checks ...
		return_if_fail(script_engine);

		script_engine->append_command(log_stream, CommandNode, Name, Arguments);
		log_stream.flush();
	}

	/// Signal for notifying observers whenever recording starts/stops
	sigc::signal<void> recording_changed_signal;
	/// Stream for recording black box data
	k3d::filesystem::ofstream log_stream;
	/// Script engine for recording black box data
	k3d::iscript_engine* script_engine;
};

/////////////////////////////////////////////////////////////////////////////////
// black_box_recorder

black_box_recorder& black_box_recorder::instance()
{
	static black_box_recorder g_instance;
	return g_instance;
}

black_box_recorder::black_box_recorder() :
	m_implementation(new implementation())
{
}

black_box_recorder::~black_box_recorder()
{
	delete m_implementation;
}

void black_box_recorder::start()
{
	if(recording())
		return;

	if(!m_implementation->script_engine)
		m_implementation->script_engine = dynamic_cast<k3d::iscript_engine*>(k3d::create_plugin(k3d::classes::K3DScriptEngine()));
	return_if_fail(m_implementation->script_engine);

	m_implementation->validate_old_logs();

	const boost::filesystem::path log_path = 
		k3d::system::get_home_directory() / boost::filesystem::path(".k3d/blackbox.log", boost::filesystem::native);
	m_implementation->log_stream.open(log_path);
	return_if_fail(m_implementation->log_stream.good());

	std::ostringstream buffer;
	buffer << detail::started_token << " " << boost::posix_time::second_clock::universal_time() << " UTC\n";
	buffer << "Package: " << K3D_PACKAGE << "\n";
	buffer << "Version: " << K3D_VERSION << "\n";
	buffer << "Platform: " << K3D_HOST << "\n";
	buffer << "Compiler: " << __VERSION__ << "\n";
	buffer << "Build Time: " << __DATE__ << " " << __TIME__ << " local\n";

	m_implementation->script_engine->bless_script(m_implementation->log_stream);
	m_implementation->script_engine->append_comment(m_implementation->log_stream, buffer.str());
	m_implementation->log_stream.flush();

	m_implementation->recording_changed_signal.emit();
}

const bool black_box_recorder::recording()
{
	return m_implementation->log_stream.is_open();
}

sigc::connection black_box_recorder::connect_recording_changed_signal(const sigc::slot<void>& Slot)
{
	return m_implementation->recording_changed_signal.connect(Slot);
}

void black_box_recorder::start_file(const boost::filesystem::path& File)
{
	std::ostringstream buffer;
	buffer << "<startfile>" << File.native_file_string() << "</startfile>";
	record_message(buffer.str());
}

void black_box_recorder::complete_file(const boost::filesystem::path& File)
{
	std::ostringstream buffer;
	buffer << "<completefile>" << File.native_file_string() << "</completefile>";
	record_message(buffer.str());
}
	
void black_box_recorder::record_message(const std::string& Message)
{
	if(!recording())
		return;

	return_if_fail(m_implementation->script_engine);
	m_implementation->script_engine->append_comment(m_implementation->log_stream, Message);
	m_implementation->log_stream.flush();
}

void black_box_recorder::prompt_stop()
{
	if(!recording())
		return;

	if(!application_state::instance().batch_mode())
	{
		std::vector<std::string> buttons;
		buttons.push_back(_("OK"));
		buttons.push_back(_("Cancel"));

		if(1 != query_message(_("Disable Black Box Recorder?  The Black Box Recorder is a powerful tool for reporting bugs to the K-3D development team."), _("Disable Black Box Recorder?"), 1, buttons))
			return;
	}

	stop();
}

void black_box_recorder::stop()
{
	if(!recording())
		return;

	std::ostringstream buffer;
	buffer << detail::stopped_token << " " << boost::posix_time::second_clock::universal_time() << " UTC\n";

	m_implementation->script_engine->append_comment(m_implementation->log_stream, buffer.str());
	m_implementation->log_stream.flush();
	m_implementation->log_stream.close();

	m_implementation->recording_changed_signal.emit();
}

/////////////////////////////////////////////////////////////////////////
// black_box_recorder::process_file

black_box_recorder::process_file::process_file(const boost::filesystem::path& File) :
	file(File)
{
	black_box_recorder::instance().start_file(file);
}

black_box_recorder::process_file::~process_file()
{
	black_box_recorder::instance().complete_file(file);
}

} // namespace libk3dui

