// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, read to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the TIFF reader K-3D object, which reads bitmap images using Sam Leffler's libtiff API
		\author Timothy M. Shead
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/module.h>
#include <k3dsdk/string_modifiers.h>

#include <boost/filesystem/path.hpp>

#include <tiffio.h>

namespace libk3dtiff
{

template<typename source_bitmap_t, typename target_bitmap_t>
void copy_mirror_y(const source_bitmap_t& Source, target_bitmap_t& Target)
{
	typedef typename source_bitmap_t::pixel_type source_pixel_t;
	typedef typename target_bitmap_t::pixel_type target_pixel_t;

	Target.reset(Source.width(), Source.height());

	for(k3d::pixel_size_t row = 0; row != Source.height(); ++row)
		std::copy(Source.row_begin(row), Source.row_end(row), Target.row_begin(Source.height() - row - 1));
}

/////////////////////////////////////////////////////////////////////////////
// tiff_reader

class tiff_reader :
	public k3d::ifile_format,
	public k3d::ibitmap_read_format,
	public k3d::ideletable
{
public:
	tiff_reader()
	{
	}

	unsigned long priority()
	{
		return 128;
	}

	bool query_can_handle(const boost::filesystem::path& File)
	{
		return_val_if_fail(!File.empty(), false);
		return "tiff" == k3d::file_extension(File) || "tif" == k3d::file_extension(File);
	}

	bool read_file(const boost::filesystem::path& File, k3d::bitmap& Bitmap)
	{
		// Sanity checks ...
		return_val_if_fail(!File.empty(), false);

		k3d::log() << info << "Read " << File.native_file_string() << " using TIFFReader" << std::endl;

		TIFF* const file = TIFFOpen(File.native_file_string().c_str(), "r");
		if(!file)
			{
				k3d::log() << error << "Error opening [" << File.native_file_string() << "] for TIFF input" << std::endl;
				return false;
			}

		// Load the input bitmap using 8-bit-integer samples before converting to K-3D's internal format ...
		uint32 width = 0;
		uint32 height = 0;
		TIFFGetField(file, TIFFTAG_IMAGEWIDTH, &width);
		TIFFGetField(file, TIFFTAG_IMAGELENGTH, &height);

		typedef k3d::basic_rgba<boost::uint8_t> integer_pixel_t;
		typedef k3d::basic_bitmap<integer_pixel_t> integer_bitmap_t;
		integer_bitmap_t bitmap(width, height);

		if(!TIFFReadRGBAImage(file, width, height, reinterpret_cast<uint32*>(bitmap.data()), 0))
			{
				TIFFClose(file);

				k3d::log() << error << "Error reading [" << File.native_file_string() << "]" << std::endl;
				return false;
			}

		copy_mirror_y(bitmap, Bitmap);

		TIFFClose(file);
		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::application_plugin<tiff_reader>,
			k3d::interface_list<k3d::ibitmap_read_format> > factory(
				k3d::uuid(0x3cfedd91, 0xd5764d3a, 0x91b15d47, 0xdcdcf962),
				"TIFFReader",
				_("TIFF (*.tiff)"),
				"Bitmap BitmapReader");

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// tiff_reader_factory

k3d::iplugin_factory& tiff_reader_factory()
{
	return tiff_reader::get_factory();
}

} // namespace libk3dtiff


