#ifndef KADU_PROTOCOL_H
#define KADU_PROTOCOL_H

#include <qobject.h>
#include <qstring.h>
#include <qdatetime.h>

#include "status.h"

class Protocol : public QObject
{
	Q_OBJECT
	protected:
		QDateTime ConnectionTime;

		QString ProtocolID;
		QString id;

		/**
			Biecy status. Zmieniany po poczeniu, oraz w przypadku zmiany statusu kiedy poczenie
			jest ju zainicjowane.

			@see login
			@see connected
			@see NextStatus
			@see UserStatus
			@see status
		**/
		UserStatus *CurrentStatus;

		/**
			Nastpny status. Ustalany zewntrznie przy wykorzystaniu metody status i odpowiednich
			slotw klasy UserStatus. Zmiana wywouje jedn z metod iWantGo... i w konsekwencji zmian
			statusu (w razie koniecznoci te zalogowanie).

			@see login
			@see connected
			@see CurrentStatus
			@see UserStatus
			@see status
		**/
		UserStatus *NextStatus;

	public:
		Protocol(const QString &proto, const QString &id, QObject *parent = 0, const char *name = 0);
		virtual ~Protocol();

		/**
			Status uytkownika. Za pomoc tej metody moemy go zmieni, pobra ikon statusu i wykona
			kilka innych ciekawych rzeczy.

			1. Zmiana statusu:
			<code>
				Protocol *proto;

				...

				proto->status().setOnline("Jestem zalogowany"); // zalogowanie i ustawienie opisu
				proto->status().setFriendsOnly(true);           // tryb tylko dla przyjaci
				...
				proto->status().setOffline();                   // wylogowanie, usunicie opisu
			</code>

			2. Sprawdzenie statusu:
			<code>
				Protocol *proto;

				if (proto->status().isOnline())                 // jestemy online
					...
				else if (proto->status().isInvisible())         // jestemy niewidzialni
					...

				// mona te:
				switch (proto->status().status())
				{
					case Online:
						break;
					case Busy:
						break;
					case Invisible:
						break;
					case Offline:
						break;
				}
			</code>

			3. Pobranie ikony i nazwy statusu
			<code>
				QPixmap pix;
				QString name;
				Protocol *proto;

				...

				pix = proto->status().pixmap();
				name = proto->status().name();
			</code>

			@see currentStatus
		**/
		UserStatus &status() { return *NextStatus; }

		/**
			Rzeczywisty aktualny status. Mona go wykorzysta tylko w trybie do odczytu (pobranie
			ikony, nazwy, sprawdzenie rzeczywistego stanu poczenia).

			@see status
		**/
		const UserStatus &currentStatus() const { return *CurrentStatus; }

		QString protocolID() const {	return ProtocolID;	}

		QString ID() const {	return id;	}

		virtual UserStatus *newStatus() const = 0;

		const QDateTime &connectionTime() const;

	signals:

		/**
			udao si zalogowa
		**/
		void connected();

		/**
			rozpoczynamy procedur logowania si
		**/
		void connecting();

		/**
			rozczylimy si z serwerem
		**/
		void disconnected();

		/**
			wystpi bd poczenia
			@param protocol protok
			@param reason napis do wywietlenia dla uytkownika
		**/
		void connectionError(Protocol *protocol, const QString &reason);
	private:
		Protocol(const Protocol &) {}
		virtual Protocol &operator=(const Protocol &){return *this;}
};

#endif
