/***************************************************************************

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

***************************************************************************
Copyright (C) 2005 by 
Pan Wojtas (Wojtek Sulewski)
wojteksulewski <at> op.pl
gg: 2087202

Przystosowanie do Kadu 0.6 i inne zmiany
Copyright (C) 2007 by
Tomasz 'Dorregaray' Rostanski
rozteck <at> interia.pl
***************************************************************************/

#include <stdlib.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qspinbox.h>
#include <qpixmap.h>

#include "icons_manager.h"
#include "osd_manager.h"
#include "chat_manager.h"
#include "kadu_parser.h"
#include "misc.h"
#include "kadu.h"
#include "userbox.h" //mapToGlobal
#include "debug.h"

OSDManager::OSDManager(QWidget *parent, const char *name) : Notifier(parent, name), ToolTipClass(), tipWidget(0)
{
	kdebugf();
	
	height = 0;

	importConfiguration();
	setDefaultValues();
	reload();

	connect(chat_manager, SIGNAL(chatWidgetActivated(ChatWidget *)), this, SLOT(chatCreated(ChatWidget *)));
	connect(this, SIGNAL(searchingForTrayPosition(QPoint &)), kadu, SIGNAL(searchingForTrayPosition(QPoint &)));
	
	notification_manager->registerNotifier(QT_TRANSLATE_NOOP("@default", "OSDHints"), this);
	tool_tip_class_manager->registerToolTipClass(QT_TRANSLATE_NOOP("@default", "OSDHints"), this);
	
	kdebugf2();
}

OSDManager::~OSDManager()
{
	tool_tip_class_manager->unregisterToolTipClass("OSDHints");
	notification_manager->unregisterNotifier("OSDHints");

	disconnect(chat_manager, SIGNAL(chatWidgetActivated(ChatWidget *)), this,
			   SLOT(chatCreated(ChatWidget *)));

	deleteAllWidgets();
}

void OSDManager::chatCreated(ChatWidget *widget)
{
	kdebugf();

	const UserGroup *group = widget->users();
	
	FOREACH(i, *group)
	{
		FOREACH(j, widgets)
		{
			if ((*j)->getUsers().contains(*i))
			{
				timeout((*j)->getID(), true);
				break;
			}
		}
	}

	kdebugf2();
}

void OSDManager::reload(void)
{
	kdebugf();
	
	deleteAllWidgets();
	new_osd = config_file.readUnsignedNumEntry("osdhints","NewOSD", 0);
	corner = config_file.readUnsignedNumEntry("osdhints", "Corner");
	separator = config_file.readUnsignedNumEntry("osdhints","OSD_separator");
	if (config_file.readBoolEntry("osdhints", "Use translucency", false))
		translucency_level = config_file.readNumEntry("osdhints", "Translucency level", 80) * 0.01;
	else
		translucency_level = 1;
	def_x = config_file.readNumEntry("osdhints", "OSD_x", 300);
	def_y = config_file.readNumEntry("osdhints", "OSD_y", 300);
	
	kdebugf2();
}

void OSDManager::addOSD(Notification *notification)
{
	kdebugf();
	// TODO: co sie da przeniesc stad do konstruktora OSDWidget
kdebugm(KDEBUG_ERROR, "\n\n\n\n\n\naddOSD\n");
	if (new_osd == 4)
		deleteAllWidgets();

	QString event = notification->type();
	if (config_file.readBoolEntry("osdhints", "SetAll", false))
		event = "SetAll";

	QFont font =
			config_file.readFontEntry("osdhints", event + "_font");
	QColor fg_color =
			config_file.readColorEntry("osdhints", event + "_fgcolor");
	QColor bg_color =
			config_file.readColorEntry("osdhints", event + "_bgcolor");
	QColor border_color =
			config_file.readColorEntry("osdhints", event + "_bordercolor");
	unsigned int timeout =
			config_file.readUnsignedNumEntry("osdhints", event + "_timeout", 1) * 1000;
	int maskEffect =
			config_file.readNumEntry("osdhints", event + "_maskeffect", 0);
	QString syntax =
			config_file.readEntry("osdhints", notification->type() + "_syntax", defaultSyntax);
	int iconSize =
			config_file.readNumEntry("osdhints", "icon_size", 32);
	QString icon = dataPath("kadu/modules/data/osdhints_notify/") + notification->icon().lower() +
			QString::number(iconSize) + ".png";

	// sprawdz czy mamy odpowiednia ikonke, w przeciwnym razie uzyj ikonki modulu
	// (lepsze to niz szare miejsce)
	if (icons_manager->loadIcon(icon).isNull())
		icon = dataPath("kadu/modules/data/osdhints_notify/osd_icon.png");

	QString text = syntax.replace("%&m", notification->text());
	text = text.replace("%&d", notification->details());
	text = text.replace("%&i", icon);

	UserListElement ule;
	if (notification->userListElements().count())
		ule = notification->userListElements()[0];
	text = KaduParser::parse(text, ule);

	OSDWidget *widget = new OSDWidget(0, notification);
	widgets.append(widget);
	unsigned int id = widgets.count() - 1;
	widget->setFont(font);
	
	OSDWidget::OSDData data;
	
	HtmlDocument doc;
	doc.parseHtml(text);
	doc.convertUrlsToHtml();
	
	if(config_file.readBoolEntry("osdhints", "show_emoticons", false))
	{
		widget->mimeSourceFactory()->addFilePath(emoticons->themePath());
		emoticons->expandEmoticons(doc, bg_color);
	}
	text = doc.generateHtml();

	data.message = text;

	switch(maskEffect)
	{
		case 1:
			data.maskEffect = OSDWidget::Dissolve;
			break;
		default:
			data.maskEffect = OSDWidget::Plain;
			break;
	}
	
	data.fg_color = fg_color;
	data.bg_color = bg_color;
	data.border_color = border_color;
	data.duration = timeout;
	data.id = id;
	data.translucency_val = translucency_level;

	widget->setData(data);
	widget->prepare();

	OSDWidget *tmp;
	QPoint trayPosition;
	emit searchingForTrayPosition(trayPosition);
	if (!config_file.readBoolEntry("osdhints", "UseUserPosition") && !trayPosition.isNull()) {
		const QRect desktopSize = QApplication::desktop()->screenGeometry(0);
		if (trayPosition.x() < 0)
			trayPosition.setX(0);
		else if (trayPosition.x() > desktopSize.width())
			trayPosition.setX(desktopSize.width() - 2);
		if (trayPosition.y() < 0)
			trayPosition.setY(0);
		else if (trayPosition.y() > desktopSize.height())
			trayPosition.setY(desktopSize.height() - 2);

		if (trayPosition.x() < desktopSize.width() / 2) // tray is on left
			def_x = trayPosition.x() + 32;
		else // tray is on right
			def_x = trayPosition.x() - widget->width();

		if (trayPosition.y() < desktopSize.height() / 2) { // tray is on top
			def_y = trayPosition.y() + 32;
			corner = 1;
		}
		else { // tray is on bottom
			def_y = trayPosition.y() - widget->height();
			corner = 0;
		}
		
		corner = 0;
	}

	// TODO: zoptymalizowa? :)
	widget->setX(def_x);
	switch(corner)
	{
		case 0:		// Left top
			if (((widgets.count() > 1) && (new_osd == 0)) || (new_osd != 0))
				height += widget->getHeight() + separator;
			break;
		case 1:		// Left bottom
			if (((widgets.count() > 1) && (new_osd == 2)) || (new_osd != 2))
				height += widget->getHeight() + separator;
			break;
		case 2:		// Right top
			widget->setX(def_x - widget->getWidth());
			if (((widgets.count() > 1) && (new_osd == 0)) || (new_osd != 0))
				height += widget->getHeight() + separator;
			break;
		case 3:		// Right bottom
			widget->setX(def_x - widget->getWidth());
			if (((widgets.count() > 1) && (new_osd == 2)) || (new_osd != 2))
				height += widget->getHeight() + separator;
			break;
	}

	widget->setY(def_y);
	switch(new_osd)
	{
		case 0:
			if (widgets.count())
				widget->setY(def_y - height);
			break;
		case 1:
			if (widgets.count() > 1)
			{
				unsigned int shift;
				if ((corner == 0) || (corner == 2))
					shift = widget->getHeight() + separator;
				else
					shift = widgets.at(id - 1)->getHeight() + separator;

				for (unsigned int iter = 0; iter <= widgets.count() - 2; iter++)
				{
					tmp = widgets.at(iter);
					tmp->setY(tmp->getY() + shift);
					tmp->QWidget::hide();
					tmp->display();
				}
			}
			if (corner == 1 || corner == 3)
				widget->setY(def_y - widget->getHeight());
			break;
		case 2:
			widget->setY(def_y + height - widget->getHeight() - separator);
			break;
		case 3:
			if (widgets.count() > 1)
			{
				unsigned int shift;
				if (corner == 1 || corner == 3)
					shift = widget->getHeight() + separator;
				else
					shift = widgets.at(id - 1)->getHeight() + separator;
		
				for (unsigned int iter = 0; iter <= widgets.count() - 2; iter++)
				{
					tmp = widgets.at(iter);
					tmp->setY(tmp->getY() - shift);
					tmp->QWidget::hide();		//na prawd nie rozumiem, po co :|
					tmp->display();
				}
			}
			if (corner == 1 || corner == 3)
				widget->setY(def_y - widget->getHeight());
			break;
		case 4:
			break;
	}

	connect(widget, SIGNAL(timeout(int, bool)), this, SLOT(timeout(int, bool)));
	connect(widget, SIGNAL(closing(OSDWidget *)), this, SLOT(deleteWidget(OSDWidget *)));

	connect(widget, SIGNAL(leftButtonClicked(OSDWidget *)), this,
			  SLOT(leftButtonSlot(OSDWidget *)));
	connect(widget, SIGNAL(rightButtonClicked(OSDWidget *)), this,
			  SLOT(rightButtonSlot(OSDWidget *)));
	connect(widget, SIGNAL(midButtonClicked(OSDWidget *)), this,
			  SLOT(midButtonSlot(OSDWidget *)));

	widget->display();

	kdebugf2();
}

void OSDManager::timeout(int id, bool chat)
{
	kdebugf();
	kdebugm(KDEBUG_INFO, "timeout: OSDWidget id: %d\n", id);
	
	UserListElements users = widgets.at(id)->getUsers();
	int shift = 0;	
	int iter = 0;
	OSDWidget *ptr = 0;

	switch(new_osd)
	{
		case 0:
		{
			int new_id = 0;
			int count = widgets.count();
kdebugm(KDEBUG_INFO, "widget count: %d\n", count);
			while (iter < count)
			{
				ptr = widgets.at(iter);
kdebugm(KDEBUG_INFO, "got widget at pos: %d\n", iter);
				if (ptr->getID() == id || (chat && ptr->hasUsers() && (ptr->getUsers() == users)))
				{
					if (count > 1)
					{
						int a = ptr->getHeight() + separator;
						if (iter != 0 || corner == 1 || corner == 3)
						{
							height -= a;
							shift += a;
						}
					}
					else
					{
						shift = 0;
						height = 0;
					}
kdebugm(KDEBUG_INFO, "remove widget at pos: %d\n", iter);
					ptr->close();
					widgets.remove(ptr);
					count--;
				}
				else
				{
					if (iter == 0 && (corner == 0 || corner == 2))
					{
						shift = abs(def_y - ptr->getY());
						height -= def_y - ptr->getY();
					}
					if (shift != 0)
					{
kdebugm(KDEBUG_INFO, "shift widget at pos: %d\n", iter);
						ptr->setY(ptr->getY() + shift);
						ptr->setID(new_id);
						ptr->QWidget::hide();
						ptr->display();
					}
					iter++;
					new_id++;
				}
			}
			break;
		}
		case 2:
		{
			int new_id = 0;
			int count = widgets.count();
			while(iter < count)
			{
				ptr = widgets.at(iter);
				if (ptr->getID() == id || (chat && ptr->hasUsers() && (ptr->getUsers() == users)))
				{
					if (count > 1)
					{
						int a = ptr->getHeight() + separator;
						if (iter != 0 || corner == 0 || corner == 2)
						{
							shift += a;
							height -= a;
						}
					}
					else
					{
						shift = 0;
						height = 0;
					}
					ptr->close();
					widgets.remove(ptr);
					count--;
				}
				else
				{
					if (iter == 0 && (corner == 1 || corner == 3))
					{
						shift = abs(def_y - ptr->getY() - ptr->getHeight());
						height += def_y - ptr->getY() - ptr->getHeight();
					}
					if (shift != 0)
					{
						ptr->setY(ptr->getY() - shift);
						ptr->setID(new_id);
						ptr->QWidget::hide();
						ptr->display();
					}
					iter++;
					new_id++;
				}
			}
			break;
		}
		case 1:
		{
			ptr = widgets.last();
			do
			{
				if (ptr->getID() == id || (chat && ptr->hasUsers() && (ptr->getUsers() == users)))
				{
					shift += ptr->getHeight() + separator;
					ptr->close();
					widgets.remove(ptr);
					
					if ((uint)(ptr->getID()) == widgets.count())	//TODO: jako tak adniej :)
					{
						ptr = widgets.current();
					}
					else
						ptr = widgets.prev();
				}
				else
				{
					if (ptr == widgets.getLast() && (corner == 1 || corner == 3))
						shift = abs(def_y - ptr->getY() - ptr->getHeight());
					
					if (shift != 0)
					{
						ptr->setY(ptr->getY() - shift);
						ptr->QWidget::hide();
						ptr->display();
					}
					
					ptr = widgets.prev();
				}
			} while (ptr);
			
			
			int new_id = 0;
			for (ptr = widgets.first(); ptr; ptr = widgets.next())
			{
				ptr->setID(new_id);
				new_id++;
			}
			break;
		}
		case 3:
		{
			OSDWidget *ptr = widgets.last();
			do
			{
				if (ptr->getID() == id || (chat && ptr->hasUsers() && (ptr->getUsers() == users)))
				{
					shift += ptr->getHeight() + separator;
					
					ptr->close();
					widgets.remove(ptr);
					
					if ((uint)(ptr->getID()) == widgets.count())	//TODO: jako tak adniej :)
					{
						ptr = widgets.current();
					}
					else
						ptr = widgets.prev();
				}
				else
				{
					if (ptr == widgets.getLast() && (corner == 0 || corner == 2))
						shift = abs(def_y - ptr->getY());
					
					if (shift != 0)
					{
						ptr->setY(ptr->getY() + shift);
						ptr->QWidget::hide();
						ptr->display();
					}
					
					ptr = widgets.prev();
				}
			} while (ptr);

			int new_id = 0;
			for (ptr = widgets.first(); ptr; ptr = widgets.next())
			{
				ptr->setID(new_id);
				new_id++;
			}
			break;
		}
	}

	kdebugf2();
}

void OSDManager::deleteAllWidgets(void)
{
	kdebugf();
	
	height = 0;

	CONST_FOREACH(it, widgets)
	{
		if (!(*it)->requireManualClosing()) {
			(*it)->close();
			(*it)->deleteLater();
			widgets.remove(*it);
		}
	}
	//widgets.clear();

	kdebugf2();
}

void OSDManager::deleteWidget(OSDWidget *widget)
{
	widget->close();
	widget->deleteLater();
	widgets.remove(widget);
}

void OSDManager::leftButtonSlot(OSDWidget *widget)
{
	processButtonPress("LeftButton", widget);
}

void OSDManager::rightButtonSlot(OSDWidget *widget)
{
	processButtonPress("RightButton", widget);
}

void OSDManager::midButtonSlot(OSDWidget *widget)
{
	processButtonPress("MiddleButton", widget);
}

void OSDManager::openChat(OSDWidget *widget)
{
	kdebugf();

	if (!widget->hasUsers())
		return;

	if (!config_file.readBoolEntry("osdhints", "OpenChatOnEveryNotification"))
		if ((widget->getNotification()->type() != "NewChat") && (widget->getNotification()->type() != "NewMessage"))
			return;

	chat_manager->openPendingMsgs(widget->getUsers());

	kdebugf2();
}

void OSDManager::processButtonPress(const QString &buttonName, OSDWidget *widget)
{
	switch(config_file.readNumEntry("osdhints", buttonName))
	{
		case 1:
			openChat(widget);
			widget->acceptNotification();

			deleteWidget(widget);
			break;

		case 2:
			if (widget->hasUsers() &&
						 config_file.readBoolEntry("osdhints", "DeletePendingMsg", false))
				chat_manager->deletePendingMsgs(widget->getUsers());

			widget->discardNotification();

			deleteWidget(widget);
			break;

		case 3:
			deleteAllWidgets();
			break;
	}

	kdebugf2();
}

void OSDManager::windowActivationChanged(bool b, const UserGroup *users)
{
	kdebugf();
	
	if (!b)
	{
		kdebugm(KDEBUG_INFO, "windowActivated\n");
		
		UserListElements u_list = users->toUserListElements();
		CONST_FOREACH(i, widgets)
		{
			if ((*i)->getUsers() == u_list)
			{
				timeout((*i)->getID(), true);
				
				kdebugf2();
				return;
			}
			
		}
	}

	kdebugf2();
}

void OSDManager::hideToolTip()
{
	if (tipWidget)
	{
		tipWidget->close();
		tipWidget->deleteLater();
		tipWidget = 0;
	}
}

void OSDManager::showToolTip(const QPoint &point, const UserListElement &user)
{
	kdebugf();
	kdebugm(KDEBUG_INFO, "user: '%s', x:%d, y:%d\n", user.altNick().local8Bit().data(), point.x(), point.y());

	if (tipWidget)
		return;

	tipWidget = new OSDWidget(0);

	OSDWidget::OSDData data;

	QString name = "OSDToolTip";
	if (config_file.readBoolEntry("osdhints", "SetAll", false))
		name = "SetAll";

	tipWidget->setFont(config_file.readFontEntry("osdhints", name + "_font"));
	data.fg_color = config_file.readColorEntry("osdhints", name + "_fgcolor");
	data.bg_color = config_file.readColorEntry("osdhints", name + "_bgcolor");
	data.border_color = config_file.readColorEntry("osdhints", name + "_bordercolor");
	switch(config_file.readNumEntry("osdhints", name + "_maskeffect"))
	{
		case 1:
		data.maskEffect = OSDWidget::Dissolve;
			break;
		default:
			data.maskEffect = OSDWidget::Plain;
			break;
	}

	data.duration = 0;
	data.id = -1;
	data.translucency_val = translucency_level;

	QPoint pos(kadu->userbox()->mapToGlobal(point) + QPoint(5, 5));
	data.x = pos.x();
	data.y = pos.y();

	QString text = KaduParser::parse(config_file.readEntry("osdhints", "MouseOverUserSyntax"), user);

	while (text.endsWith("<br/>"))
		text.setLength(text.length() - 5 /* 5 == QString("<br/>").length()*/);
	while (text.startsWith("<br/>"))
		text = text.right(text.length() - 5 /* 5 == QString("<br/>").length()*/);

	HtmlDocument doc;
	doc.parseHtml(text);
	doc.convertUrlsToHtml();

	if(config_file.readBoolEntry("osdhints", "show_emoticons", false))
	{
		tipWidget->mimeSourceFactory()->addFilePath(emoticons->themePath());
		emoticons->expandEmoticons(doc, data.bg_color);
	}
	text = doc.generateHtml();
	data.message = text;
	
	tipWidget->setData(data);
	tipWidget->prepare();

	// czy nie zasania czego; czy nie wychodzi za pulpit jest i tak sprawdzane w prepare()
	QSize desktopSize = QApplication::desktop()->size();
	if (pos.x() + tipWidget->getWidth() >= desktopSize.width())
		pos.setX(pos.x() - tipWidget->getWidth() - 10);
	if (pos.y() + tipWidget->getHeight() >= desktopSize.height())
		pos.setY(pos.y() - tipWidget->getHeight() - 10);

	tipWidget->setX(pos.x());
	tipWidget->setY(pos.y());
	tipWidget->display();

	kdebugf2();
}

void OSDManager::notify(Notification *notification)
{
	kdebugf();

	addOSD(notification);

	kdebugf2();
}

void OSDManager::copyConfiguration(const QString &fromEvent, const QString &toEvent)
{
	config_file.writeEntry("osdhints", toEvent + "_font",
					config_file.readFontEntry("osdhints", fromEvent + "_font"));
	config_file.writeEntry("osdhints", toEvent + "_fgcolor",
					config_file.readColorEntry("osdhints", fromEvent + "_fgcolor"));
	config_file.writeEntry("osdhints", toEvent + "_bgcolor",
					config_file.readColorEntry("osdhints", fromEvent + "_bgcolor"));
	config_file.writeEntry("osdhints", toEvent + "_bordercolor",
					config_file.readColorEntry("osdhints", fromEvent + "_bordercolor"));
	config_file.writeEntry("osdhints", toEvent + "_timeout",
					(int)config_file.readUnsignedNumEntry("osdhints",  fromEvent + "_timeout"));
	config_file.writeEntry("osdhints", toEvent + "_maskeffect",
					(int)config_file.readUnsignedNumEntry("osdhints",  fromEvent + "_maskeffect"));
	config_file.writeEntry("osdhints", toEvent + "_syntax",
					config_file.readEntry("osdhints", fromEvent + "_syntax"));
}

NotifierConfigurationWidget *OSDManager::createConfigurationWidget(QWidget *parent, char *name)
{
	configurationWidget = new OSDConfigurationWidget(parent, name);
	return configurationWidget;
}

void OSDManager::mainConfigurationWindowCreated(MainConfigurationWindow *mainConfigurationWindow)
{
	kdebugf();
	QCheckBox *setAll = dynamic_cast<QCheckBox *>
			(mainConfigurationWindow->widgetById("osdhints/setAll"));

	connect(setAll, SIGNAL(toggled(bool)),
			   mainConfigurationWindow->widgetById("osdhints/setAllPreview"),
				SLOT(setEnabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
			   mainConfigurationWindow->widgetById("osdhints/setAll_font"),
				SLOT(setEnabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
			   mainConfigurationWindow->widgetById("osdhints/setAll_fgcolor"),
				SLOT(setEnabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
			   mainConfigurationWindow->widgetById("osdhints/setAll_bgcolor"),
				SLOT(setEnabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
			   mainConfigurationWindow->widgetById("osdhints/setAll_bordercolor"),
				SLOT(setEnabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
			   mainConfigurationWindow->widgetById("osdhints/setAll_timeout"),
				SLOT(setEnabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
			   mainConfigurationWindow->widgetById("osdhints/setAll_maskeffect"),
				SLOT(setEnabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)), configurationWidget, SLOT(setAllEnabled(bool)));

	connect(setAll, SIGNAL(toggled(bool)),
				mainConfigurationWindow->widgetById("osdhints/OSDToolTip_font"),
				SLOT(setDisabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
				mainConfigurationWindow->widgetById("osdhints/OSDToolTip_fgcolor"),
				SLOT(setDisabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
				mainConfigurationWindow->widgetById("osdhints/OSDToolTip_bgcolor"),
				SLOT(setDisabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
				mainConfigurationWindow->widgetById("osdhints/OSDToolTip_bordercolor"),
				SLOT(setDisabled(bool)));
	connect(setAll, SIGNAL(toggled(bool)),
				mainConfigurationWindow->widgetById("osdhints/OSDToolTip_maskeffect"),
				SLOT(setDisabled(bool)));
	
	connect(setAll, SIGNAL(toggled(bool)),
				mainConfigurationWindow->widgetById("osdhints/overUserSettingsLabel"),
				SLOT(setShown(bool)));
	
	configurationWidget->setAllEnabled(setAll->isChecked());

	overUserSyntax = mainConfigurationWindow->widgetById("osdhints/overUserSyntax");

	toolTipClassesHighlighted((dynamic_cast<QComboBox *>
			(mainConfigurationWindow->widgetById("toolTipClasses")))->currentText());

	connect(mainConfigurationWindow->widgetById("toolTipClasses"),
				SIGNAL(highlighted(const QString &)), this,
				SLOT(toolTipClassesHighlighted(const QString &)));

	QCheckBox *userPos = dynamic_cast<QCheckBox *>
			(mainConfigurationWindow->widgetById("osdhints/ownPosition"));
	connect(userPos, SIGNAL(toggled(bool)),
			  mainConfigurationWindow->widgetById("osdhints/posx"), SLOT(setEnabled(bool)));
	connect(userPos, SIGNAL(toggled(bool)),
			  mainConfigurationWindow->widgetById("osdhints/posy"), SLOT(setEnabled(bool)));
	connect(userPos, SIGNAL(toggled(bool)),
			  mainConfigurationWindow->widgetById("osdhints/corner"), SLOT(setEnabled(bool)));

	(dynamic_cast<QSpinBox *>(mainConfigurationWindow->widgetById("osdhints/setAll_timeout")))->setSpecialValueText(tr("Dont hide"));

	kdebugf2();
}

void OSDManager::toolTipClassesHighlighted(const QString &value)
{
	kdebugf();
	overUserSyntax->setEnabled(value == qApp->translate("@default", "OSDHints"));
	kdebugf2();
}

void OSDManager::configurationUpdated()
{
	reload();
}

void OSDManager::setDefaultValues()
{
	defaultSyntax =
		"<table border=0 cellspacing=0 cellpadding=0><tr><td align=\"left\" valign=\"middle\">"
		"<img src=\"%&i\" border=0>"
		"</td><td align=\"right\" valign=\"middle\"><b>%&m<br><small>%&d</small>"
		"</td></tr></table>";

	static const QString tooltipSyntax =
		"<b>%a</b><br>[Status:<i>%s</i><br/>][<br/><b>Opis:</b><br/>%d<br/><br/>]"
		"[<i>Telefon:</i><b>%m</b><br/>]";

	static const QString events[8] = { "NewChat", "NewMessage",
							"ConnectionError", "StatusChanged/ToOnline",
							"StatusChanged/ToBusy", "StatusChanged/ToOffline",
							"FileTransfer/IncomingFile", "FileTransfer/Finished" };

	for (int i = 0; i < 8; ++i)
			config_file.addVariable("osdhints", events[i] + "_syntax", defaultSyntax);

	QWidget w;
	config_file.addVariable("osdhints", "MouseOverUserSyntax", tooltipSyntax);
	config_file.addVariable("osdhints", "SetAll", true);
	config_file.addVariable("osdhints", "SetAll_timeout", 10);
	config_file.addVariable("osdhints", "SetAll_bgcolor", w.paletteBackgroundColor());
	config_file.addVariable("osdhints", "SetAll_fgcolor", w.paletteForegroundColor());
	config_file.addVariable("osdhints", "SetAll_bordercolor", w.paletteForegroundColor());
	config_file.addVariable("osdhints", "SetAll_font", defaultFont);
	config_file.addVariable("osdhints", "SetAll_maskeffect", 0);
}

void OSDManager::importConfiguration()
{
	kdebugf();

	int i;
	const QString old_names[12] = {
		"OSDNewMessage", "OSDNewChat", "OSDError", "OSDOffline",
		"OSDInvisible", "OSDBusy", "OSDOnline", "OSDOfflineD",
		"OSDInvisibleD", "OSDBusyD", "OSDOnlineD", "OSDMessage"};
	const QString new_names[7] = {
		"NewMessage", "NewChat", "ConnectionError", "StatusChanged/ToOffline",
		"StatusChanged/ToInvisible", "StatusChanged/ToBusy", "StatusChanged/ToOnline"};

	// sprawdz czy mamy stara skladnie dla nowej wiadomosci i brak nowej
	if (!config_file.readEntry("osdhints", "OSDNewMessage_syntax").isEmpty() &&
			 	config_file.readEntry("osdhints", "NewMessage_syntax").isEmpty()) {

		kdebugm(KDEBUG_INFO, "Old configuration found (and no new). Importing.\n");

		for (i = 0; i < 7; ++i) {
			config_file.writeEntry("osdhints", new_names[i] + "_font",
						  config_file.readFontEntry("osdhints", old_names[i] + "_font"));
			config_file.writeEntry("osdhints", new_names[i] + "_fgcolor",
						  config_file.readColorEntry("osdhints", old_names[i] + "_fgcolor"));
			config_file.writeEntry("osdhints", new_names[i] + "_bgcolor",
						  config_file.readColorEntry("osdhints", old_names[i] + "_bgcolor"));
			config_file.writeEntry("osdhints", new_names[i] + "_bordercolor",
						  config_file.readColorEntry("osdhints", old_names[i] + "_bordercolor"));
			config_file.writeEntry("osdhints", new_names[i] + "_timeout",
						  config_file.readNumEntry("osdhints", old_names[i] + "_timeout"));
			config_file.writeEntry("osdhints", new_names[i] + "_maskeffect",
						  config_file.readNumEntry("osdhints", old_names[i] + "_mask_effect"));
			config_file.writeEntry("osdhints", new_names[i] + "_syntax",
						  config_file.readEntry("osdhints", old_names[i] + "_syntax"));

			//remove the old entries
			config_file.removeVariable("osdhints", old_names[i] + "_font");
			config_file.removeVariable("osdhints", old_names[i] + "_fgcolor");
			config_file.removeVariable("osdhints", old_names[i] + "_bgcolor");
			config_file.removeVariable("osdhints", old_names[i] + "_bordercolor");
			config_file.removeVariable("osdhints", old_names[i] + "_timeout");
			config_file.removeVariable("osdhints", old_names[i] + "_mask_effect");
			config_file.removeVariable("osdhints", old_names[i] + "_syntax");
			config_file.removeVariable("osdhints", old_names[i] + "_icon");
			config_file.removeVariable("osdhints", old_names[i] + "_def_colors");
		}

		for (i = 7; i < 12; ++i) {
			//remove the rest (unportable) entries
			config_file.removeVariable("osdhints", old_names[i] + "_font");
			config_file.removeVariable("osdhints", old_names[i] + "_fgcolor");
			config_file.removeVariable("osdhints", old_names[i] + "_bgcolor");
			config_file.removeVariable("osdhints", old_names[i] + "_bordercolor");
			config_file.removeVariable("osdhints", old_names[i] + "_timeout");
			config_file.removeVariable("osdhints", old_names[i] + "_maskeffect");
			config_file.removeVariable("osdhints", old_names[i] + "_syntax");
			config_file.removeVariable("osdhints", old_names[i] + "_icon");
			config_file.removeVariable("osdhints", old_names[i] + "_def_colors");
		}

		kdebugm(KDEBUG_INFO, "Old configuration imported.\n");
	}

	kdebugf2();
}
