/*
    ziprommetadata.cpp - ZIP File support.

    Copyright (c) 2005      by Michaël Larouche       <michael.larouche@kdemail.net>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/
#include "ziprommetadata.h"

// Kamefu includes
#include "systemmanager.h"
#include "rommetadatafactory.h" // RomMetaData in included in current class header.

// KDE includes
#include <kzip.h>
#include <karchive.h>
#include <kurl.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <kmimetype.h>

namespace Kamefu {

class ZipRomMetaData::Private
{
public:
	Private()
	{}
};

ZipRomMetaData::ZipRomMetaData()
 : RomMetaData(), d(new Private())
{
}


ZipRomMetaData::~ZipRomMetaData()
{
	kdDebug() << k_funcinfo << endl;
	delete d;
	
}

bool ZipRomMetaData::isValidRomFile(const QString &mimeType)
{
	QStringList mimeTypeList = SystemManager::self()->getRegistredMimeTypes();
	QStringList::const_iterator it, itEnd = mimeTypeList.constEnd();

	for(it = mimeTypeList.constBegin(); it != itEnd; ++it)
	{
		if(mimeType == *it)
		{
			return true;
		}
	}

	return false;
}

RomMetaInformation ZipRomMetaData::extractMetaInformation(KFileItem* item)
{
	RomMetaInformation romMetaInfo;

	// Decompress ZIP file.
	KZip zipFile(item->url().path());
	zipFile.open( IO_ReadOnly );

	const KArchiveDirectory *dir = zipFile.directory();
	if(dir)
	{
		// Here we have a limitation, it only catch the latest valid file in the ZIP file.
		// But anyway, a normal ROM ZIP file should containt only one file.
		QStringList entries = dir->entries();
		QStringList::const_iterator it, itEnd = entries.constEnd();
		for(it = entries.constBegin(); it != itEnd; ++it)
		{
			QString currentMimeType;
			currentMimeType = KMimeType::findByPath(*it, 0, true)->name();
			kdDebug() << k_funcinfo << "Found mimetype: " << currentMimeType << endl;
			// Check if the current file is a ROM file.
			if( isValidRomFile(currentMimeType) )
			{
				kdDebug() << k_funcinfo << "We have a valid ROM file in ZIP file " << item->url().path() << endl;

				RomMetaData *metaData = RomMetaDataFactory::self()->createRomMetaData(item->url().path(), currentMimeType );
				
				QIODevice *fileEntryDevice = ((KZipFileEntry*)dir->entry(*it))->device();
				romMetaInfo = metaData->extractMetaInformation( fileEntryDevice );
				delete fileEntryDevice;

				if( romMetaInfo.isValid() )
				{
					// Override those informations to use ZIP file url and name instead.
					// Most emulators can launch directly a ZIP file, that's why this class exists :)
					QString url = item->url().path();
					QString romName = item->url().fileName();
					// Remove extension from romName.
					romName = romName.left(romName.findRev("."));
					romMetaInfo.setSystem( SystemManager::self()->guessSystemNameFromMimeType(currentMimeType) );
					romMetaInfo.setUrl(url);
					romMetaInfo.setRomName(romName);
				}
			}
		}
	}
	zipFile.close();

	return romMetaInfo;
}

}
