/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *   Praveen Kandikuppa : praveen9@gmail.com                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <qlabel.h>
#include <qbuttongroup.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <kdebug.h>
#include <klocale.h>
#include <kprogress.h>
#include <katcatalog.h>
#include <katutils.h>

#include "katindexer.h"
#include "catalogstatusitem.h"

CatalogStatusItem::CatalogStatusItem( QWidget* parent, const char* name )
    : QFrame ( parent, name )
{
    QVBoxLayout *layout1 = new QVBoxLayout( this, KDialog::marginHint(), KDialog::spacingHint(), "layout1" );

    tlCatalogDetail = new QLabel( this, "tlCatalogDetail" );
    tlCatalogDetail->clear();
    layout1->addWidget( tlCatalogDetail );

    tlStatus = new QLabel( this, "tlStatus" );
    tlStatus->setAlignment( int( QLabel::AlignCenter ) );
    tlStatus->setText( i18n( "The indexer is stopped." ) );
    layout1->addWidget( tlStatus );

    tlSubStatus = new QLabel( this, "tlSubStatus" );
    tlSubStatus->setAlignment( int( QLabel::AlignCenter ) );
    tlSubStatus->setText( i18n( "The indexer is stopped." ) );
    layout1->addWidget( tlSubStatus );

    tlCurrentFile = new QLabel( this, "tlCurrentFile" );
    tlCurrentFile->setAlignment( int( QLabel::AlignCenter ) );
    tlCurrentFile->setText( ""  );
    layout1->addWidget( tlCurrentFile );

    pProgress = new KProgress( this, "pProgress" );
    pProgress->setPercentageVisible( true );
    pProgress->setTotalSteps( 100 );
    pProgress->setProgress( 0 );
    pProgress->setTextEnabled( true );
    layout1->addWidget( pProgress );

    bgStats = new QButtonGroup( this, "bgStats" );
    bgStats->setTitle( i18n( "Analyzed" ) );
    layout1->addWidget( bgStats );

    QVBoxLayout* layout3=new QVBoxLayout( bgStats, KDialog::marginHint(), KDialog::spacingHint(), "layout3" );
    QGridLayout* grid1 = new QGridLayout( 0, 2, 4 , KDialog::marginHint(), KDialog::spacingHint(), "grid1" );

    tlFiles = new QLabel( bgStats, "tlFiles" );
    tlFiles->setText( i18n( "Files" ) );
    grid1->addWidget( tlFiles, 0, 0 );

    tlFilesNum = new QLabel( bgStats, "tlFilesNum" );
    tlFilesNum->setAlignment( Qt::AlignRight );
    tlFilesNum->setText( QString::null );
    tlFilesNum->setFrameStyle( QFrame::StyledPanel | QFrame::Plain );
    tlFilesNum->setLineWidth( 1 );
    grid1->addWidget( tlFilesNum, 1, 0 );

    tlFolders = new QLabel( bgStats, "tlFolders" );
    tlFolders->setText( i18n( "Folders" ) );
    grid1->addWidget( tlFolders, 0, 1 );

    tlFoldersNum = new QLabel( bgStats, "tlFoldersNum" );
    tlFoldersNum->setAlignment( Qt::AlignRight );
    tlFoldersNum->setText( QString::null );
    tlFoldersNum->setFrameStyle( QFrame::StyledPanel | QFrame::Plain );
    grid1->addWidget( tlFoldersNum, 1, 1 );

    tlArchives = new QLabel( bgStats, "tlArchives" );
    tlArchives->setText( i18n( "Archives" ) );
    grid1->addWidget( tlArchives, 0, 2 );

    tlArchivesNum = new QLabel( bgStats, "tlArchivesNum" );
    tlArchivesNum->setAlignment( Qt::AlignRight );
    tlArchivesNum->setText( QString::null );
    tlArchivesNum->setFrameStyle( QFrame::StyledPanel | QFrame::Plain );
    grid1->addWidget( tlArchivesNum, 1, 2 );

    tlFileSize = new QLabel( bgStats, "tlFileSize" );
    tlFileSize->setText( i18n( "File size" ) );
    grid1->addWidget( tlFileSize, 0, 3 );

    tlFileSizeNum = new QLabel( bgStats, "tlFileSizeNum" );
    tlFileSizeNum->setAlignment( Qt::AlignRight );
    tlFileSizeNum->setText( QString::null );
    tlFileSizeNum->setFrameStyle( QFrame::StyledPanel | QFrame::Plain );
    grid1->addWidget( tlFileSizeNum, 1, 3 );

    layout3->addLayout( grid1 );

    grid1 = new QGridLayout( 0, 4, 2, "Grid2" );

    tlWords = new QLabel( bgStats, "tlWords" );
    tlWords->setText( i18n( "Words imported:" ) );
    grid1->addWidget( tlWords, 0, 0 );

    tlWordsNum = new QLabel( bgStats, "tlWordsNum" );
    tlWordsNum->setAlignment( Qt::AlignRight );
    tlWordsNum->setText( QString::null );
    grid1->addWidget( tlWordsNum, 0, 1 );

    tlMetaData = new QLabel( bgStats, "tlMetaData" );
    tlMetaData->setText( i18n( "Metadata imported:" ) );
    grid1->addWidget( tlMetaData, 1, 0 );

    tlMetaDataNum = new QLabel( bgStats, "tlMetaDataNum" );
    tlMetaDataNum->setAlignment( QLabel::AlignRight );
    tlMetaDataNum->setText( QString::null );
    grid1->addWidget( tlMetaDataNum, 1, 1 );

    tlFullTexts = new QLabel( bgStats, "tlFullTexts" );
    tlFullTexts->setText( i18n( "Fulltexts imported:" ) );
    grid1->addWidget( tlFullTexts, 2, 0 );

    tlFullTextsNum = new QLabel( bgStats, "tlFullTextsNum" );
    tlFullTextsNum->setAlignment( QLabel::AlignRight );
    tlFullTextsNum->setText( QString::null );
    grid1->addWidget( tlFullTextsNum, 2, 1 );

    tlThumbnails = new QLabel( bgStats, "tlThumbnails" );
    tlThumbnails->setText( i18n( "Thumbnails imported:" ) );
    grid1->addWidget( tlThumbnails, 3, 0 );

    tlThumbnailsNum = new QLabel( bgStats, "tlThumbnailsNum" );
    tlThumbnailsNum->setAlignment( QLabel::AlignRight );
    tlThumbnailsNum->setText( QString::null );
    grid1->addWidget( tlThumbnailsNum, 3, 1 );
    layout3->addLayout( grid1 );

    QHBoxLayout* hLayout = new QHBoxLayout( this, 0, 6 );

    detailButton = new QPushButton( i18n( "&More" ), this );
    pauseButton = new QPushButton( i18n( "&Pause" ), this );
    startButton = new QPushButton( i18n( "&Start" ), this );
    stopButton = new QPushButton( i18n( "&Stop" ), this );

    QSpacerItem* hSpacing = new QSpacerItem( 20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum );

    detailButton->setToggleButton( true );

    hLayout->addWidget( detailButton );
    hLayout->addItem( hSpacing );
    hLayout->addWidget( startButton );
    hLayout->addWidget( pauseButton );
    hLayout->addWidget( stopButton );

    layout1->addLayout( hLayout );

    resize( QSize( 560, 460 ).expandedTo( minimumSizeHint() ) );
    clearWState( WState_Polished );
    setSizePolicy(QSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum));

    setFrameStyle( QFrame::GroupBoxPanel | QFrame::Plain );

    setLineWidth( 1 );

    connect( detailButton, SIGNAL( toggled( bool ) ), this, SLOT( slotDetail( bool ) ) );
    connect( startButton, SIGNAL( clicked() ), this, SIGNAL( start() ) );
    connect( pauseButton, SIGNAL( clicked() ), this, SIGNAL( pause() ) );
    connect( stopButton, SIGNAL( clicked() ), this, SIGNAL( stop() ) );

    slotDetail( false );

    pProgress->hide();
}

CatalogStatusItem::~CatalogStatusItem()
{
}

void CatalogStatusItem::updateButton()
{
    bool state=false;
    //TODO for the futur: disable all button when we don't have catalog
    startButton->setEnabled(state);
    pauseButton->setEnabled(state);
    stopButton->setEnabled(state);
}

void CatalogStatusItem::slotDetail( bool show )
{
    if ( show )
    {
        bgStats->show();
        detailButton->setText( i18n( "&Less" ));
    }
    else
    {
        bgStats->hide();
        detailButton->setText( i18n( "&More" ) );
    }
    adjustSize();
    emit ajustDialogSize();
}

void CatalogStatusItem::slotUpdateCurrentFile ( const QString& file )
{
    tlCurrentFile->setText( file );
}

void CatalogStatusItem::slotUpdateStatus( const int& status, const QString& statusStr )
{
    tlStatus->setText( statusStr );

    tlCurrentFile->clear();

    if ( status == KatIndexer::Stop )
    {
        startButton->setEnabled( true );
        stopButton->setEnabled( false );
        pauseButton->setEnabled( false );
    }
    else if ( status == KatIndexer::Pause )
    {
        // Disable start/stop buttons
        startButton->setEnabled( true );
        stopButton->setEnabled( true );
        pauseButton->setEnabled( false );
    }
    else
    {
        startButton->setEnabled( false );
        stopButton->setEnabled( true );
        pauseButton->setEnabled( true );
    }

    slotUpdateExtractStatus();
}

void CatalogStatusItem::slotUpdateSubStatus( const int&, const QString& statusStr )
{
    if ( statusStr.isEmpty() )
        tlSubStatus->setText( statusStr );
    else
    {
        tlSubStatus->setText( statusStr );
        slotUpdateExtractStatus();
    }
}

void CatalogStatusItem::setCatalog( KatCatalog* cat,
                                    const int& status,
                                    const QString& statusString,
                                    const int& subStatus,
                                    const QString& subStatusString,
                                    const int& progress,
                                    const QString& file )
{
    kdDebug() << "setCatalog called" << endl;
    m_cat = cat;

    QString ctext = m_cat->name() + i18n( " (indexing folder %1)" ).arg(  m_cat->path() );
    tlCatalogDetail->setText( ctext );

    slotUpdateStatus( status, statusString );
    slotUpdateSubStatus( subStatus, subStatusString );

    slotUpdateCurrentFile( file );
    slotUpdateProgress( progress );

    slotDetail( detailButton->isOn() );
}

void CatalogStatusItem::removeCatalog()
{
    m_cat = 0;

    tlCatalogDetail->clear();
    tlStatus->clear();
    tlSubStatus->clear();
    tlCurrentFile->clear();
    pProgress->setProgress( 0 );

    QString zero = QString::number( 0 );

    tlFoldersNum->setText( zero );
    tlFilesNum->setText( zero );
    tlFileSizeNum->setText( zero );
    tlMetaDataNum->setText( zero );
    tlThumbnailsNum->setText( zero );
    tlFullTextsNum->setText( zero );
    tlWordsNum->setText( zero );

    slotDetail( false );
}

void CatalogStatusItem::slotUpdateProgress( int progress )
{
    pProgress->setProgress ( progress );
    if ( ( progress >= 0 ) && ( progress < 100 ) )
        pProgress->show();
    else
        pProgress->hide();
}

void CatalogStatusItem::slotUpdateExtractStatus()
{
    QString folders = QString::number( m_cat->folders() );
    addSeparators( folders );

    QString files = QString::number( m_cat->files() );
    addSeparators( files );

    QString fileSize = QString::number( m_cat->fileSize() );
    addSeparators( fileSize );

    tlFoldersNum->setText( folders );
    tlFilesNum->setText( files );
    tlFileSizeNum->setText( fileSize );

    QString metadata = QString::number( m_cat->metaData() );
    addSeparators( metadata );

    QString thumbnails = QString::number( m_cat->thumbnails() );
    addSeparators( thumbnails );

    QString fulltexts = QString::number( m_cat->fullTexts() );
    addSeparators( fulltexts );

    QString words = QString::number( m_cat->words() );
    addSeparators( words );

    tlMetaDataNum->setText( metadata );
    tlThumbnailsNum->setText( thumbnails );
    tlFullTextsNum->setText( fulltexts );
    tlWordsNum->setText( words );
}

#include "catalogstatusitem.moc"
