/*
 * ps.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Many parts are borrowed from paps  A postscript printing program using pango.
 * Copyright (C) 2002 Dov Grobgeld <dov@imagic.weizmann.ac.il>
 * Paps is distributed under the terms of the GNU Library General Public
 * License
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <cassert>
#include <glibmm/fileutils.h>
#include "ps.hh"
#include "pipe.hh"
#include <iostream>

using std::endl;

PS::PS(Conf& _conf) :
  conf(_conf)
{
  dpi_x = _conf.print_get("dpi_x", 150);
  dpi_y = _conf.print_get("dpi_y", 150);
  width = 0, height = 0;
  top = 0, left = 0, bottom = 0, right = 0;

  pages = 0;
  y_pos = 0;

  pt_to_pixel = 1.0*dpi_x/72;
  pixel_to_pt = 1.0*72/dpi_x;
  orientation = -1;
}

PS::~PS()
{

}

void PS::set_dimensions(int w, int h)
{
  assert(w > 0);
  assert(h > 0);
  width = w;
  height = h;
}

void PS::set_margins(int t, int l, int b, int r)
{
  /*
  assert(t > 0);
  assert(l > 0);
  assert(b > 0);
  assert(r > 0);
  */
  top = t;
  left = l;
  bottom = b;
  right = r;
}

void PS::init()
{
  assert (width > 0);
  assert (height > 0);
  assert (orientation >= 0);
  /*
  assert (top > 0);
  assert (left > 0);
  assert (bottom > 0);
  assert (right > 0);
  */
  PangoContext *_context = pango_ft2_get_context(dpi_x, dpi_y);

  context = Glib::wrap(_context, false);
  std::string print_font = conf.print_get("print_font", "Sans Regular 12");
  font = Pango::FontDescription(print_font);
  context->set_font_description(font);
  text_width = width - left - right;
  text_height = height - top - bottom;

  pango_text_height = text_height * pt_to_pixel * PANGO_SCALE;
}

void PS::header()
{
  ps << "%!PS-Adobe-3.0" << endl
     << "%%Creator: " << PACKAGE << " version " << VERSION << endl
     << "%%Pages: (atend)" << endl
     << "%%BeginProlog" << endl
     << endl
     << "/inch {72 mul} bind def" << endl
     << "/mm {1 inch 25.4 div mul} bind def" << endl
     << endl
     << "% override setpagedevice if it is not defined" << endl
     << "/setpagedevice where {" << endl
     << "    pop % get rid of its dictionary" << endl
     << "    /setpagesize { " << endl
     << "       2 dict begin" << endl
     << "         /pageheight exch def " << endl
     << "         /pagewidth exch def" << endl
     << "         % Exchange pagewidth and pageheight so that pagewidth is bigger" << endl
     << "         pagewidth pageheight gt {  " << endl
     << "             pagewidth" << endl
     << "             /pagewidth pageheight def" << endl
     << "             /pageheight exch def" << endl
     << "         } if" << endl
     << "         2 dict dup /PageSize [pagewidth pageheight] put setpagedevice " << endl
     << "       end" << endl
     << "    } def" << endl
     << "}" << endl
     << "{" << endl
     << "    /setpagesize { pop pop } def" << endl
     << "} ifelse" << endl
     << "% Turn the page around" << endl
     << "/turnpage {" << endl
     << "  90 rotate" << endl
     << "  0 pageheight neg translate" << endl
     << "} def" << endl
     << "% User settings" << endl
     << "/pagewidth " << width << " def" << endl
     << "/pageheight " << height << " def" << endl
     << "/column_width " << text_width << " def" << endl
     << "/bodyheight " << text_height << " def" << endl
     << "/lmarg " << left << " def" << endl
     << "/ytop " << height - top << " def" << endl
     << "/do_landscape " << (orientation ? "true" : "false") << " def" << endl
     << "/dpi_x " << dpi_x << " def" << endl
     << "/dpi_y " << dpi_y << " def" << endl
     << "% Procedures to translate position to first and second column" << endl
     << "/lw 20 def %% whatever" << endl
     << "/setnumcolumns {" << endl
     << "    /numcolumns exch def" << endl
     << "    /firstcolumn { /xpos lmarg def /ypos ytop def} def" << endl
     << "    /nextcolumn { " << endl
     << "      do_separation_line {" << endl
     << "          xpos column_width add gutter_width 2 div add %% x start" << endl
     << "           ytop lw add moveto              % y start" << endl
     << "          0 bodyheight lw add neg rlineto 0 setlinewidth stroke" << endl
     << "      } if" << endl
     << "      /xpos xpos column_width add gutter_width add def " << endl
     << "      /ypos ytop def" << endl
     << "    } def" << endl
     << "} def" << endl
     << endl
     << "/bop {  % Beginning of page definitions" << endl
     << "    gsave" << endl
     << "    do_landscape {turnpage} if " << endl
     << "firstcolumn " << endl
     << "} def" << endl
     << endl
     << "/eop {  % End of page cleanups" << endl
     << "    grestore    " << endl
     << "} def" << endl
     << "1 setnumcolumns" << endl
     << "/conicto {" << endl
     << "    /to_y exch def" << endl
     << "    /to_x exch def" << endl
     << "    /conic_cntrl_y exch def" << endl
     << "    /conic_cntrl_x exch def" << endl
     << "    currentpoint" << endl
     << "    /p0_y exch def" << endl
     << "    /p0_x exch def" << endl
     << "    /p1_x p0_x conic_cntrl_x p0_x sub 2 3 div mul add def" << endl
     << "    /p1_y p0_y conic_cntrl_y p0_y sub 2 3 div mul add def" << endl
     << "    /p2_x p1_x to_x p0_x sub 1 3 div mul add def" << endl
     << "    /p2_y p1_y to_y p0_y sub 1 3 div mul add def" << endl
     << "    p1_x p1_y p2_x p2_y to_x to_y curveto" << endl
     << "} bind def" << endl
     << "/start_ol { gsave 1.1 dpi_x div dup scale} bind def" << endl
     << "/end_ol { closepath fill grestore } bind def" << endl
     << "%%EndPrologue" << endl;
  start_page();
}

void PS::start_page()
{
  pages++;
  ps << "%%Page: " << pages << " " << pages << endl << "bop" << endl;
}

void PS::add_line(std::string& line)
{
  PangoDirection dir;
  Glib::RefPtr<Pango::Layout> layout = Pango::Layout::create (context);

  dir = pango_find_base_dir(line.c_str(), -1);
  if (dir == PANGO_DIRECTION_NEUTRAL)
    dir = PANGO_DIRECTION_LTR;

  layout->set_text(line);

  layout->set_alignment(dir == PANGO_DIRECTION_LTR ? Pango::ALIGN_LEFT : Pango::ALIGN_RIGHT);

  layout->set_width (text_width * pt_to_pixel * PANGO_SCALE);

  GSList *lines = pango_layout_get_lines(layout->gobj());
  //  Pango::SListHandle_ConstLayoutLine lines = layout->get_lines();

  int i = g_slist_length(lines);
  for (int x = 0; x < i; x++)
    {
      PangoLayoutLine *_line = (PangoLayoutLine *)g_slist_nth(lines, x)->data;
      Glib::RefPtr<Pango::LayoutLine> line = Glib::wrap (_line, true);
      Pango::Rectangle logical_rect, ink_rect;
      line->get_extents (ink_rect, logical_rect);

      // Do we need to move to a new page ??
      if (y_pos + logical_rect.get_height() >= pango_text_height)
        {
          y_pos = 0;
          end_page ();
          start_page ();
        }

      draw_line(y_pos + logical_rect.get_height(), line, dir);

      y_pos += logical_rect.get_height();
    }
}

void PS::draw_line(int ypos, Glib::RefPtr<Pango::LayoutLine>& line, PangoDirection dir)
{
  FT_Bitmap bitmap;
  Pango::Rectangle ink_rect, logical_rect;

  line->get_pixel_extents (ink_rect, logical_rect);

  /* Round pixel width to be divisible by 8 */
  int byte_width = (logical_rect.get_width() + 7) / 8 * 8;

  unsigned char *buf = new unsigned char[byte_width * logical_rect.get_height()];

  /* Set up the bitmap to hold the logical_rect */
  memset (buf, 0x00, byte_width * logical_rect.get_height());
  bitmap.rows = logical_rect.get_height();
  bitmap.width = byte_width;
  bitmap.pitch = byte_width;
  bitmap.buffer = buf;
  bitmap.num_grays = 256;
  bitmap.pixel_mode = ft_pixel_mode_grays;

  pango_ft2_render_layout_line (&bitmap, line->gobj(), 0, bitmap.rows * 2 / 3);

 double x_pos = left;
 double _ypos = height - top - ypos / PANGO_SCALE * pixel_to_pt;
 if (bitmap.width > 0 && bitmap.rows > 0)
   {
     if (dir == PANGO_DIRECTION_RTL)
       x_pos += text_width - bitmap.width * pixel_to_pt;
     draw_bitmap (&bitmap, x_pos, _ypos);
   }
 delete[] buf;
}

void PS::draw_bitmap(FT_Bitmap * bitmap, double x, double y)
{
  ps << "gsave" << endl
     << x << " " << y << " translate" << endl
     << "/img_width " << bitmap->width << " def" << endl
     << "/img_height " << bitmap->rows << " def" << endl
     << "/picstr img_width 8 idiv string def" << endl
     << "  img_width 72 dpi_x div mul" << endl
     << "  img_height 72 dpi_y div mul scale" << endl
     << "  0 setgray" << endl
     << "  img_width img_height" << endl
     << "  true" << endl
     << "  [img_width 0 0 img_height neg 0 img_height 0.67 mul]" << endl
     << "  { currentfile" << endl
     << "    picstr readhexstring pop }" << endl
     << "  imagemask";

  /* Output the bitmap in hexadecimal */
  for (int b_idx = 0; b_idx < bitmap->width / 8 * bitmap->rows; b_idx++)
    {
      unsigned char packed_b = 0;

      if (b_idx % (bitmap->width / 8) == 0)
	ps << endl;

      for (int bit_idx = 0; bit_idx < 8; bit_idx++)
        {
          unsigned char this_bit = bitmap->buffer[b_idx * 8 + bit_idx] > 128;
          packed_b = (packed_b << 1) + this_bit;
        }
      // TODO: How to append in hex ?
      char *tmp = g_strdup_printf("%02x", packed_b);
      ps << tmp;
      g_free(tmp);
    }

  ps << endl << "grestore" << endl;
}

void PS::end_page()
{
  ps << "eop" << endl << "showpage" << endl;
}

void PS::footer()
{
  end_page();
  ps << "%%Pages: " << pages << endl << "%%Trailer" << endl << "%%EOF" << endl;
}

bool PS::process(std::string& err)
{
  // TODO: BAD, Is there a way to pass by reference other than this ??
  std::string _ps = ps.str();
  bool st = write(_ps, err);

  ps.clear();

  return st;
}
