/*
 * utils.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <glibmm.h>
#include "utils.hh"
#include <iostream>
#include <unistd.h>
#include <cerrno>
#include <fcntl.h>
#include <gtkmm/textview.h>
#include "macros.h"

std::string Utils::get_data_dir()
{
#ifndef _WIN32
  return std::string(PREFIX + Utils::get_dir_separator() + "share" + Utils::get_dir_separator() + PACKAGE + Utils::get_dir_separator());
#else
  // FIXME:
  return "./";
#endif
}

std::string Utils::get_data_path(char *str)
{
  std::string s = Utils::get_data_dir();
  s += Utils::get_dir_separator();
  s += str;
  return s;
}

std::string Utils::get_dir_separator()
{
  return G_DIR_SEPARATOR_S;
}

std::string Utils::prepend_home_dir(char *str)
{
  return Glib::build_filename(Glib::get_home_dir(), str);
}

bool Utils::is_lam_alef(const Glib::ustring& src, Glib::ustring& res)
{
  res = src;
  bool ret = false;
  ret = is_lam_alef(res, static_cast<gunichar>(0x000fef7));
  ret = (is_lam_alef(res, static_cast<gunichar>(0x000fefb)) == false ? ret : true);
  ret = (is_lam_alef(res, static_cast<gunichar>(0x000fef5)) == false ? ret : true);
  ret = (is_lam_alef(res, static_cast<gunichar>(0x000fef9)) == false ? ret : true);

  return ret;
}

bool inline Utils::is_lam_alef(Glib::ustring& src, gunichar ch)
{
  gunichar repl;
  switch (ch)
    {
    case 0x000fef7:
      repl = 0x0623;
      break;
    case 0x000fefb:
      repl = 0x0627;
      break;
    case 0x000fef5:
      repl = 0x0622;
      break;
    case 0x000fef9:
      repl = 0x0625;
      break;
    }

  bool ret = false;
  Glib::ustring::size_type offset = src.find(ch);

  if (offset != Glib::ustring::npos)
    ret = true;

  while (offset != Glib::ustring::npos) {
    src.replace(offset, 1, 1, static_cast<gunichar>(0x0644));
    src.insert(offset+1, 1, repl);
    offset = src.find(ch, offset);
  }

  return ret;
}

bool Utils::katoob_file_is_writable(std::string& file)
{
  if (Glib::file_test(file, Glib::FILE_TEST_EXISTS))
    return ( access(file.c_str(), W_OK) == 0 ? true : false);
  else
    return true;
}

void Utils::katoob_set_color(Conf& conf, Gtk::Label& label, KatoobColor c) {
  int r = 0, g = 0, b = 0;
  switch (c)
    {
    case KATOOB_COLOR_READONLY:
      {
	r = conf.get("readonly_red", 0);
	g = conf.get("readonly_green", 0);
	b = conf.get("readonly_blue", 65535);
      }
      break;
    case KATOOB_COLOR_MODIFIED:
      {
	r = conf.get("modified_red", 65535);
	g = conf.get("modified_green", 0);
	b = conf.get("modified_blue", 0);
      }
      break;
    default:
      r = conf.get("normal_red", 0);
      g = conf.get("normal_green", 0);
      b = conf.get("normal_blue", 0);
    }
  Pango::AttrColor color = Pango::Attribute::create_attr_foreground(r, g, b);
  Pango::AttrList attrs;
  attrs.insert(color);
  label.set_attributes(attrs);
}

void Utils::katoob_set_color(Conf& cf, Gtk::Label *l, KatoobColor c) {
  katoob_set_color(cf, *l, c);
}

bool Utils::katoob_write(Conf& conf, std::string& file, std::string& text, std::string& error)
{
  // TODO: I'll do it the gtk style until glibmm wraps file_set_contents().
  gchar *f = NULL;
  GError *er = NULL;

  if (Glib::file_test(file, Glib::FILE_TEST_IS_SYMLINK))
    {
      f = g_file_read_link(file.c_str(), &er);
      if (!f)
	{
	  error = er->message;
	  g_error_free(er);
	  return false;
	}
    }

  bool exists = Glib::file_test(f ? f : file.c_str(), Glib::FILE_TEST_EXISTS);
  struct stat buf;
  int stat_error = -1;

  if (exists) {
    stat_error = stat(f ? f : file.c_str(), &buf);
    if (stat_error)
      std::cerr << std::strerror(errno) << std::endl;

    // Let's take the backup first.
    if (conf.get("backup", true))
      {
	std::string new_path = f ? f : file.c_str() + conf.get("backup_ext", "~");
	if (!file_copy(f ? f : file.c_str(), new_path.c_str(), error))
	  {
	    if (f)
	      g_free(f);
	    return false;
	  }
	if (!stat_error)
	  katoob_set_perms(new_path.c_str(), buf);
      }
  }

  if (g_file_set_contents(f ? f : file.c_str(), text.c_str(), text.size(), &er))
    {
      if (!stat_error)
	katoob_set_perms(f ? f : file.c_str(), buf);

      if (f)
	g_free(f);
      return true;
    }

  if (f)
    g_free(f);

  error = er->message;
  g_error_free(er);
  return false;
}

bool Utils::katoob_read(std::string& file, std::string& out)
{
  if (Glib::file_test(file, Glib::FILE_TEST_IS_DIR))
  {
    out = file + _(" Is a directory.");
    return false;
  }

  try {
    out = Glib::file_get_contents(file);
    return true;
  }
  catch (Glib::FileError& err)
    {
      out = err.what();
      return false;
    }
}

void Utils::katoob_set_perms(const char *file, const struct stat& buff)
{
  chmod(file, buff.st_mode);
}

bool Utils::file_copy(const char *f1, const char *f2, std::string& error)
{
  int ifd, ofd;
  ifd = open(f1, O_RDONLY);
  if (ifd == -1)
    {
      error = std::strerror(errno);
      return false;
    }

  ofd = open(f2, O_WRONLY|O_CREAT|O_TRUNC);
  if (ofd == -1)
    {
      error = std::strerror(errno);
      close(ifd);
      return false;
    }

  char buff[1024];
  ssize_t rfd, wfd;

  while (true)
    {
      rfd = read(ifd, buff, 1024);
      if (rfd == 0)
	{
	  break;
	}
      if (rfd == -1)
	{
	  error = std::strerror(errno);
	  close(ifd);
	  close(ofd);
	  return false;
	}
      wfd = write(ofd, buff, rfd);
      if (wfd == -1)
	{
	  error = std::strerror(errno);
	  close(ifd);
	  close(ofd);
	  return false;
	}
    }
  close(ifd);
  close(ofd);
  return true;
}

std::string Utils::substitute(const std::string& orig, const int n)
{
  gchar *tmp_str = g_strdup_printf(orig.c_str(), n);
  std::string final_str(tmp_str);
  g_free(tmp_str);
  return final_str;
}

std::string Utils::substitute(const std::string& orig, const std::string& arg)
{
  gchar *tmp_str = g_strdup_printf(orig.c_str(), arg.c_str());
  std::string final_str(tmp_str);
  g_free(tmp_str);
  return final_str;
}

std::string Utils::substitute(const std::string& orig, const std::string& a, const std::string& b)
{
  gchar *tmp_str = g_strdup_printf(orig.c_str(), a.c_str(), b.c_str());
  std::string final_str(tmp_str);
  g_free(tmp_str);
  return final_str;
}

std::string Utils::substitute(const std::string& orig, const int a, const int b)
{
  gchar *tmp_str = g_strdup_printf(orig.c_str(), a, b);
  std::string final_str(tmp_str);
  g_free(tmp_str);
  return final_str;
}

std::string Utils::substitute(const std::string& orig, const int a, const int b, const std::string& c)
{
  gchar *tmp_str = g_strdup_printf(orig.c_str(), a, b, c.c_str());
  std::string final_str(tmp_str);
  g_free(tmp_str);
  return final_str;
}

std::string Utils::katoob_get_default_font()
{
  Gtk::TextView tv;
  tv.show();
  GtkStyle *s = gtk_rc_get_style(GTK_WIDGET(tv.gobj()));
  char *font = pango_font_description_to_string(s->font_desc);
  std::string _font(font);
  g_free(font);
  return _font;
}
