/*
 * window.hh
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __WINDOW_HH__
#define __WINDOW_HH__

#include <vector>
#include <string>
#include <gtkmm/window.h>
#include <gtkmm/box.h>
#include "menubar.hh"
#include "toolbar.hh"
#include "mdi.hh"
#include "statusbar.hh"
#include "conf.hh"
#include "encodings.hh"
#include "emulator.hh"

class Window : public Gtk::Window, private Emulator
{
public:
  Window(Conf&, Encodings&, std::vector<std::string>&);
  ~Window();

  void about_cb();

  void set_title(char * = NULL);

  void statusbar_cb();
  void toolbar_cb();
  void extended_toolbar_cb();
  void preferences_cb();

  void text_cb();
  void icons_cb();
  void both_cb();
  void beside_cb();

#ifdef HAVE_SPELL
  void auto_spell_cb();
  void on_document_spell_enabled_cb(bool);
  void spell_cb();
#endif
  void quit_cb();

  void wrap_text_cb() { mdi.wrap_text(dynamic_cast<Gtk::CheckMenuItem *>(menubar.wrap_text())->get_active()); }

  void line_numbers_cb() { mdi.line_numbers(dynamic_cast<Gtk::CheckMenuItem *>(menubar.line_numbers())->get_active()); }

  bool on_delete_event (GdkEventAny*);
  sigc::signal<void> signal_quit;

  bool on_key_press_event(GdkEventKey*);

  void on_emulator_activated_cb(int x) { _emulator = x; }
  void on_encoding_activated_cb(int);

#ifdef ENABLE_DBUS
  void open_files(std::vector<std::string>&);
#endif
private:
  void signal_drag_data_received_cb(const Glib::RefPtr<Gdk::DragContext>&, int, int, const Gtk::SelectionData&, guint, guint);
  void signal_document_encoding_changed_cb(int e);
  void signal_document_overwrite_toggled_cb(bool o) { statusbar.set_overwrite(o); }
  void signal_document_cursor_moved_cb(int c, int l) { statusbar.set_position(c, l); }
  void signal_document_file_changed_cb(std::string);
  void signal_document_readonly_cb(int, bool);
  void signal_document_can_redo_cb(bool);
  void signal_document_can_undo_cb(bool);
  void signal_document_modified_cb(int, bool);
  void signal_document_title_changed_cb(std::string, int);
  void signal_document_wrap_text_cb(bool);
  void signal_document_line_numbers_cb(bool);
#ifdef HAVE_SPELL
  void signal_document_dictionary_changed_cb(std::string d) { toolbar.set_dictionary(d); }
#endif
#ifdef ENABLE_HIGHLIGHT
  void signal_document_highlight_cb(int);
#endif
  void preferences_dialog_apply_clicked_cb() { reset_gui(); }
  void on_doc_activated(int);
  void on_reset_gui(int);

  void on_document_added_cb(bool, bool, std::string);
  void on_document_removed_cb(int);

  void dictionary_cb(std::string);
  void search_cb(std::string);

  void reset_gui();

  int _emulator;

  Conf& _conf;
  Encodings& _encodings;

  MenuBar menubar;
  Toolbar toolbar;
  MDI mdi;
  Statusbar statusbar;

  Gtk::VBox box;
#ifdef HAVE_SPELL
  sigc::connection auto_spell_conn;
#endif

  sigc::connection wrap_text_conn;
  sigc::connection line_numbers_conn;
};

#endif /* __WINDOW_HH__ */
