{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module APIDocs where

import Language
import Debug.Trace
import System.Directory
import List
import Module

xmlise :: String -> String
xmlise [] = []
xmlise ('<':cs) = "&lt;"++(xmlise cs)
xmlise (c:cs) = c:(xmlise cs)

showfname (NS s n) = showfname n
showfname x = showuser x

writeXMLDocs :: FilePath -> Program -> String -> IO ()
writeXMLDocs fn prog inf
  = do let str = mkXMLDocs prog inf
       writeFile fn str

mkXMLDocs :: Program -> String -> String
mkXMLDocs [] f = ""
mkXMLDocs ((FunBind (f,l,n,ty,fopts,Defined def) comm):xs) inf 
       | inf == f && (elem Public fopts) =
	   fundoc f l n ty def comm ++ "\n" ++ mkXMLDocs xs inf
mkXMLDocs ((DataDecl f l dopts n tys cons comm):xs) inf 
       | inf == f =
	   datadoc f l n tys cons comm ++ "\n" ++ mkXMLDocs xs inf
mkXMLDocs (x:xs) inf = mkXMLDocs xs inf

data FunInfo = Fun Name -- Name
	           String -- Synopsis
		   String -- Comment
data DataInfo = Data Name -- Name
                     [String] -- Params
		     [String] -- Constructor synopses
		     String -- Comment

writeHTMLDocs :: FilePath -> Program -> String -> IO ()
writeHTMLDocs fn prog inf
  = do let (fi,di) = mkHTMLDocs prog inf
       let funinfo = sortBy (\ (Fun n1 _ _) (Fun n2 _ _) -> 
			     compare (showfname n1) (showfname n2)) fi
       let datainfo = sortBy (\ (Data n1 _ _ _) (Data n2 _ _ _) ->
			     compare (showfname n1) (showfname n2)) di
       let str = header inf ++ 
		 "<h2>Data Type Summary</h2>\n" ++
		 dsummary datainfo ++ 
		 "<h2>Function Summary</h2>\n" ++
		 fsummary funinfo ++ 
		 "<h2>Data Type Details</h2>\n" ++
		 ddetails datainfo ++ 
		 "<h2>Function Details</h2>\n" ++
		 fdetails funinfo ++
		 footer
       writeFile fn str
  where header inf = "<html><head><title>" ++ inf ++ " API reference" ++ 
		     "</title></head><body><h1>" ++
		     inf ++ " API reference</h1>\n"
	footer = "</body></html>"

getFirstLine cs = head (lines cs)

dsummary [] = ""
dsummary ((Data n ps _ comm):xs) =
    "data <a href=\"#"++showfname n++"\">" ++
    showuser n ++ "</a>" ++ showlist ps ++ fl comm ++ "<br>\n" ++
    dsummary xs
  where showlist [] = ""
	showlist xs = "&lt;"++sl' xs++">"
	sl' [x] = x
	sl' (x:xs) = x ++ "," ++ sl' xs
	fl "" = ""
	fl c = " - <em>" ++ getFirstLine c ++ "</em>"

ddetails [] = ""
ddetails ((Data n ps cons comm):xs) = 
    "<hr><a name=\""++showfname n++"\">\n" ++
    "<h3>data "++showuser n ++ showlist ps ++ "</h3>\n" ++
    "<p><strong>Constructors</strong><ul>" ++ showcons cons ++ "</ul>\n" ++
    "<p>"++comm++"</p>\n" ++ ddetails xs
  where showlist [] = ""
	showlist xs = "&lt;"++sl' xs++">"
	sl' [x] = x
	sl' (x:xs) = x ++ "," ++ sl' xs
	showcons [] = ""
	showcons (x:xs) = "<li>"++x++"</li>\n"++showcons xs

fsummary [] = ""
fsummary ((Fun n syn comm):xs) = syn ++ fl comm ++ "<br>\n" ++
				 fsummary xs
  where	fl "" = ""
	fl c = " - <em>" ++ getFirstLine c ++ "</em>"
		
fdetails [] = ""
fdetails ((Fun n syn comm):xs) = "<hr><a name=\""++showfname n++"\">\n" ++
				 "<h3>"++showuser n++"</h3>\n" ++
				 "<p><strong>" ++ syn++"</strong></p>\n" ++
				 "<p>"++comm++"</p>\n" ++
				 fdetails xs

mkHTMLDocs :: Program -> String -> ([FunInfo],[DataInfo])
mkHTMLDocs [] f = ([],[])
mkHTMLDocs ((FunBind (f,l,n,ty,fopts,Defined def) comm):xs) inf 
       | inf == f && (elem Public fopts) =
	   let (fi,di) = mkHTMLDocs xs inf in
	   ((funhtml f l n ty def comm):fi,di)
mkHTMLDocs ((DataDecl f l dopts n tys cons comm):xs) inf 
       | inf == f =
	   let (fi,di) = mkHTMLDocs xs inf in
	   (fi,(datahtml f l n tys cons comm):di)
mkHTMLDocs (x:xs) inf = mkHTMLDocs xs inf

fundoc :: String -> Int -> Name -> Type -> Expr Name -> String -> String
fundoc f l n ty def comm = "<name>"++showfname n++"</name>\n"++
			   showcomment comm++
			   "<location><file>"++f++"</file>"++
			   "<line>"++show l++"</line></location>\n"++
			   "<return>"++xmlise (show (rettype ty))++
			   "</return>\n"++
			   listargs def (argtype ty) ++ 
--			   "<comment>"++show comm++"</comment>"++
			   "</fun>\n"
   where showcomment "" = ""
	 showcomment comm = "<comment>"++show comm++"</comment>\n"
  	 rettype (Fn _ _ ty) = ty
	 argtype (Fn _ args _) = args

         listargs (Lambda tys _) args = la' tys args
	 listargs _ _ = ""
	 la' [] [] = ""
	 la' ((n,_):tys) (ty:tys') = 
	     "<arg><argname>" ++ showuser n ++ "</argname><type>" ++
	     xmlise (show ty) ++ "</type></arg>" ++ la' tys tys'

datadoc :: String -> Int -> Name -> [Type] -> [ConDecl] -> String -> String
datadoc f l n tys cons comm = "<data><name>"++showuser n++"</name>\n"++
			      showcomment comm ++
			      "<location><file>"++f++"</file>"++
			      "<line>"++show l++"</line></location>\n"++
			      listparams tys ++ "\n" ++
			      listcons cons ++
--			      "<comment>"++show comm++"</comment>"++
			      "</data>\n"
   where showcomment "" = ""
	 showcomment comm = "<comment>"++show comm++"</comment>\n"
         listparams [] = ""
	 listparams (p:ps) = "<param>"++show p++"</param>" ++ listparams ps

         listcons [] = ""
	 listcons (c:cs) = showcon c ++ listcons cs

	 argtype (Fn _ args _) = args
         showcon (Con n ty ns _) =
	     "<con><conname>"++showfname n++"</conname>"++
	     showconargs (argtype ty) ns ++
	     "</con>\n"
	 showconargs [] [] = ""
	 showconargs (a:as) (None:ns) = 
	     "<arg><type>" ++
	     xmlise (show a) ++ "</type></arg>" ++ showconargs as ns
	 showconargs (a:as) (n:ns) = 
	     "<arg><argname>"++showuser n++"</argname><type>" ++
	     xmlise (show a) ++ "</type></arg>" ++ showconargs as ns

funhtml :: String -> Int -> Name -> Type -> Expr Name -> String -> FunInfo
funhtml f l n ty def comm = Fun n synopsis comm
   where synopsis = xmlise (show (rettype ty)) ++ " " ++
		    "<a href=\"#"++showfname n++"\">" ++
		    showfname n ++ "</a>(" ++
		    showlist (synargs def (argtype ty) (defvals ty)) ++ ")"

         showlist [] = ""
	 showlist [x] = x
	 showlist (x:xs) = x ++ "," ++ showlist xs

  	 rettype (Fn _ _ ty) = ty
	 argtype (Fn _ args _) = args
	 defvals (Fn defs _ _) = defs

         synargs (Lambda tys _) args defs = la' tys args defs
	 synargs _ _ _ = []
	 la' [] [] [] = []
	 la' ((n,_):tys) (ty:tys') (def:defs) = 
	     (xmlise (show ty) ++ " " ++ showuser n ++showdef def):
	       (la' tys tys' defs)
	 showdef Nothing = ""
	 showdef (Just d) = " = " ++ (stripat (showexp d))
	 stripat ('@':xs) = xs
	 stripat xs = xs

datahtml :: String -> Int -> Name -> [Type] -> [ConDecl] -> String -> DataInfo
datahtml f l n tys cons comm = 
    Data n (listparams tys) (listcons cons) comm
   where listparams [] = []
	 listparams (p:ps) = (show p):(listparams ps)

         listcons [] = []
	 listcons ((Con n ty ns _):cs) = (synopsis n ty ns):(listcons cs)

         showlist [] = ""
	 showlist [x] = x
	 showlist (x:xs) = x ++ "," ++ showlist xs

         synopsis n ty ns 
		  | argtype ty == [] = showfname n
		  | otherwise = showfname n ++ "(" ++
				showlist (conargs (argtype ty) ns) ++ ")"

	 argtype (Fn _ args _) = args

	 conargs [] [] = []
	 conargs (a:as) (None:ns) = (xmlise (show a)):(conargs as ns)
	 conargs (a:as) (n:ns) = 
	     (xmlise (show a) ++ " " ++ showuser n):(conargs as ns)
