{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module Driver(compileFile) where

import Parser
import Language
-- import Typecheck
import Inference
import TAC
import CodegenCPP
import Module
import APIDocs
import Options
import LambdaLift
import Optimise

import Lib

import System
import System.Directory
import Portability
import System.Random
import IO
import Debug.Trace
import List

-- Compile a file, with extra options to gcc, and main options
-- Also chase modules.
compileFile :: FilePath -> [FilePath] -> [String] -> Options -> IO [FilePath]
compileFile fn done extra opts 
	    | fn `elem` done = return done
	    | otherwise =
    do 
       let libdirs = getlibdir opts ("./":libdir:(libdir++"/imports/"):[])
       pinput <- if (doprelude opts) 
		     then do
			    prelude <- 
				do foo <- Module.findFile libdirs "Prelude.ki" 
				   case foo of
				     Nothing -> return ""
				     (Just p) -> return p
			    return (parseprog "Prelude" libdirs prelude "Prelude.ki")
		     else return $ Success []
       prog <- readFile fn
       let imppt = parse (getroot fn) libdirs prog fn
	  -- Chase modules, then get the current module name.
       (done',modname) <- 
	   chaseModules (nochase opts) imppt (fn:done) libdirs extra opts
       let (UN newroot) = modname
	  -- Parse properly, using the real module name, and startup code.
	  -- (This is a bit of a hack, but at least the last parse was mostly 
	  -- lazy)
       startup <- getStartup (getInputtype imppt) libdirs
       let pt = addToPT (parse newroot libdirs (prog++startup) fn) pinput 
       needed <- compileNeeded pt fn
       if (forcebuild opts || needed) 
	  then do compile newroot libdirs opts pt extra
		  return done'
	  else do showIfTrue (not ((nochase opts) || (quiet opts))) $ "Skipping "++fn
		  return done'
  where getroot ".k" = ""
	getroot (x:xs) = x:(getroot xs)
	getroot [] = ""
        getInputtype (Success (PR t _ _)) = t
	getInputtype _ = Module

compileNeeded :: Result ParseResult -> String -> IO Bool
compileNeeded (Failure _ _ _) _ = return True
-- Always compile the main program (since this links...)
compileNeeded (Success (PR Program mod _)) infile = return True
compileNeeded (Success (PR Shebang mod _)) infile = return True
compileNeeded (Success (PR Webapp mod _)) infile = return True
compileNeeded (Success (PR t mod _)) infile =
    do let outfile = outputfile t mod
       ex <- doesFileExist outfile	  
       if (not ex) then return True
	  else do inmod <- getModificationTime infile
		  outmod <- getModificationTime outfile
		  return (inmod>outmod)

-- Compile all the imported modules, if necessary.
chaseModules :: Bool -> Result ParseResult -> [FilePath] -> [FilePath] -> 
		[String] -> Options -> IO ([FilePath], Name)
-- Since this is a module, the first case can't happen.
chaseModules _ (Failure _ _ _) done _ _ _ = return (done, (UN "error"))
chaseModules True (Success (PR _ mod pt)) _ _ _ _ = return ([], mod)
chaseModules False (Success (PR _ mod pt)) done libdirs extra opts
    = do fns <- compileMods pt done libdirs extra opts
	 return (fns,mod)

compileMods :: [RawDecl] -> [FilePath] ->
	       [FilePath] -> [String] -> Options -> IO [FilePath]
compileMods [] done _ extra opts = return done
compileMods ((SearchImport x):xs) done libdirs extra opts
    = do catch (do fn <- findLib [""] (x++".k")
		   done' <- compileFile fn done extra opts
		   compileMods xs done' libdirs extra opts)
	       (\e -> do fn <- findLib libdirs (x++".ki") -- Make sure the ki exists at least!
		         compileMods xs done libdirs extra opts)
compileMods (x:xs) done libdirs extra opts 
    = compileMods xs done libdirs extra opts

outputfile Module mod = showuser mod ++ ".o"
outputfile Program mod = showuser mod
outputfile Shebang mod = showuser mod
outputfile Webapp mod = showuser mod ++ ".cgi"

compile :: String -> [FilePath] -> Options -> 
	   (Result ParseResult) -> [String] -> IO ()
compile root libdirs opts (Failure err file line) extra 
    = do putStrLn err
	 exitWith (ExitFailure 1)
compile root libdirs opts prog@(Success (PR t mod xs)) extra = 
    do dump (dumpraw opts) xs
       showIfTrue (not ((nochase opts) || (quiet opts))) $ "Compiling " ++show t ++ " " ++ root
       case (inferAll (UN root) [] [] [] [] [] (dumpeqns opts) xs) of
	    Success (ok,ctxt,_,_,_) -> do let ok' = if (doprelude opts)
						       then (Imported "Prelude"):ok
						       else ok
					  comp' t root libdirs opts ctxt
					       (nub (lambdalift ok')) extra mod
	    Failure err f l -> do putStrLn err
				  exitWith (ExitFailure 1)
  where	dump True raw = putStr $ concat (map ((\x -> x++"\n\n").show) raw)
	dump False raw = return ()

comp' t root libdirs opts ctxt ok extra mod = do
       optlist <- useOpts allOptNames
       let optimised = if (noopts opts) then ok
	                  else optAll optlist ok
       dumppt (dumptree opts) optimised
       case (compileAll optimised mod) of
          Failure err f l -> do putStrLn err
				exitWith (ExitFailure 1)
	  Success comp -> do
	     let dynlinks = getdynlinks opts []
	     dlinks <- linkfiles libdirs dynlinks
	     (ofiles,linkopts) <- case t of 
		   Module -> return ([],[])
		   _ -> getObjs ok libdirs dlinks
	     let ifile = root ++ ".ki"
	     let xfile = root ++ ".xml"
	     let hfile = root ++ ".html"
	     doWriteIface t ifile ok
	     doWriteXMLDocs (xmldocs opts) xfile ok (root++".k")
	     doWriteHTMLDocs (htmldocs opts) hfile ok (root++".k")
--       putStrLn (show ok)
--       let name = tmpdir++root++".vcc"
--       putStrLn $ tmpdir++ " is tmp"
	     (tmpn,tmph) <- tempfile
--       putStrLn $ tmpn++ " is file"
	     deskey <- case getseed opts of
		    Nothing -> mkdeskey 0
		    (Just x) -> mkdeskey (hash x)
	     ivec <- case getseed opts of
		    Nothing -> mkivec 0
		    (Just x) -> mkivec (hash (x++"ivec"))
	     dump (dumptac opts) comp
    -- FIXME: (UN root) really ought to be mod, but I need to fix the
    -- parser to update the module name when it discovers what it is.
	     writeC (UN root) t libdirs ctxt 
			(addfnmap t optimised comp [deskey,ivec]) tmph
	     let doprofile = if (profile opts) then "-g -pg " else ""
	     let cmd = "g++ " ++ addc t ++ " " ++ 
		       " -O2 -foptimize-sibling-calls " ++ 
		       --		 "-g " ++
		       doprofile ++
		       "-x c++ " ++
		       tmpn ++ " -x none -o " ++ 
		       outputfile t mod ++ " " ++ (unwords extra) ++ " " ++ 
		       incl libdirs ++ 
		       showlist ofiles ++ " " ++ showlist linkopts ++ " " ++
		       dolink t
	     case showgcc opts of
		 True -> putStrLn cmd
		 False -> return ()
	     exit <- system cmd
	     copyc (keepc opts) tmpn root
	     removeFile tmpn
	     if (exit /= ExitSuccess) 
		then exitWith exit
		else return ()
  where addc Module = "-c"
	addc _ = ""
	dolink Module = ""
	dolink _ = " -lgc -L"++libdir++" -lvm -Xlinker -rpath -Xlinker "++libdir
        doWriteIface Module ifile ok = writeIface ifile ok
	doWriteIface _ _ _ = return ()
	doWriteXMLDocs True dfile ok inf = writeXMLDocs dfile ok inf
	doWriteXMLDocs _ _ _ _ = return ()
	doWriteHTMLDocs True dfile ok inf = writeHTMLDocs dfile ok inf
	doWriteHTMLDocs _ _ _ _ = return ()
	addfnmap Webapp ok comp dk = comp++((mkfnmap ok):dk)
	addfnmap Program ok comp dk = comp++((mkfnmap ok):dk)
	addfnmap Shebang ok comp dk = comp++((mkfnmap ok):dk)
	addfnmap Module ok comp dk = comp
	dump True tac = putStr $ showtac tac
	dump False tac = return ()
	dumppt True tree = putStr $ showtree tree
	dumppt False tree = return ()
	copyc True n cfn = do foo <- system $ "cp " ++ n ++ " " ++ cfn ++ ".cc"
			      return ()
	copyc False _ _ = return ()
	showlist [] = " "
	showlist (x:xs) = x ++ " " ++ showlist xs
	incl [] = ""
	incl (x:xs) = "-I"++x++" -L"++x++" "++incl xs


hash [] = 0
hash (x:xs) = 131*(hash xs)+(fromEnum x)

showIfTrue :: Bool -> String -> IO ()
showIfTrue True str = putStrLn str
showIfTrue False _ = return ()
