{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module Inference where

-- Type inference algorithm

import Language
import Debug.Trace
import List

-- Step one : insert type declarations (with UnknownType) so that at least
-- we know what all the variables are.

mkpub (x,y) = (x,(y,[Public]))

insertdecl :: GContext -> Context -> Raw -> Raw
-- Make sure we insert *below* the top level lambda, or we'll have problems.
insertdecl globs ctxt (RLambda str l tvs ns exp) =
      (RLambda str l tvs ns (insertdecl globs 
			     (ctxt++(map mkpub ns)) exp))
insertdecl globs ctxt t = 
    declare (nubBy cmpnames (gu ((map fst ctxt)++(map fst globs)) t)) t
 -- Pass through list of vars declared
 where
   cmpnames (_,_,x) (_,_,y) = x==y
   gu ds (RVar str l n) = checkdecl (str,l,n) ds []
   gu ds (RLambda str l tvs ns exp) = 
         gu (ds++(map fst ns)) exp
   gu ds (RClosure str l ns exp) = 
         gu (ds++(map fst ns)) exp
   gu ds (RDeclare str l n ty exp) =
         gu (n:ds) exp
   gu ds (RBind str l n ty val exp) =
         {- gu ds val ++-} gu (n:ds) exp
   -- We want to check the rhs of an assignment, but not if it's a name 
   gu ds (RAssign str l lval (RVar _ _ _)) 
       = checkdecl (getlv lval) ds [] --(gu ds exp)
   gu ds (RAssign str l lval exp) = checkdecl (getlv lval) ds (gu ds exp)
   gu ds (RAssignOp str l op lval exp) = 
       checkdecl (getlv lval) ds [] --(gu ds exp)
   gu ds (RSeq str l lseq rseq) =
       gu ds lseq ++ gu ds rseq
   gu ds (RReturn _ _ (RCase str l v exps)) 
       = {-gu ds v ++-} (concat (map (gua ds) exps))
   gu ds (RReturn str l a) = []
   gu ds (RApply str l fn args) = []
       -- Names in applications must be declared or used already.
       --checkdecl fn ds (concat (map (gu ds) args))
   gu ds (RPartial str l fn args) = []
       -- Names in applications must be declared or used already.
       --checkdecl fn ds (concat (map (gu ds) args))
   gu ds (RForeign str l ty fn args) = []
       -- Names in applications must be declared or used already.
       --checkdecl fn ds (concat (map (gu ds) args))
       --concat (map (gu ds) args)
   gu ds (RWhile str l c body) =
       {-gu ds c ++-} gu ds body
   gu ds (RDoWhile str l body c) =
       {-gu ds c ++-} gu ds body
   gu ds (RFor str l lval t body) =
       checkdecl (getlv lval) ds (gu ds body)
   gu ds (RTryCatch str l tr ca x fin) =
       checkdecl (str,l,x) ds ((gu ds tr) ++ (gu ds ca) ++ (gu ds fin))
   gu ds (RThrow str l a) = []
   gu ds (RExcept str l a b) = []
   gu ds (RPrint str l v) = [] --gu ds v
   gu ds (RInfix str l op x y) = [] -- gu ds x ++ gu ds y
   gu ds (RUnary str l op v) = [] -- gu ds v
   gu ds (RCoerce str l t v) = [] --gu ds v
   gu ds (RCase str l v exps) = {-gu ds v ++-} (concat (map (gua ds) exps))
   gu ds (RIf str l c t f) = {-gu ds c ++-} gu ds t ++ gu ds f
   gu ds (RIndex str l x y) = [] -- gu ds x ++ gu ds y
   gu ds (RArrayInit str l xs) = [] --(concat (map (gu ds) xs))
   gu ds x = []

   gua ds (RAlt str l f as exp) = (cd str l as ds) ++ (gu ds exp)
   cd str l [] ds = []
   cd str l (x:xs) ds = checkdecl (str,l,x) ds (cd str l xs ds)

   checkdecl (f,l,var) ds rest = case var `elem` ds of
				   True -> rest
				   False -> (f,l,var):rest

   getlv (RAName str l x) = (str,l,x)
   getlv (RAIndex str l lv t) = getlv lv
   getlv (RAField str l lv n) = getlv lv

   declare :: [(String,Int,Name)] -> Raw -> Raw
   declare [] t = t
   declare ((str,l,x):xs) t = RDeclare str l x UnknownType (declare xs t)


-- Substitutions and unification (from SPJ87)

type Subst = Name -> Type

-- Apply a substitution to a type
subst :: Subst -> Type -> Type
subst s (Prim t) = Prim t
subst s (Fn ns ts t) = Fn ns (map (subst s) ts) (subst s t)
subst s (Array t) = Array (subst s t)
subst s (User n ts) = User n (map (subst s) ts)
--subst s (Syn n) = Syn n
subst s (TyVar n) = s n
subst s x = x

-- Substitution composition
scomp :: Subst -> Subst -> Subst
scomp s2 s1 tn = subst s2 (s1 tn)

id_subst :: Subst
id_subst tn = TyVar tn

delta :: Name -> Type -> Subst
delta tn t tn' | tn == tn' = t
	       | otherwise = TyVar tn'

-- Extend a substitution with a new one, or fail if there's an error
extend :: Monad m => String -> Int -> Subst -> Name -> Type -> m Subst
extend file line phi tvn (TyVar n) | tvn == n = return phi
extend file line phi tvn t | tvn `elem` (getVars t) = fail $ file++":"++show line++":Unification error - possible infinite type"
			   | otherwise = return {- $ trace ("Extending with " ++ show tvn ++ " -> " ++ show t)-} $ 
					 (scomp $! (delta tvn t)) $! phi

unify :: Monad m => Subst -> (Type, Type, String, Int) -> m Subst
unify phi e@(t1,t2,f,l) = {- trace ("Unifying " ++ show t1 ++ " & " ++ show t2) $ -}
			  unify' phi e
unify' phi ((TyVar tvn),t,f,l) 
    | phitvn == (TyVar tvn) = extend f l phi tvn phit
    | otherwise = unify phi (phitvn,phit,f,l)
   where phitvn = phi tvn
	 phit = subst phi t
unify' phi ((Array t),(Array t'),f,l) = unify phi (t,t',f,l)
unify' phi (t1@(Fn ns ts t),t2@(Fn ns' ts' t'),f,l) 
    = do zls <- (zipfl (t:ts) (t':ts') f l err)
	 unifyl phi zls
  where err = f ++ ":" ++ show l ++ ":Can't unify " ++ show t1 ++ " and "
	      ++ show t2
unify' phi (t1@(User n ts),t2@(User n' ts'),f,l) 
   | n == n' && (length ts == length ts') = do zl <- (zipfl (ts) (ts') f l err)
					       unifyl phi zl
   | otherwise = fail $ err
  where err = f ++ ":"++ show l ++ ":Can't unify " ++ 
	      show t1 ++ " and " ++ show t2
-- Try it the other way...
unify' phi (t,(TyVar tvn),f,l) = unify phi ((TyVar tvn),t,f,l)
-- And now we must have something primitive
unify' phi (t,t',f,l) | t == t' = return phi
		      | otherwise = fail $ f ++ ":" ++ 
				          show l ++ ":Can't unify " ++ 
					  show t ++ " and " ++ show t'

zipfl :: Monad m => [a] -> [b] -> c -> d -> String -> m [(a,b,c,d)]
zipfl [] [] _ _ err = return []
zipfl (x:xs) (y:ys) z w err = do zl <- zipfl xs ys z w err
				 return $ (x,y,z,w):zl
zipfl _ _ _ _ err = fail err

unifyl :: Monad m => Subst -> [(Type,Type,String,Int)] -> m Subst
unifyl phi [] = return phi
unifyl phi (x:xs) = do phi' <- unify phi x
		       unifyl phi' xs

-- Now do type inference/checking. If there's a declaration, work out the
-- type declared.
-- Step two: Create a list of type equations to solve.

{- We record equations as a list of the equations we initially made (so
   they can be dumped for debugging) and build a substitution as we go (for
   the sake of efficiency; we don't want to have to rebuild it) -} 
type Equation = ([(Type,Type, String,Int)], Subst)

ideq = ([],id_subst)

infer :: Monad m => 
	 Name -> -- Module name
	 Context -> -- Global definitions
	 GContext -> -- Global variables
	 Tags -> -- Constructor tags
	 Types -> -- Type information
	 Fields -> -- Field names
	 Raw -> -- Term to check
	 Type -> -- Expected type
	 m (Expr Name, Equation) -- Well typed term, substitution
infer mod ctxt globs tags tys flds t ftype = do 
      (exp',eqn,_) <- ti [] 0 t ideq (Prim Void) mkrexp
      return (exp',eqn)
 where 
   mkrexp = getrtype (appsyn tys ftype)
   getrtype (Fn _ _ ret) = ret

   ti env next (RVar file line n) eqns exp rexp =
   -- reverse the environment to get most recent binding first.
--      trace ("*** " ++ (show n) ++ "\n" ++ (show ctxt) ++"\n") $
      case (lookup n (reverse env)) of
        Nothing -> findfun n
        (Just t) -> do --let (nt, next') = fudgevars t next
                       let (realt, realn) = isapply t (Global n)
		       neweqn <- addEq file line (realt,exp) eqns
		       return (realn, neweqn, next)
     where findfun n
	    = case (ctxtlookup mod n ctxt) of
	      -- TMP HACK!
	        (Failure err f l) -> if take 4 err == "Ambi" || take 4 err == "Can'"
				      then fail $ file++":"++show line ++ ":" ++err
				      else findglob n
		(Success (rn,t)) -> do let (nt, next') = fudgevars t next
				       let (realt, realn) = isapply nt (Global rn)
				       neweqn <- addEq file line (realt,exp) eqns
				       return (realn, neweqn, next')
	   findglob n 
                = case lookup n globs of
		     Nothing -> fail $ tyerror file line ("Unknown name:" ++ showuser n ++ "\n")-- ++ ctxtdump ctxt)
		     (Just (t,i)) -> do
		         neweqn <- addEq file line (t,exp) eqns
			 return (GVar i, neweqn, next)
	   isapply t@(Fn [] [] ty) f = (ty,Apply f [])
	   isapply t f = (t,f)
   ti env next (RQVar file line n) eqns exp rexp =
   -- reverse the environment to get most recent binding first.
      case (lookup n (reverse env)) of
        Nothing -> findfun n
        (Just t) -> do let (nt, next') = fudgevars t next
		       neweqn <- addEq file line (nt,exp) eqns
		       return (Global n, neweqn, next')
      where findfun n = case (ctxtlookup mod n ctxt) of
	      -- TMP HACK!
	        (Failure err f l) -> if take 4 err == "Ambi" || take 4 err == "Can'" 
				      then fail $ file++":"++show line++ ":" ++err
				      else findglob n
	        (Success (rn,t)) -> do let (nt, next') = fudgevars t next
				       neweqn <- addEq file line (nt,exp) eqns
				       return (Global rn, neweqn, next')
	    findglob n 
                = case lookup n globs of
		     Nothing -> fail $ tyerror file line ("Unknown name:" ++ showuser n ++ "\n")-- ++ ctxtdump ctxt)
		     (Just (t,i)) -> do
		         neweqn <- addEq file line (t,exp) eqns
			 return (GVar i, neweqn, next)

   ti env next (RConst file line c) eqns exp rexp = do
      let (cv,ct) = tcConst c
      eqn' <- addEq file line (ct,exp) eqns
      return (cv, eqn', next)
   ti env next (RLambda file line tvs ns body) eqns exp rexp = do
      ns' <- mapM (normalisectx file line tys) ns
      (bcheck,beqns,next') <- ti (env++(map (synctx tys) ns')) next body eqns (Prim Void) rexp 
      let vbody = pToV (map mkpub ns') bcheck 
      return (Lambda ns' vbody, beqns, next')
   ti env next (RClosure file line ns body) eqns exp rexp = do
      (tns,next') <- inserttvs ns next
      ns' <- mapM (normalisectx file line tys) tns
      let newrexp = TyVar (MN ("CLOSRET",next'))
      let closenv = (env++(map (synctx tys) ns'))
      (bcheck,beqns,next'') <- {-trace (show closenv) $-} ti closenv (next'+1) body eqns (Prim Void) newrexp 
      let vbody = pToV (map mkpub closenv) bcheck 
      let clostype = Fn [] (map snd ns') newrexp
      ceqns <- addEq file line (clostype,exp) beqns
      return (Closure ns' newrexp vbody, ceqns, next'')
    where inserttvs [] next = return ([],next)
	  inserttvs ((n,UnknownType):xs) next = do
	      (xs',next') <- inserttvs xs next
	      let newtv = TyVar (MN ("CLOS",next'))
	      return ((n,newtv):xs',next'+1)
	  inserttvs (x:xs) next = do
	      (xs',next') <- inserttvs xs next
	      return ((x:xs'),next')
   ti env next (RBind file line n t v sc) eqns exp rexp = do
      let syntin = appsyn tys t
      synt <- normalise file line tys syntin
      let newenv = env ++ [(n,synt)]
      (vinf,veqns,next') <- ti env (next+1) v eqns synt rexp 
      (scinf,sceqns,next'') <- ti newenv next' sc veqns exp rexp
      let vsc = pToV (map mkpub newenv) scinf
      return (Bind n synt vinf vsc, sceqns, next'')
   ti env next (RDeclare file line n UnknownType sc) eqns exp rexp = do
      let newtv = TyVar (MN ("TYPE",next))
      let newenv = env ++ [(n,newtv)]
      (scinf,sceqns,next') <- ti newenv (next+1) sc eqns exp rexp
      let vsc = pToV (map mkpub newenv) scinf
      return (Declare file line n newtv vsc, sceqns, next')
   ti env next (RDeclare file line n t sc) eqns exp rexp = do
--      let newtv = TyVar (MN ("TYPE",next))
      let synt = appsyn tys t
      let newenv = env ++ [(n,synt)]
      (scinf,sceqns,next') <- ti newenv (next+1) sc eqns exp rexp
      let vsc = pToV (map mkpub newenv) scinf
      return (Declare file line n synt vsc, sceqns, next')
   ti env next (RReturn file line v) eqns exp rexp = do
      (vinf, veqns, next') <- ti env next v eqns rexp rexp
      neweqns <- addEq file line (Prim Void, exp) veqns
      return (Return vinf, neweqns, next')
   ti env next (RAssign file line l v) eqns exp rexp = do
      (lcheck,lt,leqns,next') <- tclval env next l eqns rexp
      (vinf,veqns,next'') <- ti env next' v leqns lt rexp
      neweqns <- addEq file line (Prim Void,exp) veqns
      return (Assign lcheck vinf, neweqns, next'')
   ti env next (RAssignOp file line op l v) eqns exp rexp = do
      (lcheck,lt,leqns,next') <- tclval env next l eqns rexp
      (vinf,veqns,next'') <- ti env next' v leqns lt rexp
      neweqns <- addEq file line (Prim Void,exp) veqns
      return (AssignOp op lcheck vinf, neweqns, next'')
   ti env next (RSeq file line a b) eqns exp  rexp= do
      (ainf, aeqns, next') <- ti env next a eqns (Prim Void) rexp
      (binf, beqns, next'') <- ti env next' b aeqns exp rexp
      return (Seq ainf binf, beqns, next'')
   ti env next (RApply file line f args) eqns exp rexp = do
      -- Change the application so that default args are added
      let args' = insertdefaults args f ctxt
      (achecks, atypes,aeqns,next') <- checkargs args' eqns next 
      let reqtype = Fn [] atypes exp
      (finf,feqns,next'') <- ti env next' f aeqns reqtype rexp
--      (finf,feqns,next'') <- ti env next' (RQVar file line f) aeqns reqtype
      return (Apply finf achecks, feqns, next'')
     where checkargs [] eqns next = return ([],[],eqns,next)
	   checkargs (a:as) eqns next = do
              (achecks,atypes,aeqns,next') <- checkargs as eqns next
	      let newtv = TyVar (MN ("TYPE",next'))
	      (ainf,aeqns',next'') <- ti env (next'+1) a aeqns newtv rexp
	      return (ainf:achecks,newtv:atypes,aeqns',next'')
-- ECB: This is awful. I only *think* I know how it works...
   ti env next (RPartial file line f args) eqns exp rexp = do
      (achecks, atypes,aeqns,next') <- checkargs args eqns next 
      let newtv = TyVar (MN ("TYPE",next'))
      (finf,feqns,next'') <- ti env (next'+1) f aeqns newtv rexp
      phi <- mkSubst feqns
      -- Replace f's real arg types with atype, to get equations right
      let (ftype,numleft) = insertatypes atypes (subst phi newtv)
      -- Work out the type of the whole partial application
      reqtype <- mangleType atypes ftype
      eqns' <- addEq file line (reqtype,exp) feqns
      eqns'' <- addEq file line (ftype,newtv) eqns'
--      (finf,feqns,next'') <- ti env next' (RQVar file line f) aeqns reqtype
      return (Partial finf achecks numleft, eqns'', next'')
     where checkargs [] eqns next = return ([],[],eqns,next)
	   checkargs (a:as) eqns next = do
              (achecks,atypes,aeqns,next') <- checkargs as eqns next
	      let newtv = TyVar (MN ("TYPE",next'))
	      (ainf,aeqns',next'') <- ti env (next'+1) a aeqns newtv rexp
	      return (ainf:achecks,newtv:atypes,aeqns',next'')
	   mangleType atypes (Fn _ args ret) 
	       = do newargs <- remove atypes args
		    return $ Fn [] newargs ret
	   mangleType atypes ret = return $ Fn [] atypes ret

	   remove (a:as) (b:bs) = remove as bs
	   remove [] xs = return xs
	   remove xs [] = fail $ file ++":" ++show line ++ 
			   ":Too many arguments in partial application"

	   insertatypes xs (Fn _ args ret) = 
	       let (newargs,numleft) = ia xs args in
		   (Fn [] newargs ret, numleft)
	   ia [] as = (as, length as)
	   ia (x:xs) (a:as) = let (newargs,numleft) = (ia xs as) in
				  (x:newargs,numleft)
	   ia xs [] = ([], 0)
   ti env next (RForeign file line ty f args) eqns exp rexp = do
      (achecks,aeqns,next') <- checkargs args eqns next 
      neweqns <- addEq file line (appsyn tys ty, exp) aeqns
      return (Foreign (appsyn tys ty) f achecks, neweqns, next')
     where checkargs [] eqns next = return ([],eqns,next)
	   checkargs (a:as) eqns next = do
              (achecks,aeqns,next') <- checkargs as eqns next
	      let newtv = TyVar (MN ("TYPE",next'))
	      (ainf,aeqns',next'') <- ti env (next'+1) a aeqns newtv rexp
	      return ((ainf,newtv):achecks,aeqns',next'')
   ti env next (RWhile file line cond loop) eqns exp rexp = do
      (cinf, ceqns, next') <- ti env next cond eqns (Prim Boolean) rexp
      (linf, leqns, next'') <- ti env next' loop ceqns (Prim Void) rexp
      neweqns <- addEq file line (Prim Void, exp) leqns
      return (While cinf linf, neweqns, next'')
   ti env next (RDoWhile file line loop cond) eqns exp rexp = do
      (cinf, ceqns, next') <- ti env next cond eqns (Prim Boolean) rexp
      (linf, leqns, next'') <- ti env next' loop ceqns (Prim Void) rexp
      neweqns <- addEq file line (Prim Void, exp) leqns
      return (DoWhile linf cinf, neweqns, next'')
   ti env next (RFor file line l vals loop) eqns exp rexp = do
      let counter1 = MN ("counter",0)
      let counter2 = MN ("counter",1)
      (lcheck,lt,leqns,next') <- tclval env next l eqns rexp
      let arraylt = TyVar (MN ("TYPE",next'))
      let env' = env ++ [(counter1,Prim Number),(counter2,arraylt)]
      (vinf, veqns, next'') <- ti env' (next'+1) vals leqns arraylt rexp
      -- if arraylt is a lazy list, add a while loop.
      phi <- mkSubst veqns
      let atype = subst phi arraylt
      case atype of
         Array _ -> do
	   veqns' <- addEq file line (arraylt, Array lt) veqns
	   (linf, leqns, next''') <- ti env' next'' loop veqns' (Prim Void) rexp
	   neweqns <- addEq file line (Prim Void, exp) leqns
	   let counteri = length env
	   let counterj = (length env)+1
	   return (For counteri counterj lcheck vinf linf, neweqns, next''')
         t@(User (UN "LazyArray") _) -> do
           -- reset(vals); while(!end(vals)) { l=getNext(vals); loop }
	   let realcode = -- Ooh, lispy ;)
	           bind cname t vals
		   (seq (apply "reset" [count])
		       (while (not (apply "end" [count]))
			  (seq (assign l (apply "getNext" [count]))
			       loop)))
	   ti env (next+1) realcode eqns exp rexp
      where seq x y = RSeq file line x y
	    bind n t v b = RBind file line n t v b
	    cname = (MN ("lazycount",next))
	    count = RVar file line cname
	    apply x y = RApply file line 
			  (RQVar file line (NS (UN "LazyArray") (UN x))) y
	    while x y = RWhile file line x y
	    not b = RUnary file line Not b
	    assign l v = RAssign file line l v
	    assignname l v = RAssign file line (RAName file line l) v
   ti env next (RTryCatch file line tr ca x fin) eqns exp rexp = do
      (trinf, treqns, next') <- ti env next tr eqns (Prim Void) rexp
      (cainf, caeqns, next'') <- ti env next' ca treqns (Prim Void) rexp
      (xinf, xeqns, next''') <- ti env next'' (RVar file line x) caeqns (Prim Exception) rexp
      (fininf, feqns, next'''') <- ti env next''' fin xeqns (Prim Void) rexp
--      (yinf, yeqns, next'''') <- ti env next''' (RVar file line y) xeqns (Prim Number)
      return (TryCatch trinf cainf xinf fininf, feqns, next'''')
   ti env next (RThrow file line exc) eqns exp rexp = do
      (einf, eeqns, next') <- ti env next exc eqns (Prim Exception) rexp
      return (Throw einf, eeqns, next')
   ti env next (RExcept file line str code) eqns exp rexp = do
      (sinf, seqns, next') <- ti env next str eqns (Prim StringType) rexp
      (cinf, ceqns, next'') <- ti env next' code seqns (Prim Number) rexp
      return (Except sinf cinf, ceqns, next'')
   ti env next (RBreak file line) eqns exp rexp = do
       neweqns <- addEq file line (exp, Prim Void) eqns
       return (Break file line, neweqns, next)
   ti env next (RVoidReturn file line) eqns exp rexp = do
-- Check void return type is expected?
       neweqns <- addEq file line (exp, Prim Void) eqns
       return (VoidReturn, neweqns, next)
   ti env next (RPrint file line r) eqns exp rexp = do
      let rtv = TyVar (MN ("TYPE",next))
      (rinf, reqns, next') <- ti env (next+1) r eqns rtv rexp
      neweqns <- addEq file line (Prim Void,exp) reqns
      return (InferPrint rinf rtv file line, neweqns, next')
   ti env next (RInfix file line op x y) eqns exp rexp = do
      let xtv = TyVar (MN ("TYPE",next))
      (xinf, xeqns, next') <- ti env (next+1) x eqns xtv rexp
      let ytv = TyVar (MN ("TYPE",next'))
      (yinf, yeqns, next'') <- ti env (next'+1) y xeqns ytv rexp
      phi <- mkSubst yeqns
      let xtype = subst phi xtv
      let ytype = subst phi ytv
      neweqns <- addEq file line ((getOpType op xtype ytype),exp) yeqns
      -- Some infix operators must act on variables
      -- of the same type, others can have coercions inserted
      neweqns' <- addOpEq op file line (xtv,ytv) neweqns
      return (InferInfix op xinf yinf (xtype,ytype,(getOpType op xtype ytype)) file line, neweqns', next'')
    where addOpEq Equal f l eq eqns = addEq f l eq eqns
	  addOpEq NEqual f l eq eqns = addEq f l eq eqns
	  addOpEq _ f l eq eqns = return eqns
   ti env next (RUnary file line op x) eqns exp rexp = do
      let xtv = TyVar (MN ("TYPE",next))
      (xinf, xeqns, next') <- ti env (next+1) x eqns xtv rexp
      neweqns <- addEq file line ((getUnaryType op xtv),exp) xeqns
      return (InferUnary op xinf (xtv,getUnaryType op xtv) file line, neweqns, next')
   ti env next (RCoerce file line t v) eqns exp rexp = do
      let vtv = TyVar (MN ("TYPE",next))
      (vinf,veqns,next') <- ti env (next+1) v eqns vtv rexp
      neweqns <- addEq file line (t,exp) veqns
      return (InferCoerce vtv t vinf file line, neweqns, next')
   ti env next (RCase file line v alts) eqns exp rexp = do
      let vtv = TyVar (MN ("TYPE",next))
      (vinf,veqns,next') <- ti env (next+1) v eqns vtv rexp
      (alts,aeqns,next'') <- altinf env next' alts veqns vtv exp rexp
      return (Case vinf alts, aeqns, next'')
   ti env next (RIf file line cond ift iff) eqns exp rexp = do
      (cinf, ceqns, next') <- ti env next cond eqns (Prim Boolean) rexp
      (tinf, teqns, next'') <- ti env next' ift ceqns exp rexp
      (finf, eeqns, next''') <- ti env next'' iff teqns exp rexp
      return (If cinf tinf finf, eeqns, next''')
   ti env next (RIndex file line e i) eqns exp rexp = do
       (einf, eeqns, next') <- ti env next e eqns (Array exp) rexp
       (iinf, ieqns, next'') <- ti env next' i eeqns (Prim Number) rexp
       return (Index einf iinf, ieqns, next'')
   ti env next (RField file line v f) eqns exp rexp = do
       let newtv = TyVar (MN ("FV",next))
       (vinf, veqns, next') <- ti env (next+1) v eqns newtv rexp
       -- Need to know what newtv really is before we go on...
       phi <- mkSubst veqns
       let vtype = subst phi newtv
       let (fudgedt, next'') = fudgevars vtype next'
       (fty, arg, tag) <- getFieldType flds f fudgedt file line
       eqns' <- addEq file line (exp, fty) veqns
       return (Field vinf f arg tag, eqns', next'')
   ti env next (RArrayInit file line xs) eqns exp rexp = do
       let newtv = TyVar (MN ("TYPE",next))
       eqns' <- addEq file line (exp, Array newtv) eqns 
       (xsinf, xseqns, next') <- tis env (next+1) xs eqns' newtv
       return (ArrayInit xsinf, xseqns, next')
     where tis env next [] eqns exp = return ([], eqns, next)
	   tis env next (x:xs) eqns exp = do
	       (xinf,xeqns,next') <- ti env next x eqns exp rexp
	       (xsinf,xseqns,next'') <- tis env next' xs xeqns exp
	       return (xinf:xsinf, xseqns, next'')
   ti env next (RNoop file line) eqns exp rexp = do
       neweqns <- addEq file line (exp, Prim Void) eqns
       return (Noop, neweqns, next)
   ti env next (RVMPtr file line) eqns exp rexp = do
       neweqns <- addEq file line (exp, Prim Pointer) eqns
       return (VMPtr, neweqns, next)
   ti env next (RMetavar f l i) eqns exp rexp = 
       return (Metavar f l i, eqns, next)
       
--   ti env next x eqns exp = error $ show x

   altinf env next [] eqns vt exp rexp = return ([],eqns,next)
   altinf env next ((RAlt file line con args r):xs) eqns vt exp rexp 
    = do
       conname <- ctxtlookup mod con ctxt
       let app = RApply file line (RQVar file line (fst conname)) (map (RVar file line) args)
--       let app = RApply file line con (map (RVar file line) args)
       (ainf,aeqns,next') <- ti env next app eqns vt rexp
       (xsinf, xseqns, next'') <- altinf env next' xs aeqns vt exp rexp
       (rinf,reqns,next''') <- ti env next'' r xseqns exp rexp
       (tag,tot) <- gettag (fst conname)
       return ((Alt tag tot (map Global args) rinf):xsinf,reqns,next''')

   gettag n = case lookup n tags of
	         Nothing -> fail $ "Internal tag error (" ++ showuser n ++ "\n" ++ show tags ++ ")"
		 (Just a) -> return a

   tclval env next (RAName file line n) eqns rexp = do
       let np = getpos n (map mkpub env)
       case (getType n env) of
               Just nt -> return (AName np,(appsyn tys nt),eqns, next)
	       Nothing -> do case getType n globs of
				Just (gt,gid) ->
				    return (AGlob gid,(appsyn tys gt), eqns, next)
				Nothing -> fail $ file++":"++show line++":Need type declaration for name " ++ showuser n
   tclval env next (RAIndex file line r idx) eqns rexp = do
       (iinf,ieqns,next') <- ti env next idx eqns (Prim Number) rexp
       (rcheck,rt,reqns,next'') <- tclval env next' r ieqns rexp
       let newtv = TyVar (MN ("TYPE",next''))
       neweqns <- addEq file line (rt,Array newtv) reqns 
       return (AIndex rcheck iinf, newtv, neweqns,(next''+1))
   tclval env next (RAField file line r f) eqns rexp = do
       (rcheck,rt,reqns,next') <- tclval env next r eqns rexp
       -- n needs to be a valid field for the type of r
       -- get the real type of r
       phi <- mkSubst reqns
       let realrt = subst phi rt
       let (fudgedt, next'') = fudgevars realrt next'
       (fty, arg, tag) <- getFieldType flds f fudgedt file line
       return (AField rcheck f arg tag, (appsyn tys fty), reqns, next'')

-- Need to take possible coercions into account too!
   getOpType Plus x y = (biggert x y)
   getOpType Minus x y = (biggert x y)
   getOpType Times x y = (biggert x y)
   getOpType Divide x y = (biggert x y)
   getOpType Modulo x y = (biggert x y)
   getOpType Power x y = (biggert x y)
   getOpType OpShLeft x y = (biggert x y)
   getOpType OpShRight x y = (biggert x y)
   getOpType OpAnd x y = (biggert x y)
   getOpType OpOr x y = (biggert x y)
   getOpType OpXOR x y = (biggert x y)
   getOpType _ _ _ = Prim Boolean

   getUnaryType Not x = x
   getUnaryType Neg x = x

   tyerror file line x = file ++ ":" ++ show line ++ ":" ++ x

   -- FIXME: Check that the default arguments are valid things
   insertdefaults :: [Raw] -> Raw -> Context -> [Raw]
   insertdefaults args (RQVar _ _ f) ctxt =
       case (ctxtlookup mod f ctxt) of
          Nothing -> args
	  Just (n,(Fn defs _ _)) -> idefs' defs args
   insertdefaults args (RVar _ _ f) ctxt =
       case (ctxtlookup mod f ctxt) of
          Nothing -> args
	  Just (n,(Fn defs _ _)) -> idefs' defs args
   insertdefaults args _ _ = args
   idefs' [] [] = []
   idefs' (x:xs) (y:ys) = y:(idefs' xs ys)
   idefs' (Just v:xs) [] = v:(idefs' xs [])
   idefs' _ ys = ys

addEq :: Monad m => String -> Int -> (Type,Type) -> Equation -> m Equation
--addEq file line (lt,rt) phi = unify phi ((subst phi lt),(subst phi rt),file,line)
addEq file line (lt,rt) (eqn,subst) 
    = case unify subst (lt,rt,file,line) of
          Success subst' -> return ((lt,rt,file,line):eqn,subst')
	  Failure err f l -> fail $ err

getFieldType :: Monad m => Fields -> 
		Name -> -- Field name
		Type -> -- Type to project field from (needs to be concrete)
		String ->
		Int ->
		m (Type, Int, Int) -- Field type, argument and tag.
getFieldType fs f ty file line = {- trace (show fs) $ -}
		       fieldLookup fs f ty file line

fieldLookup :: Monad m => Fields -> Name -> Type -> String -> Int ->
	       m (Type, Int, Int)
fieldLookup [] n ty f l = fail $ f ++ ":" ++ show l ++ ":" ++ showuser n ++ " is an unknown field of " ++ show ty 
fieldLookup (((n,ty),(fty, arg, tag)): xs) fn exp f l
    | n == fn = case (getVarTypes ty exp) of
	          Nothing -> fieldLookup xs fn exp f l
		  (Just phi) -> return ({-trace ("Found " ++ show ty ++ "," ++ show fty ++ "," ++ show (subst phi fty))-} (subst phi fty, arg,tag))
    | otherwise = fieldLookup xs fn exp f l

getpos n xs = getpos' n 0 (-1) xs
getpos' n _ last [] = last
getpos' n i last ((x,(t,_)):xs) | n==x = getpos' n (i+1) i xs
	  			| otherwise = getpos' n (i+1) last xs

pToV :: Context -> Expr Name -> Expr Name
pToV cs (Global n) | getpos n cs >= 0 = (Loc (getpos n cs))
		   | otherwise = (Global n)
pToV cs (Loc l) = Loc l
pToV cs (GVar x) = GVar x
pToV cs (GConst c) = GConst c
pToV cs (Lambda ns sc) = Lambda ns (pToV cs sc)
pToV cs (Closure ns rt sc) = Closure ns rt (pToV cs sc)
pToV cs (Bind n t v sc) = Bind n t (pToV cs v) (pToV (cs++[(n,(t,[Public]))]) sc)
pToV cs (Declare f l n t sc) = Declare f l n t (pToV (cs++[(n,(t,[Public]))]) sc)
pToV cs (Return r) = Return (pToV cs r)
pToV cs (Assign l e) = Assign (pToVlval l) (pToV cs e)
  where pToVlval (AName i) = AName i
	pToVlval (AGlob i) = AGlob i
	pToVlval (AIndex l r) = AIndex (pToVlval l) (pToV cs r)
	pToVlval (AField l n a t) = AField (pToVlval l) n a t
pToV cs (AssignOp op l e) = AssignOp op (pToVlval l) (pToV cs e)
  where pToVlval (AName i) = AName i
	pToVlval (AGlob i) = AGlob i
	pToVlval (AIndex l r) = AIndex (pToVlval l) (pToV cs r)
	pToVlval (AField l n a t) = AField (pToVlval l) n a t
pToV cs (Seq a b) = Seq (pToV cs a) (pToV cs b)
pToV cs (Apply f as) = Apply (pToV cs f) (fmap (pToV cs) as)
pToV cs (Partial f as i) = Partial (pToV cs f) (fmap (pToV cs) as) i
pToV cs (Foreign ty f as) = Foreign ty f 
			    (fmap (\ (x,y) -> ((pToV cs x),y)) as)
pToV cs (While t e) = While (pToV cs t) (pToV cs e)
pToV cs (DoWhile e t) = DoWhile (pToV cs e) (pToV cs t)
pToV cs (For x y l ar e) = For x y (pToVlval l) (pToV cs ar) (pToV cs e)
  where pToVlval (AName i) = AName i
	pToVlval (AIndex l r) = AIndex (pToVlval l) (pToV cs r)
pToV cs (TryCatch e1 e2 n f) = TryCatch (pToV cs e1) (pToV cs e2) 
			                (pToV cs n) (pToV cs f)
pToV cs (Throw e) = Throw (pToV cs e)
pToV cs (Except e1 e2) = Except (pToV cs e1) (pToV cs e2)
pToV cs (InferPrint e t f l) = InferPrint (pToV cs e) t f l
pToV cs (PrintStr e) = PrintStr (pToV cs e)
pToV cs (PrintNum e) = PrintNum (pToV cs e)
pToV cs (PrintExc e) = PrintExc (pToV cs e)
pToV cs (Infix op a b) = Infix op (pToV cs a) (pToV cs b)
pToV cs (InferInfix op a b ts f l) = InferInfix op (pToV cs a) (pToV cs b) ts f l
pToV cs (Append a b) = Append (pToV cs a) (pToV cs b)
pToV cs (Unary op a) = Unary op (pToV cs a)
pToV cs (InferUnary op a ts f l) = InferUnary op (pToV cs a) ts f l
pToV cs (Coerce t1 t2 v) = Coerce t1 t2 (pToV cs v)
pToV cs (InferCoerce t1 t2 v f l) = InferCoerce t1 t2 (pToV cs v) f l
pToV cs (Case t e) = Case (pToV cs t) (pvAlt e)
  where pvAlt [] = []
	pvAlt ((Alt n t exs ex):xs) = (Alt n t (map (pToV cs) exs) (pToV cs ex)):
				      (pvAlt xs)
pToV cs (ArrayInit xs) = ArrayInit (map (pToV cs) xs)
pToV cs (If a t e) = If (pToV cs a) (pToV cs t) (pToV cs e)
pToV cs (Index l es) = Index (pToV cs l) (pToV cs es)
pToV cs (Field v n a t) = Field (pToV cs v) n a t
pToV cs Noop = Noop
pToV cs VMPtr = VMPtr
pToV cs (Break f l) = Break f l
pToV cs VoidReturn = VoidReturn
pToV cs (Metavar f l i) = Metavar f l i

tcConst :: Const -> (Expr Name, Type)
tcConst (Num x) = (GConst (Num x), Prim Number)
tcConst (Ch c) = (GConst (Ch c), Prim Character)
tcConst (Bo b) = (GConst (Bo b), Prim Boolean)
tcConst (Re r) = (GConst (Re r), Prim RealNum)
tcConst (Str s) = (GConst (Str s), Prim StringType)
tcConst (Exc s i) = (GConst (Exc s i), Prim Exception)
tcConst Empty = (GConst Empty, Prim Void)

appsyn tys t = {- trace ((show t) ++ " to " ++ show (syn t)) $ -} syn t where
    syn (Prim p) = (Prim p)
    syn (Fn ns ts t) = Fn ns (map syn ts) (syn t)
    syn (Array t) = Array (syn t)
    syn (User n ts) = User (resolve tys n) (map syn ts)
       where resolve [] n = n
	     resolve ((cn@(NS _ bn),ti):xs) n | bn == n = cn
	     resolve ((x,ti):xs) n | x == n = x
	     resolve (_:xs) n = resolve xs n
--    syn (Syn n) = error "Internal error"
{- case lookup n tys of
		     Nothing -> (TyVar n)
		     (Just x) -> x -}
    syn (TyVar n) = TyVar n

synctx tys (n,ty) = (n,appsyn tys ty)
normalisectx f l tys (n,ty) = do ty' <- normalise f l tys ty
                                 return (n,ty')

checkTy :: Monad m => Types -> Type -> Type -> String -> m ()
checkTy tys x y err = if convert (appsyn tys x) (appsyn tys y)
			     then return ()
			     else fail err


-- Step 3: Given a list of type equations, work out what types the
-- type variables actually are. Do this by unifying all the equations.
-- Do something hairy to maintain a list of all the errors that come up.

mkSubst :: Monad m => Equation -> m Subst
mkSubst (eqs,s) = return s

{-
mkSubst eqs = do phi <- mkS' [] eqs
		 case phi of
		     (x,[]) -> return x
		     (_,errs) -> fail $ concat $ reverse $ map (++"\n") (nub errs)
mkS' errs [] = return (id_subst,[])
mkS' errs ((t1,t2,f,l):xs) = do
    (phi',errs') <- mkS' errs xs
    case unify phi' (t1,t2,f,l) of
        Success newphi -> return (newphi,errs')
	Failure err f l -> return (phi',err:errs')
-}

-- Step 4: Substitute the type equations into the term with the substitution
-- we got from the last step.

substTerm :: Monad m => Subst -> Expr Name -> m (Expr Name)
substTerm phi t = st t
   where
     st (Lambda args exp) = do exp' <- st exp
			       return $ Lambda (lamsubst args) exp'
 	where lamsubst [] = []
 	      lamsubst ((n,ty):xs) = (n,subst phi ty):(lamsubst xs)
     st (Closure args rt exp) = 
	 do exp' <- st exp
	    return $ Closure (lamsubst args) (subst phi rt) exp'
 	where lamsubst [] = []
 	      lamsubst ((n,ty):xs) = (n,subst phi ty):(lamsubst xs)
     st (Bind n ty e1 e2) = do e1' <- st e1
			       e2' <- st e2
			       return $ Bind n (subst phi ty) e1' e2'
     st (Declare f l n ty e) = 
	 do e' <- st e
	    let ty' = subst phi ty
	    case ty' of
	        (Prim Void) ->
		     fail $ f ++":"++ show l ++":"++
			  showuser n ++ " has type Void"
		_ -> return $ Declare f l n ty' e'
     st (Return r) = do r' <- st r
			return $ Return r'
     st (Assign a e) = do e' <- st e
			  a' <- asubst a
			  return $ Assign a' e'
         where asubst (AName x) = return $ AName x
	       asubst (AGlob i) = return $ AGlob i
 	       asubst (AIndex a e) = do a' <- asubst a
					e' <- st e
					return $ AIndex a' e'
	       asubst (AField a n arg t) = do a' <- asubst a
					      return $ AField a' n arg t
     st (AssignOp op a e) = do e' <- st e
			       a' <- asubst a
			       return $ AssignOp op a' e'
         where asubst (AName x) = return $ AName x
	       asubst (AGlob i) = return $ AGlob i
 	       asubst (AIndex a e) = do a' <- asubst a
					e' <- st e
					return $ AIndex a' e'
	       asubst (AField a n arg t) = do a' <- asubst a
					      return $ AField a' n arg t
     st (Seq e1 e2) = do e1' <- st e1
			 e2' <- st e2
			 return $ Seq e1' e2'
     st (Apply e1 args) = do e1' <- st e1
			     args' <- mapM st args
			     return $ Apply e1' args'
     st (Partial e1 args i) 
	   = do e1' <- st e1
		args' <- mapM st args
		return $ Partial e1' args' i
     st (Foreign ty n args) = do args' <- fsubst args
				 return $ Foreign (subst phi ty) n args'
         where fsubst [] = return []
 	       fsubst ((e,ty):xs) = do e' <- st e
				       rest <- fsubst xs
				       return $ (e',subst phi ty):rest
     st (While e1 e2) = do e1' <- st e1
			   e2' <- st e2
			   return $ While e1' e2'
     st (DoWhile e1 e2) = do e1' <- st e1
			     e2' <- st e2
			     return $ DoWhile e1' e2'
     st (For x y a e1 e2) = do e1' <- st e1
			       e2' <- st e2
			       a' <- asubst a
			       return $ For x y a' e1' e2'
         where asubst (AName x) = return $ AName x
 	       asubst (AIndex a e) = do a' <- asubst a
					e' <- st e
					return $ AIndex a' e'
     st (TryCatch t c x f) = do t' <- st t
				c' <- st c
				f' <- st f
				return $ TryCatch t' c' x f'
     st (Throw x) = do x' <- st x
		       return (Throw x')
     st (Except x y) = do x' <- st x
			  y' <- st y
			  return (Except x' y')
     st (InferPrint e t f l) = do 
       e' <- st e
       case (subst phi t) of
            (Prim Number) -> return $ PrintNum e'
	    (Prim StringType) -> return $ PrintStr e'
	    (Prim Exception) -> return $ PrintExc e'
	    _ -> fail $ f ++ ":" ++ show l ++ ":Can't print type " ++ show t
     st (InferInfix op e1 e2 (t1,t2,t3) f l) = do
	 e1' <- st e1
	 e2' <- st e2
	 checkInfix op e1' e2' (subst phi t1) (subst phi t2) (subst phi t3) f l
-- 	InferInfix op (st e1) (st e2) (subst phi t1,subst phi t2,subst phi t3)
-- 		   f l
     st (Append e1 e2) = do
	 e1' <- st e1
	 e2' <- st e2
	 return $ Append e1' e2'
     st (InferUnary op e1 (t1,t2) f l) = do
         e1' <- st e1
	 checkUnary op e1' (subst phi t1) (subst phi t2) f l
-- 	InferUnary op (st e1) (subst phi t1,subst phi t2) f l
     st (InferCoerce t1 t2 e f l) = do 
          e' <- st e
	  checkCoerce (subst phi t1) (subst phi t2) e' f l
     st (Case e1 e2) = do e1' <- st e1
			  e2' <- stalt e2
			  return $ Case e1' e2'
          where stalt [] = return []
		stalt ((Alt n t exps ex):xs) = 
		    do ex' <- st ex
		       exps' <- mapM st exps
		       rest <- stalt xs
		       return $ (Alt n t exps' ex'):rest
     st (ArrayInit e) = do e' <- mapM st e
			   return $ ArrayInit e'
     st (If e1 e2 e3) = do e1' <- st e1
			   e2' <- st e2
			   e3' <- st e3
			   return $ If e1' e2' e3'
     st (Index e1 e2) = do e1' <- st e1
			   e2' <- st e2
			   return $ Index e1' e2'
     st x = return x

checkInfix :: Monad m => Op -> Expr Name -> Expr Name -> 
	                 Type -> Type -> Type -> 
			 String -> Int -> m (Expr Name)
-- If the types are different, try to insert a coercion
checkInfix op l r (Prim x) (Prim y) ret file line
    | x `tlt` y = do col <- implicitCoerce x y l file line
		     checkInfix op col r (Prim y) (Prim y) ret file line
    | y `tlt` x = do cor <- implicitCoerce y x r file line
		     checkInfix op l cor (Prim x) (Prim x) ret file line
checkInfix op l r (Prim StringType) (Prim StringType) (Prim Boolean) file line
  | (op==Equal || op == NEqual) =
    return $ CmpStr op l r
checkInfix Plus l r (Prim StringType) (Prim StringType) (Prim StringType) file line
 = return $ Append l r
checkInfix op l r (Prim Number) (Prim Number) (Prim Number) file line
  | (op==Plus || op==Minus || op==Times || op==Divide || op==Modulo || 
     op==Power || op==OpAnd || op==OpOr || op==OpXOR || 
     op==OpShLeft || op==OpShRight)
      = return $ Infix op l r
checkInfix op l r (Prim RealNum) (Prim RealNum) (Prim RealNum) file line
  | (op==Plus || op==Minus || op==Times || op==Divide || 
     op==Power || op==OpAnd || op==OpOr)
      = return $ RealInfix op l r
checkInfix op l r (Prim Number) (Prim Number) (Prim Boolean) file line
  | (op==Equal || op==NEqual || op==OpLT || op==OpGT ||
     op==OpLE || op==OpGE)
      = return $ Infix op l r
checkInfix op l r (Prim RealNum) (Prim RealNum) (Prim Boolean) file line
  | (op==Equal || op==NEqual || op==OpLT || op==OpGT ||
     op==OpLE || op==OpGE)
      = return $ RealInfix op l r
checkInfix op l r (Prim Character) (Prim Character) (Prim Boolean) file line
  | (op==Equal || op==NEqual || op==OpLT || op==OpGT ||
     op==OpLE || op==OpGE)
      = return $ Infix op l r
checkInfix op l r (Prim Boolean) (Prim Boolean) (Prim Boolean) file line
  | (op==Equal || op==NEqual || op==OpAndBool || op==OpOrBool)
      = return $ Infix op l r
checkInfix op l r (Prim Exception) (Prim Exception) (Prim Boolean) file line
  | (op==Equal || op==NEqual)
      = return $ CmpExcept op l r

-- Defaults for equality, pass through to built in equal function
checkInfix Equal l r t1 t2 (Prim Boolean) file line
      | t1==t2 = return $ Apply (Global eqfun) [l,r]
checkInfix NEqual l r t1 t2 (Prim Boolean) file line
      | t1==t2 = return $ Unary Not (Apply (Global eqfun) [l,r])

checkInfix op l r tl tr ret file line = fail $ file ++ ":" ++ show line ++ ":Can't apply operator '" ++ show op ++ "' to " ++ show tl ++ " and " ++ show tr

eqfun = NS (UN "Builtins") (UN "equal")
--eqfun = (UN "equal")

checkUnary :: Monad m => UnOp -> Expr Name -> 
	                 Type -> Type -> 
			 String -> Int -> m (Expr Name)
checkUnary Not l (Prim Boolean) (Prim Boolean) file line = 
    return $ Unary Not l
checkUnary Neg l (Prim Number) (Prim Number) file line = 
    return $ Unary Neg l
checkUnary Neg l (Prim RealNum) (Prim RealNum) file line = 
    return $ RealUnary Neg l
checkUnary op l tl ret file line = fail $ file ++ ":" ++ show line ++ ":Can't apply operator '" ++ show op ++ "' to " ++ show tl

checkCoerce :: Monad m => Type -> Type -> Expr Name -> String -> Int ->
	       m (Expr Name)
checkCoerce t1 t2 e file line
  | (t1==(Prim Number) && t2==(Prim StringType)) ||
    (t1==(Prim StringType) && t2==(Prim Number)) ||
    (t1==(Prim RealNum) && t2==(Prim StringType)) ||
    (t1==(Prim StringType) && t2==(Prim RealNum)) ||
    (t1==(Prim Character) && t2==(Prim StringType)) ||
    (t1==(Prim Boolean) && t2==(Prim StringType)) ||
    (t1==(Prim RealNum) && t2==(Prim Number)) ||
    (t1==(Prim Number) && t2==(Prim RealNum)) ||
    (t1==(Prim Character) && t2==(Prim Number)) ||
    (t1==(Prim Number) && t2==(Prim Character))
	= return $ Coerce t1 t2 e
  | otherwise = fail $ file ++ ":" ++ show line ++ ":Can't coerce from " ++ show t1 ++ " to " ++ show t2

implicitCoerce :: Monad m => PrimType -> PrimType -> Expr Name -> 
		  String -> Int -> m (Expr Name)
implicitCoerce x y e f l = checkCoerce (Prim x) (Prim y) e f l

{-
mkCtxt :: ParseResult -> Tags -> Types -> Context
mkCtxt [] tags syns = []
mkCtxt ((FB (x,xt,_)):xs) tags syns 
  = let xft = appsyn syns xt in
      ((x,xft):(mkCtxt xs))
mkCtxt (_:xs) tags syns = mkCtxt xs
-}

inferAll :: Monad m => Name -> -- Current module name
	    Context -> GContext -> Tags -> Types -> Fields ->
	    Bool -> [RawDecl] -> 
	        m (Program, Context, GContext, Tags, Types)
-- Finished
inferAll mod ctxt globs tags syns flds dump [] 
    = return ([], ctxt,globs,tags,syns)
-- C Includes
inferAll mod ctxt globs tags syns flds dump ((CInc str):xs) = 
  do (rest, ctxt', globs', tags',syns') <- inferAll mod ctxt globs tags syns flds dump xs
     return $ ((CInclude str):rest, ctxt', globs', tags',syns')
-- Import statements
inferAll mod ctxt globs tags syns flds dump ((Imp str):xs) = 
  do (rest, ctxt', globs', tags',syns') <- inferAll mod ctxt globs tags syns flds dump xs
     return $ ((Imported str):rest, ctxt', globs', tags',syns')
-- Import searches (ignoreable)
inferAll mod ctxt globs tags syns flds dump ((SearchImport str):xs) = 
  do inferAll mod ctxt globs tags syns flds dump xs
-- Link directives
inferAll mod ctxt globs tags syns flds dump ((Link str):xs) = 
  do (rest, ctxt', globs', tags',syns') <- inferAll mod ctxt globs tags syns flds dump xs
     return $ ((Linker str):rest, ctxt', globs', tags',syns')
-- Global variables
inferAll mod ctxt globs tags syns flds dump ((GlobDecl file line (n,t)):xs) =
  do st <- normalise file line syns t
     let gid = length globs
     (rest, ctxt', globs', tags',syns') <- 
	 inferAll mod ctxt ((n,(st,gid)):globs) tags syns flds dump xs
     return ((Glob (n,st,gid):rest), ctxt', ((n,(st,gid)):globs), tags', syns')
-- Function definitions
inferAll mod ctxt globs tags syns flds dump ((FB (file,line,x,xt,fopts,Defined xr) comm):xs) = 
  do let xftin = appsyn syns xt
     xft <- normalise file line syns xftin
     -- If it's been defined before, and it's a DefaultDef, just ignore it.
     if (DefaultDef `elem` fopts) && (lookup x ctxt /= Nothing)
       then inferAll mod ctxt globs tags syns flds dump xs
       else do
	    checkRpt file line x ctxt
	    (rest, ctxt', globs', tags', syns') <- 
		inferAll mod ((x,(xft,fopts)):ctxt) 
			 globs tags syns flds dump xs
	    let xrdec = insertdecl globs ((x,(xft,fopts)):ctxt') xr
	    (xrv,xeq) <- {-trace (show xrdec) $-} 
	         infer mod ((x,(xft,fopts)):ctxt') 
		       globs' tags' syns' flds xrdec xft
	    xeq' <- return $! 
		      (if dump then (trace (showeqns x xft xeq) xeq) else xeq)
	    phi' <- {- trace (show xrv) $ -} mkSubst xeq'
	    xfn <- substTerm phi' xrv
	    let xinft = subst phi' xft
	    {- trace (show xinft ++ "," ++ show xft) $ -} 
	    checkEq file line xinft xft
	    checkReturn file line x xft xfn
	    return {-$ trace ((show xrv)++"\n"++(show xfn)) $ -}
                ((FunBind (file,line,x,xft,fopts,Defined xfn) comm):rest, 
		 ((x,(xft,fopts)):ctxt'), globs', tags', syns')
 where mkfuntype f@(Fn _ _ _) = f
       mkfuntype x = (Fn [] [] x)
-- Imported function declaration
inferAll mod ctxt globs tags syns flds dump ((FB (file,line,x,xt,fopts,Unbound) comm):xs) =
  do let xft = mkfuntype xt
     (rest, ctxt', globs', tags', syns') <- inferAll mod ((x,(xft,fopts)):ctxt) globs tags syns flds dump xs
     return $ ((FunBind (file,line,x,xft,fopts,Unbound) comm):rest, (x,(xft,fopts)):ctxt', globs', tags', syns')
 where mkfuntype f@(Fn _ _ _) = f
       mkfuntype x = (Fn [] [] x)

-- Type synonyms
inferAll mod ctxt globs tags tydefs flds dump ((TSyn (file,line,n,ps,ty)):xs) = 
  do let def = Syn (map getname ps) ty
  -- All typevars in <ty> must be listed in <ps>
     checkTyVars file line ps ty
  -- If it's a duplicate, check it's identical to what we already had.
     newdefs <- checkDefined file line n tydefs def
     (rest, ctxt', globs', tags',syns') <- 
	 inferAll mod ctxt globs tags newdefs flds dump xs
     -- normalise to check for cycles
     foo <- normalise file line syns' (User n ps)
     return $ ((TySyn (n,(map getname ps),ty):rest), ctxt', globs', tags', syns')

  where getname (TyVar x) = x
--    fail $ file ++ ":" ++ show line ++ ":Type synonyms are broken"
--   inferAll ctxt globs tags ((n,ty):syns) flds dump xs
-- Data declarations
inferAll mod ctxt globs tags tydefs flds dump ((DDecl f l dopts n tys cons comm):xs) =
  do let newdefs = ((n,UserData tys):tydefs)
     -- Check it's not already defined
     checkData f l newdefs tys cons
     (fbs, newflds, codegen) <- getFuns ctxt f l cons newdefs 0
     checkRptFlds f l newflds n
     if (codegen && n `elem` (map fst tydefs)) 
	then fail $ f ++ ":" ++ show l ++ ": Data type " ++ 
	            showuser n ++ " already defined"
	else return ()
     let ctxt' = (map mkpub (gettys newdefs fbs)) ++ ctxt
     let tags' = (gettags fbs (length cons))++tags
     (rest,ctxt'', globs'', tags'', syns') <- inferAll mod ctxt' globs tags' newdefs 
					      (flds ++ newflds) dump xs
--     return $ (fbs++(DataDecl n tys cons):rest)
     return $ ([DataDecl f l dopts n tys cons comm]++fbs++rest, ctxt'', globs'', tags'', syns')
 where gettys s [] = []
       gettys s ((FunBind (f,l,n,ty,_,_) _):xs) = 
         ({-trace ((show n) ++ ":" ++ show ty) -}(n,appsyn s ty)):(gettys s xs)

       gettags [] l = []
       gettags ((FunBind (_,_,n,_,_,(DataCon t a _)) _):xs) l = (n,(t,l)):(gettags xs l)
{-
inferAll ctxt tags syns ((CDecl n ty i t ar):xs) =
  do (rest,ctxt',tags',syns') <- inferAll ctxt tags syns xs
     let tags'' = (n,(i,t)):tags
     return $ ((FunBind (n,ty,DataCon i ar)):rest, ctxt', tags'', syns')
-} 

-- Return the function declarations and field names from a constructor
-- declaration
-- Also returns whether this needs code generating or not.
getFuns :: Monad m => Context -> String -> Int -> [ConDecl] -> Types -> 
	   Int -> m (Program, Fields, Bool)
getFuns ctxt f l [] syns i = return ([], [], False)
getFuns ctxt f l ((Con n ty an codegen):xs) syns i
  = do (rest, flds, _) <- getFuns ctxt f l xs syns (i+1)
       if codegen 
	  then checkRpt f l n ctxt -- Check the name isn't used elsewhere
	  else return () -- Do nothing if it's from somewhere else
       return (((FunBind (f,l,n,(appsyn syns ty),[Inline,Pure,Public],DataCon i (getarglen ty) codegen) ""):rest),
	       (getFields (appsyn syns ty) an 0 i) ++ flds, codegen)
  where getarglen (Fn _ tys _) = length tys
	getarglen _ = 0

        getFields (Fn _ ty tn) as argn tag = gf' ty as argn tag tn
        gf' (x:xs) (a:as) argn tag tn = 
	    ((a,tn),(x,argn,tag)):(gf' xs as (argn+1) tag tn)
	gf' _ _ _ _ _ = []
		   
-- Check that field names are not repeated within a type definition
checkRptFlds :: Monad m => String -> Int ->
		Fields -> Name -> m ()
checkRptFlds f l [] n = return ()
checkRptFlds f l (((fld,t),_):xs) n
   | fld /= None && elem fld (map (fst.fst) xs)
       = fail $ f ++ ":" ++ show l ++ ":" ++
	    "Field " ++ showuser fld ++ " is duplicated in type " ++ showuser n
   | otherwise = checkRptFlds f l xs n


-- Check that a function which returns non-void does indeed return something
-- on all possible branches.
checkReturn :: Monad m => String -> Int ->
	       Name -> -- Function name
	       Type -> -- Function type
	       (Expr Name) -> 
	       m ()-- Function body
checkReturn _ _ _ (Fn _ _ (Prim Void)) _ 
    = return () -- Don't care, it's a void.
checkReturn file line n _ body 
    | containsReturn body = return ()
    | otherwise = fail $ file ++ ":" ++ show line ++ 
		  ":not all branches of " ++ showuser n ++ " return a value"

containsReturn :: Expr Name -> Bool
containsReturn (Return _) = True
containsReturn (Throw _) = True -- kind of the same thing!
containsReturn (Lambda _ e) = containsReturn e
containsReturn (Bind n t e1 e2) = containsReturn e2
containsReturn (Declare _ _ _ _ e) = containsReturn e
containsReturn (Seq e1 e2) = containsReturn e1 || containsReturn e2
containsReturn (While e1 e2) = containsReturn e2
containsReturn (DoWhile e1 e2) = containsReturn e1
containsReturn (For _ _ _ _ e) = containsReturn e
containsReturn (Case _ alts) = acr alts
   where acr [] = False
         acr [(Alt _ _ ts r)] = containsReturn r
	 acr ((Alt _ _ ts r):rs) = containsReturn r && acr rs
containsReturn (If a t e) = containsReturn t && containsReturn e
containsReturn (TryCatch tr ca _ f) = (containsReturn tr && containsReturn ca)
				      || containsReturn f
containsReturn _ = False

checkDefined :: Monad m => String -> Int -> Name -> Types -> 
		TypeInfo ->
		m Types
checkDefined f l n tys def
    = case lookup n tys of
         (Just x) -> if (x==def) then return tys
		       else fail $ f ++ ":" ++ show l ++ ":" ++
			    "Type "++showuser n++" already defined"
	 _ -> return ((n,def):tys)

checkTyVars :: Monad m => String -> Int -> [Type] -> Type -> m ()
checkTyVars f l ts t = ct t where
   ct (Fn ds ts t) = do mapM_ ct ts
			ct t
   ct (Array t) = ct t
   ct (User n ts) = mapM_ ct ts
   ct t@(TyVar _) | t `elem` ts = return ()
		  | otherwise = fail $ f ++ ":" ++ show l ++ ":" ++
				          show t ++ 
				         " not declared as a parameter"
   ct _ = return ()

checkData :: Monad m => String -> Int -> Types -- Synonyms
	     -> [Type] -- Parameters
	     -> [ConDecl] -- Constructor declarations
	     -> m () -- Just check it's okay
checkData f l syns tys [] = return ()
checkData f l syns tys (c:cs) = do checkConDecl f l syns tys c
				   checkData f l syns tys cs
				   
checkConDecl f l syns tys (Con n ty _ _) = allDeclared f l n tys (appsyn syns ty)

allDeclared f l n tys ty = {- trace (show ty) $ -} ad ty
  where ad (Fn _ ts t) = adM (t:ts)
	ad (Array t) = ad t
	ad (User n ts) = adM ts
	ad (TyVar x) | (TyVar x) `elem` tys = return ()
		     | otherwise = fail $ f ++ ":" ++ show l ++ ":" ++
				   "undeclared type variable " ++ showuser x
	ad _ = return ()

        adM [] = return ()
	adM (x:xs) = do ad x
			adM xs

-- Check whether the two types are equal (up to alpha conversion of type vars)
checkEq :: Monad m => String -> Int -> Type -> Type -> m ()
checkEq file line t1 t2 = do foo <- cg t1 t2 []
			     return ()
  where
     cg (TyVar x) (TyVar y) tvm = 
	 case (lookup x tvm) of
	   (Just z) -> if y==z then return tvm
		        else fail $ file ++ ":" ++ show line ++ ":" ++
			       "Inferred type less general than given type"
			       ++ " - Inferred " ++ show t1 ++ ", given " 
			       ++ show t2
	   Nothing -> return $ (x,y):tvm
     cg t (TyVar y) tvm = fail $ file ++ ":" ++ show line ++ ":" ++
			    "Inferred type less general than given type"
			    ++ " - Inferred " ++ show t1 ++ ", given " 
			    ++ show t2
     cg (Array x) (Array y) tvm = cg x y tvm
     cg (Fn ns ts t) (Fn ns' ts' t') tvm = do
          tvm' <- cg t t' tvm
	  cgl ts ts' tvm'
     cg (User n ts) (User n' ts') tvm = cgl ts ts' tvm
     cg _ _ tvm = return tvm

     cgl [] [] tvm = return tvm
     cgl (x:xs) (y:ys) tvm = do tvm' <- cg x y tvm
				cgl xs ys tvm'

-- Given a polymorphic type, and an instance, get what types the variables
-- refer to (as a substitution)
getVarTypes :: Monad m => Type -> Type -> m Subst
getVarTypes t1 t2 = do tvm <- cg t1 t2 []
		       (_,s) <- mkFSubst tvm
		       return s
  where
     mkFSubst [] = return ideq
     mkFSubst ((l,r):xs) = 
	 do xsubs <- mkFSubst xs
	    addEq "" 0 (TyVar l, r) xsubs

-- mkSubst $ map (\ (x,y) -> (TyVar x,y,"",0)) xs

     cg (TyVar x) y tvm = 
	 case (lookup x tvm) of
	   (Just z) -> if y==z then return tvm
		        else fail "Type error"
	   Nothing -> return $ (x,y):tvm
     cg t (TyVar y) tvm = fail "Type error"
     cg (Array x) (Array y) tvm = cg x y tvm
     cg (Fn ns ts t) (Fn ns' ts' t') tvm = do
          tvm' <- cg t t' tvm
	  cgl ts ts' tvm'
     cg (User n ts) (User n' ts') tvm | n == n' = cgl ts ts' tvm
				      | otherwise = fail "Type error"
     cg (Prim x) (Prim y) tvm | x == y = return tvm
			      | otherwise = fail "Type error"
     cg _ _ tvm = fail "Type error"

     cgl [] [] tvm = return tvm
     cgl (x:xs) (y:ys) tvm = do tvm' <- cg x y tvm
				cgl xs ys tvm'


checkRpt :: Monad m => String -> Int -> Name -> Context -> m ()
checkRpt f l x xs 
    = case lookup x xs of
          Just _ -> fail $ f ++ ":" ++ show l ++ ":" ++ showuser x ++ " already defined"
	  _ -> return ()

showtree :: Program -> String
showtree [] = ""
showtree ((FunBind (_,_,n,ty,fopts,Defined tr) _):xs) =
    showuser n ++ " :: " ++ show ty ++ " " ++ show fopts ++ "\n" ++ show tr ++ "\n\n" ++ showtree xs
showtree (_:xs) = showtree xs

showeqns :: Name -> Type -> Equation -> String
showeqns n ty (es,_) = showuser n ++ " :: " ++ show ty ++ ":\n" ++ se' es
  where se' [] = ""
	se' ((t1,t2,f,l):xs) = show t1 ++ " = " ++ show t2 ++ 
			       " ("++f++":"++show l++")\n" ++ se' xs
