-- Inlining optimisation
-- Replaces simple functions with their definitions in-place.

module Inliner(Inlinable, inliner, getInlinable, showInl) where

import Language
import Context

import Debug.Trace

-- Mapping from inlinable function names to their definitions
type Inlinable = NameContext (Expr Name, Type) -- type to help disambiguate

inliner :: Program -> Program
inliner xs = let inls = getInlinable xs in
                 doInlines inls xs

getInlinable :: Program -> Inlinable
getInlinable [] = empty
getInlinable ((FunBind (f,l,n,ty,ops,b) doc tyin):xs) =
    addInline n b ty (getInlinable xs)
getInlinable (_:xs) = getInlinable xs

addInline :: Name -> Binder (Expr Name) -> Type -> Inlinable -> Inlinable
addInline n (Defined def) ty inls 
    | inlinable 16 def = addToCtxt inls n (def,ty)
addInline n (ExtInlinable def) ty inls = addToCtxt inls n (def,ty)
addInline n _ _ inls = inls

doInlines :: Inlinable -> Program -> Program
doInlines inls [] = []
doInlines inls ((FunBind (f,l,n,ty,ops,Defined def) doc tyin):xs) =
      ((FunBind (f,l,n,ty,ops,Defined (inlineIn inls def)) doc tyin):
       (doInlines inls xs))
doInlines inls (x:xs) = x:(doInlines inls xs)

-- An expression is inlinable if it is smaller than the given
-- threshold.  Also, for simplicity for the moment, top level must be
-- a 'return', 'if' or function call.
-- Functions with 'var' args can't be inlined for the moment.

inlinable :: Int -> Expr Name -> Bool
inlinable i (Lambda as _ e) = (all (==Copy) as) && inlinable i e
inlinable i (Return e) = inl i e
inlinable i (If a t e) = inl (i-1) a && inl (i-1) t && inl (i-1) e
inlinable i (Apply f as) = inl (i-1) f && (and (map (inl (i-1)) as))
inlinable i (Annotation a e) = inlinable i e
-- inlinable i (Declare _ _ _ _ e) = inlinable i e
inlinable i Noop = True
inlinable i x = False

inl i (Infix op l r) = inl (i-1) l && inl (i-1) r
inl i (RealInfix op l r) = inl (i-1) l && inl (i-1) r
inl i (Unary op l) = inl (i-1) l
inl i (RealUnary op l) = inl (i-1) l
inl i (If a t e) = inl (i-1) a && inl (i-1) t && inl (i-1) e
inl i (Apply f as) = inl (i-1) f && (and (map (inl (i-1)) as))
inl i (Annotation a e) = inl i e
-- inl i (Declare _ _ _ _ e) = inl i e
inl i (GConst c) = True
inl i (GVar _) = True
inl i (Loc _) = True
inl i (Global _ _ _) = True
inl i (Coerce _ _ e) = inl i e
inl i (Seq x y) = inl (i-1) x && inl (i-1) y
inl i (Assign (AIndex _ j) e) = inl (i-1) j && inl (i-1) e
inl i (Assign _ e) = inl (i-1) e
inl i Noop = True
inl i e = inlinable i e

disambiguate :: String -> [(Expr Name, Type)] -> Maybe (Expr Name)
disambiguate _ [] = Nothing
disambiguate s ((e,t):es) | s == mangling t = Just e
disambiguate s (_:es) = disambiguate s es

inlineIn inls e = inlineApp inls (mapsubexpr (inlineIn inls) Metavar e)
inlineApp inls a@(Apply (Global n dis _) as)
    = let defs = lookupname None n inls in
          case disambiguate dis (map snd defs) of
              Just def -> inlApply a def as
              _ -> a
inlineApp inls x = x

inlApply :: Expr Name -> Expr Name -> [Expr Name] -> Expr Name
inlApply orig (Lambda _ _ e) args | all safe args = substIn args e
                                  | otherwise = orig
   where substIn args e = s' args (mapsubexpr (substIn args) Metavar e)
         s' args (Loc i) = args!!i
         s' args (Return x) = x
         s' args x = x
         -- No function calls, in case of weird side effects.
         -- FIXME: Decent side effect analysis will allow us to do this
         -- better.
         safe (Apply _ _) = False
         safe (Foreign _ _ _) = False
         safe (Annotation _ e) = safe e
         safe (Infix _ l r) = safe l && safe r
         safe (RealInfix _ l r) = safe l && safe r
         safe (Unary _ l) = safe l
         safe (RealUnary _ l) = safe l
         safe _ = True
inlApply orig (Annotation a e) args = inlApply orig e args
inlApply orig x args = trace ("Warning, weird thing to inline, please report:\n" ++ (show x)) $ orig

-- Show an inlinable definition for export
-- We keep this small... it's not quite the same as inlinable stuff in
-- general; in particular it can't refer to globals.
-- Function applications also won't work at the minute because inlined
-- definitions get rechecked and we need to be careful about what's public
-- and private. This version achieves the desired unboxing effect though
-- which is what's most important.

showInl :: Expr Name -> Maybe (String, [Name])
showInl (Annotation a e) = showInl e
showInl (Lambda as ns e) 
    | (all (==Copy) as) = do let args = map fst ns
                             tm <- si' False args e
                             return (tm, args)
   where si' ty args (Return x) = do x' <- si' True args x
                                     return $ "return " ++ x' ++ ";"
         si' ty args (Loc i) = return $ showuser (args!!i)
--          si' ty args (Apply f as) = do f' <- si' True args f
--                                        as' <- mapM (si' True args) as
--                                        return $ f' ++ "(" ++ showlist as ++ ")"
--               where showlist [] = ""
--                     showlist [x] = x
--                     showlist (x:xs) = x ++ "," ++ showlist xs
         si' ty args (Infix op x y) 
             = do x' <- si' True args x
                  y' <- si' True args y
                  return $ "(" ++ x' ++ show op ++ y' ++ ")"
         si' ty args (RealInfix op x y) 
             = do x' <- si' True args x
                  y' <- si' True args y
                  return $ "(" ++ x' ++ show op ++ y' ++ ")"
         si' ty args (Unary op x) 
             = do x' <- si' True args x
                  return $ "(" ++ show op ++ x' ++ ")"
         si' ty args (RealUnary op x) 
             = do x' <- si' True args x
                  return $ "(" ++ show op ++ x' ++ ")"
         si' ty args (GConst c) = return (show c)
         si' ty args (Coerce i o e) = do e' <- si' True args e
                                         return $ (show o) ++ "(" ++ e' ++ ")"
         si' False args (If a t e) 
             = do a' <- si' True args a
                  t' <- si' False args t
                  e' <- si' False args e
                  return $ "if (" ++ a' ++ ") {" ++
                            t' ++ "}" ++ " else {" ++ e' ++ "}"
         si' True args (If a t e) 
             = do a' <- si' True args a
                  t' <- si' True args t
                  e' <- si' True args e
                  return $ "(if (" ++ a' ++ ") " ++
                            t' ++ " else " ++ e' ++ ")"
         si' ty args Noop = return "pass;"
         si' ty args (Annotation a e) = si' ty args e
         si' _ _ x = Nothing
showInl _ = Nothing