{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module APIDocs where

import Language
import Debug.Trace
import System.Directory
import List
import Module

xmlise :: String -> String
xmlise [] = []
xmlise ('<':cs) = "&lt;"++(xmlise cs)
xmlise (c:cs) = c:(xmlise cs)

showfname (NS s n) = showfname n
showfname x = showuser x

data XMLDocType = XTySyn | XData | XFunc | XExcept
    deriving (Eq, Ord)
data XMLDoc = XMLDoc {
                      xmlty :: XMLDocType,
                      xmlfn :: String,
                      xml :: String
                     }
    deriving (Eq, Ord)
        
writeXMLDocs :: FilePath -> Program -> String -> String -> IO ()
writeXMLDocs fn prog inf mdocstr
  = do let str = "<module>\n" ++ 
                 "<modsummary>\n" ++ mdocstr ++ "</modsummary>\n" ++
                 concat (map xml (sort (mkXMLDocs prog inf))) ++ 
                 "</module>\n"
       writeFile fn str

mkXMLDocs :: Program -> String -> [XMLDoc]
mkXMLDocs [] f = []
mkXMLDocs ((FunBind (f,l,n,_,fopts,Defined def) comm ty):xs) inf 
       | inf == f && (elem Public fopts) =
	   (XMLDoc XFunc (cname n " "++(mangling ty))
                         (fundoc f l n ty def comm fopts)):(mkXMLDocs xs inf)
mkXMLDocs ((DataDecl f l dopts n tys cons comm):xs) inf 
       | inf == f && (elem DExport dopts) && ((elem DPublic dopts) || (elem DAbstract dopts))  =
	   (XMLDoc XData (show n) 
                         (datadoc f l n tys cons comm dopts)):(mkXMLDocs xs inf)
mkXMLDocs ((ExceptDecl f l n tys comm):xs) inf 
       | inf == f = 
           (XMLDoc XExcept (show n)
                           (exceptdoc f l n tys comm)):(mkXMLDocs xs inf)
mkXMLDocs ((TySyn (f, l, n, ps, ty, True)):xs) inf 
       | inf == f =
	   (XMLDoc XTySyn (show n) 
                          (tysyndoc f l n ps ty)):(mkXMLDocs xs inf)
mkXMLDocs (x:xs) inf = mkXMLDocs xs inf

data FunInfo = Fun Name -- Name
	           String -- Synopsis
		   String -- Comment
data DataInfo = Data Name -- Name
                     [String] -- Params
		     [String] -- Constructor synopses
		     String -- Comment
              | TypeSyn Name -- Name
                        [String] -- Params
                        String -- What it is


getName (TypeSyn n _ _) = (showfname n)
getName (Data n _ _ _) = (showfname n)

writeHTMLDocs :: FilePath -> Program -> String -> IO ()
writeHTMLDocs fn prog inf
  = do let (fi,di) = mkHTMLDocs prog inf
       let funinfo = sortBy (\ (Fun n1 _ _) (Fun n2 _ _) -> 
			     compare (showfname n1) (showfname n2)) fi
       let datainfo = sortBy (\ d1 d2 ->
			     compare (getName d1) (getName d2)) di
       let str = header inf ++ 
		 "<h2>Data Type Summary</h2>\n" ++
		 dsummary datainfo ++ 
		 "<h2>Function Summary</h2>\n" ++
		 fsummary funinfo ++ 
		 "<h2>Data Type Details</h2>\n" ++
		 ddetails datainfo ++ 
		 "<h2>Function Details</h2>\n" ++
		 fdetails funinfo ++
		 footer
       writeFile fn str
  where header inf = "<html><head><title>" ++ inf ++ " API reference" ++ 
		     "</title></head><body><h1>" ++
		     inf ++ " API reference</h1>\n"
	footer = "</body></html>"

getFirstLine cs = head (lines cs)

dsummary [] = ""
dsummary ((TypeSyn n ps syn):xs) =
    "type <a href=\"#"++showfname n++"\">" ++
    showuser n ++ "</a>" ++ showPlist ps ++ "<br>\n" ++
    dsummary xs
dsummary ((Data n ps _ comm):xs) =
    "data <a href=\"#"++showfname n++"\">" ++
    showuser n ++ "</a>" ++ showPlist ps ++ fl comm ++ "<br>\n" ++
    dsummary xs
  where fl "" = ""
	fl c = " - <em>" ++ getFirstLine c ++ "</em>"

showPlist [] = ""
showPlist xs = "&lt;"++sl' xs++">"
  where sl' [x] = x
	sl' (x:xs) = x ++ "," ++ sl' xs

ddetails [] = ""
ddetails ((TypeSyn n ps syn):xs) =
    "<hr><a name=\""++showfname n++"\">\n" ++
    "<h3>type " ++ showuser n ++ showPlist ps ++ "</h3>\n" ++
    "<p><strong> = " ++ xmlise syn ++ "</strong></p>\n" ++
    ddetails xs
ddetails ((Data n ps cons comm):xs) = 
    "<hr><a name=\""++showfname n++"\">\n" ++
    "<h3>data "++showuser n ++ showPlist ps ++ "</h3>\n" ++
    "<p><strong>Constructors</strong><ul>" ++ showcons cons ++ "</ul>\n" ++
    "<p>"++comm++"</p>\n" ++ ddetails xs
  where showcons [] = ""
	showcons (x:xs) = "<li>"++x++"</li>\n"++showcons xs

fsummary [] = ""
fsummary ((Fun n syn comm):xs) = syn ++ fl comm ++ "<br>\n" ++
				 fsummary xs
  where	fl "" = ""
	fl c = " - <em>" ++ getFirstLine c ++ "</em>"
		
fdetails [] = ""
fdetails ((Fun n syn comm):xs) = "<hr><a name=\""++showfname n++"\">\n" ++
				 "<h3>"++showuser n++"</h3>\n" ++
				 "<p><strong>" ++ syn++"</strong></p>\n" ++
				 "<p>"++comm++"</p>\n" ++
				 fdetails xs

mkHTMLDocs :: Program -> String -> ([FunInfo],[DataInfo])
mkHTMLDocs [] f = ([],[])
mkHTMLDocs ((FunBind (f,l,n,_,fopts,Defined def) comm ty):xs) inf 
       | inf == f && (elem Public fopts) =
	   let (fi,di) = mkHTMLDocs xs inf in
	   ((funhtml f l n ty def comm):fi,di)
mkHTMLDocs ((DataDecl f l dopts n tys cons comm):xs) inf 
       | inf == f =
	   let (fi,di) = mkHTMLDocs xs inf in
	   (fi,(datahtml f l n tys cons comm):di)
mkHTMLDocs ((TySyn (f,l,n,ps,ty,True)):xs) inf
       | inf == f =
	   let (fi,di) = mkHTMLDocs xs inf in
	   (fi,(tysynhtml f l n ps ty):di)
mkHTMLDocs (x:xs) inf = mkHTMLDocs xs inf

exceptdoc :: String -> Int -> Name -> [Type] -> String -> String
exceptdoc f l n tys comm = "<except><name>"++showuser n++"</name>\n" ++
                           showcomment comm++
			   "<location><file>"++f++"</file>"++
			   "<line>"++show l++"</line></location>\n"++
                           showexcargs tys++
                           "</except>"
    where showcomment "" = "" 
	  showcomment comm = "<comment>"++comm++"</comment>\n"
	  showexcargs [] = ""
          showexcargs (a:as) = "<arg><type>" ++
	                       xmlise (show a) ++ "</type></arg>" ++ 
                               showexcargs as


fundoc :: String -> Int -> Name -> Type -> Expr Name -> String -> [FOpt] -> String
fundoc f l n ty def comm fopts = "<fun><name>"++showfname n++"</name>\n"++
                                 showdepr (elem DeprecatedFn fopts)++
			         showcomment comm++
			         "<location><file>"++f++"</file>"++
			         "<line>"++show l++"</line></location>\n"++
			         "<return>"++xmlise (show (rettype ty))++
			         "</return>\n"++
			         listargs def (argtype ty) (deftype ty) ++ 
                                 --			   "<comment>"++show comm++"</comment>"++
			         "</fun>\n"
   where showcomment "" = ""
	 showcomment comm = "<comment>"++comm++"</comment>\n"
         showdepr False = ""
         showdepr True = "<deprecated/>\n"
  	 rettype (Fn _ _ ty) = ty
	 argtype (Fn _ args _) = args
	 deftype (Fn defs _ _) = defs

         listargs (Lambda ivs tys _) args defs = la' tys args defs (fArgs ivs tys)
         listargs (Annotation _ e) args defs = listargs e args defs
	 listargs _ _ _ = ""
	 la' [] [] [] _ = ""
	 la' ((n,_):tys) (ty:tys') (d:ds) varargs = 
	     "<arg passby=\"" ++
               (if elem n varargs then "var" else "copy") ++ 
             "\"><argname>" ++ showuser n ++ "</argname><type>" ++
	     xmlise (show ty) ++ "</type>" ++ showdef d ++"</arg>" ++ la' tys tys' ds varargs

	 showdef Nothing = ""
	 showdef (Just d) = "<defval>" ++ (stripat (showexp d)) ++ "</defval>"
	 stripat ('@':xs) = xs
	 stripat xs = xs


tysyndoc :: String -> Int -> Name -> [Name] -> Type -> String
tysyndoc f l n ps ty = "<tysyn><name>"++showuser n++"</name>\n"++
		       "<location><file>"++f++"</file>"++
		       "<line>"++show l++"</line></location>\n" ++
		       listparams ps ++ (if not (null ps) then "\n" else "") ++
                       "<syn>"++xmlise (show ty) ++ "</syn>" ++
                       "</tysyn>\n"
   where listparams [] = ""
	 listparams (p:ps) = "<param>"++showuser p++"</param>" ++ listparams ps

datadoc :: String -> Int -> Name -> [Type] -> [ConDecl] -> String -> [DOpt] -> String
datadoc f l n tys cons comm dopts = "<data><name>"++showuser n++"</name>\n"++
			            showcomment comm ++
			            "<location><file>"++f++"</file>"++
			            "<line>"++show l++"</line></location>\n"++
			            listparams tys ++ "\n" ++
			            listcons cons ++
                                    -- "<comment>"++show comm++"</comment>"++
			            "</data>\n"
   where showcomment "" = ""
	 showcomment comm = "<comment>"++comm++"</comment>\n"
         listparams [] = ""
	 listparams (p:ps) = "<param>"++show p++"</param>" ++ listparams ps

         listcons [] = ""
	 listcons (c:cs) = showcon c ++ listcons cs

	 argtype (Fn _ args _) = args
         showcon (Con n ty ns _) =
	     "<con><conname>"++showfname n++"</conname>"++
	     showconargs (argtype ty) ns (elem DAbstract dopts) ++
	     "</con>\n"

         showconargs _ _ True = ""
         showconargs [] [] False = ""
	 showconargs (a:as) (None:ns) False = 
	     "<arg><type>" ++
	     xmlise (show a) ++ "</type></arg>" ++ showconargs as ns False
	 showconargs (a:as) (n:ns) False = 
	     "<arg><argname>"++showuser n++"</argname><type>" ++
	     xmlise (show a) ++ "</type></arg>" ++ showconargs as ns False

funhtml :: String -> Int -> Name -> Type -> Expr Name -> String -> FunInfo
funhtml f l n ty def comm = Fun n synopsis comm
   where synopsis = xmlise (show (rettype ty)) ++ " " ++
		    "<a href=\"#"++showfname n++"\">" ++
		    showfname n ++ "</a>(" ++
		    showlist (synargs def (argtype ty) (defvals ty)) ++ ")"

         showlist [] = ""
	 showlist [x] = x
	 showlist (x:xs) = x ++ "," ++ showlist xs

  	 rettype (Fn _ _ ty) = ty
	 argtype (Fn _ args _) = args
	 defvals (Fn defs _ _) = defs

         synargs (Lambda ivs tys _) args defs = la' tys args defs (fArgs ivs tys)
	 synargs (Annotation _ e) a d = synargs e a d
	 synargs _ _ _ = []
	 la' [] [] [] _ = []
	 la' ((n,_):tys) (ty:tys') (def:defs) varargs = 
             ((if (elem n varargs) then "var " else "") ++
	      xmlise (show ty) ++ " " ++ showuser n ++showdef def):
	       (la' tys tys' defs varargs)
	 showdef Nothing = ""
	 showdef (Just d) = " = " ++ (stripat (showexp d))
	 stripat ('@':xs) = xs
	 stripat xs = xs

tysynhtml :: String -> Int -> Name -> [Name] -> Type -> DataInfo
tysynhtml f l n tys syn =
    TypeSyn n (listparams tys) (show syn)
   where listparams [] = []
	 listparams (p:ps) = (showuser p):(listparams ps)

datahtml :: String -> Int -> Name -> [Type] -> [ConDecl] -> String -> DataInfo
datahtml f l n tys cons comm = 
    Data n (listparams tys) (listcons cons) comm
   where listparams [] = []
	 listparams (p:ps) = (show p):(listparams ps)

         listcons [] = []
	 listcons ((Con n ty ns _):cs) = (synopsis n ty ns):(listcons cs)

         showlist [] = ""
	 showlist [x] = x
	 showlist (x:xs) = x ++ "," ++ showlist xs

         synopsis n ty ns 
		  | argtype ty == [] = showfname n
		  | otherwise = showfname n ++ "(" ++
				showlist (conargs (argtype ty) ns) ++ ")"

	 argtype (Fn _ args _) = args

	 conargs [] [] = []
	 conargs (a:as) (None:ns) = (xmlise (show a)):(conargs as ns)
	 conargs (a:as) (n:ns) = 
	     (xmlise (show a) ++ " " ++ showuser n):(conargs as ns)

fArgs (Var:vcs) ((n,ty):args) = n:(fArgs vcs args)
fArgs (_:vcs) (_:args) = fArgs vcs args
fArgs [] _ = []
fArgs _ [] = []

