/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2007 Kouhei Sutou <kou@cozmixng.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_browser.h"

#include <string.h>
#include <glib/gi18n.h>
#include "kazehakase.h"
#include "utils/gtk-utils.h"
#include "kz-embed.h"

#define DATA_KEY "KzPrefsBrowser::info"

enum
{
	ID_COL,
	TEXT_COL
};


static GtkWidget     *prefs_browser_create      (void);
static void           prefs_browser_response    (GtkWidget *widget,
						 gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	/* .path          = */ N_("/Browser"),
	/* .priority_hint = */ 0,
	/* .ui_level      = */ ~KZ_UI_LEVEL_BEGINNER,
	/* .create        = */ prefs_browser_create,
	/* .response      = */ prefs_browser_response,
};


typedef struct _KzPrefsBrowser
{
	GtkWidget     *main_vbox;
	GtkWidget     *layout_engine;
	gboolean       changed;
} KzPrefsBrowser;


static void
cb_changed (GtkWidget *widget, KzPrefsBrowser *prefsui)
{
	prefsui->changed = TRUE;
}

static void
prefs_browser_destroy (gpointer data)
{
	KzPrefsBrowser *prefsui = data;

	g_free(prefsui);
}

static void
set_layout_engine (GtkComboBox *combo)
{
	gchar *layout_engine_name;
	gint i = 0, active = 0;
	GList *node;
	GList *layout_engine_ids = NULL;
	GtkListStore *store;
	GtkTreeIter iter;

	layout_engine_name = KZ_CONF_GET_STR("Browser", "layout_engine");

	store = GTK_LIST_STORE(gtk_combo_box_get_model(combo));
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
			   ID_COL, NULL,
			   TEXT_COL, _("Default"),
			   -1);

	layout_engine_ids = kz_embed_engine_ids();
	for (node = layout_engine_ids; node; node = g_list_next(node))
	{
		gchar *engine_id = node->data;

		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
				   ID_COL, engine_id,
				   TEXT_COL, _(engine_id),
				   -1);
		i++;
		if (layout_engine_name &&
		    strcmp(engine_id, layout_engine_name) == 0)
			active = i;
		g_free(engine_id);
	}
	g_list_free(layout_engine_ids);

	if (layout_engine_name)
		g_free(layout_engine_name);

	gtk_combo_box_set_active(combo, active);
}

static void
create_title (KzPrefsBrowser *prefsui)
{
	GtkWidget *label;

	label = kz_prefs_ui_utils_create_title(_("Browser"));
	gtk_box_pack_start(GTK_BOX(prefsui->main_vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);
}

static GtkWidget *
create_layout_engine_area (KzPrefsBrowser *prefsui)
{
	GtkWidget *frame, *vbox;

	frame = gtk_frame_new(_("Layout engine"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(prefsui->main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	return vbox;
}

static void
create_layout_engine_selector (KzPrefsBrowser *prefsui,
			       GtkWidget *layout_engine_area)
{
	GtkWidget *hbox, *label, *combo;
	GtkCellRenderer *cell;
	GtkListStore *store;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(layout_engine_area), hbox);
	gtk_widget_show(hbox);
	label = gtk_label_new_with_mnemonic(_("Layout engine name"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);


	store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(store));
	prefsui->layout_engine = combo;
	g_object_unref(store);

	cell = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), cell, TRUE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo), cell,
				       "text", TEXT_COL,
				       NULL);

	set_layout_engine(GTK_COMBO_BOX(combo));
	gtk_widget_show(combo);
	gtk_box_pack_start (GTK_BOX(hbox), combo, TRUE, TRUE, 0);
	g_signal_connect(combo, "changed", G_CALLBACK(cb_changed), prefsui);
}


static GtkWidget *
prefs_browser_create (void)
{
	KzPrefsBrowser *prefsui = g_new0(KzPrefsBrowser, 1);
	GtkWidget *main_vbox, *layout_engine_area;

	main_vbox = gtk_vbox_new(FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full(G_OBJECT(main_vbox), DATA_KEY,
			       prefsui, prefs_browser_destroy);

	create_title(prefsui);
	layout_engine_area = create_layout_engine_area(prefsui);
	create_layout_engine_selector(prefsui, layout_engine_area);

	prefsui->changed = FALSE;

	return main_vbox;
}

static void
prefs_browser_response (GtkWidget *widget, gint response)
{
	KzPrefsBrowser *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
	{
		GtkComboBox *combo;

		if (!prefsui->changed)
			break;

		combo = GTK_COMBO_BOX(prefsui->layout_engine);
		if (gtk_combo_box_get_active(combo) == 0)
		{
			KZ_CONF_SET_STR("Browser", "layout_engine", "");
		}
		else
		{
			gchar *layout_engine = NULL;
			GtkTreeIter iter;
			GtkTreeModel *model;

			model = gtk_combo_box_get_model(combo);
			gtk_combo_box_get_active_iter(combo, &iter);
			gtk_tree_model_get(model, &iter,
					   ID_COL, &layout_engine,
					   -1);
			if (layout_engine)
			{
				KZ_CONF_SET_STR("Browser", "layout_engine",
						layout_engine);
				g_free(layout_engine);
			}
		}

		prefsui->changed = FALSE;

		break;
	}
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}

KzPrefsWinPageEntry *
prefs_browser_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
