/***************************************************************************
                          definition.cpp  -  description
                             -------------------
    begin                : Mit Nov 20 2002
    copyright            : (C) 2002 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "definition.h"
#include "sqltables.h"

// Qt includes
#include <qsqlquery.h>
#include <qregexp.h>

// KDE includes
#include <kapplication.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>

#define I2S(x) QString::number(x)

// a simple helper function
// that copies a file
bool filecopy( const char* src, const char* dest )
{
    FILE* s;
    FILE* d;
    int c;

    d = fopen(dest, "w");
    if( d == NULL )
        return false;

    s = fopen(src, "r");
    if( s == NULL ) {
        fclose( d );
        remove( dest );
        return false;
    }

    while(( c = getc( s )) != EOF )
        putc( c, d );

    fclose( s );
    fclose( d );
    return true;
}

class PrivateParser {
    public:
        PrivateParser( QString line );
        ~PrivateParser();

        QString getId() const { return label_def_id; }
        QString getProducer() const { return producer; }
        QString getType() const { return type; }
        measurements* getMeasurements() const { return m; };

    private:
        QString removeQuote( QString quote );

        QString label_def_id;
        QString producer;
        QString type;

        measurements* m;
};

PrivateParser::PrivateParser( QString line )
{
    line = line.stripWhiteSpace();
    int pos = line.find("(");
    line = line.mid( pos + 1, line.length() - pos - 1 );

    m = Definition::getCleanMeasurements();

    label_def_id = line.section( ",", 0, 0 ).stripWhiteSpace();
    producer = removeQuote( line.section( ",", 1, 1 ) );
    type = removeQuote( line.section( ",", 2, 2 ) );
    m->gap_top = line.section( ",", 4, 4 ).toDouble();
    m->gap_left = line.section( ",", 5, 5 ).toDouble();
    m->height = line.section( ",", 6, 6 ).toDouble();
    m->width = line.section( ",", 7, 7 ).toDouble();
    m->gap_v = line.section( ",", 8, 8 ).toDouble();
    m->gap_h = line.section( ",", 9, 9 ).toDouble();
    m->num_h = line.section( ",", 10, 10 ).toInt();
    m->num_v = line.section( ",", 11, 11 ).toInt();

    if( !m->num_h )
        m->num_h = 1;

    if( !m->num_v )
        m->num_v = 1;
}

PrivateParser::~PrivateParser()
{ }

QString PrivateParser::removeQuote( QString quote )
{
    quote = quote.stripWhiteSpace();

    if( quote.startsWith("'") )
        quote = quote.mid( 1, quote.length() - 1 );

    if( quote.endsWith("'") )
        quote = quote.left( quote.length() - 1 );

    return quote;
}


/***************************************************************************/

Definition::Definition()
{
    id = -1;
    m = getCleanMeasurements();    
}

Definition::Definition( int label_def_id )
{
    m = getCleanMeasurements();
    init( QString("%1").arg( label_def_id ) );
}

Definition::Definition( const QString & label_def_id )
{
    m = getCleanMeasurements();
    init( label_def_id );
}

Definition::Definition( const QString & producer, const QString & type )
{
    m = getCleanMeasurements();

    if( SqlTables::isConnected() ) {
        QSqlQuery query(
            "select label_no from " TABLE_LABEL_DEF " WHERE manufacture='" + producer + "' AND type='" + type + "'");
        while( query.next() )
            init( query.value( 0 ).toString() );
    } else {
        if(!openFile())
            return;

        QString s;
        while( file->readLine( s, 1000 ) != -1 ) {
            if( s.isEmpty() || s.left( 1 ) == "#" )
                continue;

            PrivateParser p( s );
            if( p.getProducer() ==producer && p.getType() == type ) {
                init( p.getId() );
                break;
            }
        }
    }

}

Definition::~Definition()
{
    delete m;
}

QFile* Definition::file = 0;
QStringList* Definition::listProducers = 0;
QMap<QString,QStringList> Definition::mapTypes;

void Definition::setId( const QString & label_def_id )
{
    init( label_def_id );    
}

void Definition::setId( int label_def_id )
{
    init( QString("%1").arg( label_def_id ) );
}

void Definition::init( const QString & label_def_id )
{
    if( SqlTables::isConnected() ) {
        QSqlQuery* query = new QSqlQuery(
            "select number_h, number_v, gap_left, gap_top, gap_v, gap_h, width, height, manufacture, type from " TABLE_LABEL_DEF
            " WHERE label_no = " + label_def_id );

        while( query->next() ) {
            m->num_h = query->value( 0 ).toInt();
            m->num_v = query->value( 1 ).toInt();
            m->gap_left = query->value( 2 ).toDouble();
            m->gap_top = query->value( 3 ).toDouble();
            m->gap_v = query->value( 4 ).toDouble();
            m->gap_h = query->value( 5 ).toDouble();
            m->width = query->value( 6 ).toDouble();
            m->height = query->value( 7 ).toDouble();
            producer = query->value( 8 ).toString();
            type = query->value( 9 ).toString();
        }
    } else {
        getFileMeasurements( label_def_id );
    }
    
    id = label_def_id.toInt();
    
}

measurements* Definition::getMeasurements() const
{
    return m;
}

measurements* Definition::getMeasurements( QPaintDevice* printer ) 
{
    /*
     * the returned measurements may be deleted
     */
    measurements* my = getCleanMeasurements();

    my->gap_left = mmToPixel( m->gap_left, printer, DpiX );
    my->gap_top = mmToPixel( m->gap_top, printer, DpiY );
    my->gap_v = mmToPixel( m->gap_v, printer, DpiY );
    my->gap_h = mmToPixel( m->gap_h, printer, DpiX );
    my->width = mmToPixel( m->width, printer, DpiX );
    my->height = mmToPixel( m->height, printer, DpiY );
    my->gap_v -= my->height;
    my->gap_h -= my->width;
    my->num_h = m->num_h;
    my->num_v = m->num_v;    

    return my;
}

void Definition::getFileMeasurements( const QString & label_def_id )
{
    if(!openFile()) {
        m = getCleanMeasurements();
        return;
    }

    QString s;
    while( file->readLine( s, 1000 ) != -1 ) {
        if( s.isEmpty() || s.left( 1 ) == "#" ) 
            continue;

        PrivateParser p( s );
        if( p.getId() != label_def_id )
            continue;

        producer = p.getProducer();
        type = p.getType();
        m = p.getMeasurements();
        break;
    }
}

measurements* Definition::getCleanMeasurements()
{
    measurements* c = new measurements;

    // init everything in case of
    // database connection fails.
    c->gap_h = 40.0;
    c->gap_left = 0.0;
    c->gap_top = 0.0;
    c->gap_v = 40.0;
    c->height = 40.0;
    c->num_h = 1;
    c->num_v = 1;
    c->width = 40.0;

    return c;
}

bool Definition::openFile()
{
    if( file ) {
        file->at( 0 );
        return true;        
    }

    QString f = locateLocal( "data", "kbarcode/labeldefinitions.sql" );
    if( !QFile::exists( f ) ) {
        KConfig* config = kapp->config();
        config->setGroup( "Definitions" );

        // copy file to new location
        QString fname = config->readEntry( "defpath", locate( "data", "kbarcode/labeldefinitions.sql" ) );
        if( !QFile::exists( fname ) || fname.isEmpty() ) 
            return ( showFileError() ? openFile() : false );

        if(!filecopy( (const char*)fname, (const char*)f ))
            return ( showFileError() ? openFile() : false );
    }

    file = new QFile( f );
    if( !file->open( IO_ReadOnly ) ) {
        delete file;
        file = 0;
        return ( showFileError() ? openFile() : false );
    }

    return true;
}

const QStringList Definition::getProducers()
{
    if( listProducers )
        return *listProducers;

    listProducers = new QStringList();

    if( SqlTables::isConnected() ) {
        QSqlQuery query("SELECT manufacture FROM " TABLE_LABEL_DEF " GROUP by manufacture;");
        while( query.next() )
            listProducers->append( query.value( 0 ).toString() );
    } else {
        if(!openFile() )
            return *listProducers;

        QString s;
        while( file->readLine( s, 1000 ) != -1 ) {
            if( s.isEmpty() || s.left( 1 ) == "#"  )
                continue;

            PrivateParser p( s );
            if( !listProducers->contains( p.getProducer() ) )
                listProducers->append( p.getProducer() );
        }
    }
    
    return *listProducers;
}

const QStringList Definition::getTypes( QString producer )
{
    if( mapTypes.contains( producer ) ) {
        return mapTypes[producer];
    }

    QStringList list;

    if( SqlTables::isConnected() ) {
        QSqlQuery query("SELECT type FROM " TABLE_LABEL_DEF " WHERE manufacture='" + producer + "'" );
        while( query.next() )
            if( !list.contains( query.value( 0 ).toString() ) )
                    list.append( query.value( 0 ).toString() );
    } else {
        if(!openFile())
            return list;

        QString s;
        while( file->readLine( s, 1000 ) != -1 ) {
            if( s.isEmpty() || s.left( 1 ) == "#" )
                continue;

            PrivateParser p( s );
            if( p.getProducer() == producer )
                if( !list.contains( p.getType() ) )
                    list.append( p.getType() );
        }
    }

    mapTypes.insert( producer, list );
    
    return list;
}

void Definition::updateProducer()
{
    if( listProducers ) {
        delete listProducers;
        listProducers = 0;
        mapTypes.clear();
    }
}

int Definition::write( measurements* c, QString type, QString producer )
{
    int r = -1;
    if( SqlTables::isConnected() )
        r = Definition::writeSQL( c, type, producer );
    else
        r = Definition::writeFile( c, type, producer ); 

    Definition::updateProducer();
    return r;
}

int Definition::writeFile( measurements* c, QString type, QString producer )
{
    if( !openFile() )
        return -1;

    QStringList data;
    bool datawritten = false;
    int index = 0;
    QString entry = ", '" +
                  producer + "', '" + type + "', 'C',"+ I2S(c->gap_top) +
                  ", " + I2S(c->gap_left) + ", " +
                  I2S(c->height) + ", " + I2S(c->width) + ", " +
                  I2S(c->gap_v) + ", " + I2S(c->gap_h) + ", " +
                  I2S(c->num_h) + ", " + I2S(c->num_v) + ", NULL, NULL )";
        
    QString s;
    while( file->readLine( s, 1000 ) != -1 ) {
        if( s.isEmpty() || s.left( 1 ) == "#" ) {
            data.append( s );
            continue;
        }
            
        PrivateParser p( s );
        if( p.getId().toInt() > index )
            index = p.getId().toInt();

        if( p.getType() == type && p.getProducer() == producer ) {
            // update an item already present in the list
            entry = entry.prepend( "INSERT INTO " TABLE_LABEL_DEF " VALUES (" + I2S(p.getId().toInt()) );
            data.append( entry );
            datawritten = true;
        } else
            data.append( s );
    }

    if( !datawritten ) {
        entry = entry.prepend( "INSERT INTO " TABLE_LABEL_DEF " VALUES (" + I2S(index+1) );
        data.append( entry );
    }
    
    file->close();
    if( !file->open( IO_WriteOnly ) ) {
        file->open( IO_ReadOnly );
        return -1;
    }

    QTextStream t( file );
    for( unsigned int i = 0; i < data.count(); i++ )
        t << data[i].replace( QRegExp("\\n"), "" ) << "\n";

    // get the file back to the normal stage
    file->close();
    file->open( IO_ReadOnly );

    return index + 1;
}

int Definition::writeSQL( measurements* c, QString type, QString producer )
{
    bool newitem = true;
    QSqlQuery q( "SELECT manufacture, type FROM " TABLE_LABEL_DEF );
    // TODO: use a more inteligent query using where=
    while( q.next() )
        if( q.value( 0 ) == producer &&
            q.value( 1 ) == type )
                newitem = false;

    if( newitem ) {
        QSqlQuery query(
              "INSERT INTO " TABLE_LABEL_DEF " (manufacture, type, gap_top, gap_left, "
              "width, height, gap_v, gap_h, number_h, number_v) VALUES ('" +
              producer + "', '" + type + "', '"+ I2S( c->gap_top ) +
              "', '" + I2S( c->gap_left ) + "', '" +
              I2S( c->width ) + "', '" + I2S( c->height ) + "', '" +
              I2S( c->gap_v ) + "', '" + I2S( c->gap_h ) + "', '" +
              I2S( c->num_h ) + "', '" + I2S( c->num_v ) + "')"
              );

        if(!query.isValid())
            qDebug("Query to insert values not valid!");
    } else {
        QSqlQuery query( "UPDATE " TABLE_LABEL_DEF " SET "
            "gap_top = " + I2S( c->gap_top ) + " ,gap_left = " + I2S( c->gap_left ) +
            " ,width = " + I2S( c->width ) + " ,height = " + I2S( c->height ) +
            " ,gap_v = " + I2S( c->gap_v ) + " ,gap_h = " + I2S( c->gap_h ) +
            " ,number_h = " + I2S( c->num_h ) + " ,number_v = " + I2S( c->num_v ) +
            " WHERE manufacture = '" + producer + "' AND"
            " type = '" + type + "'" );

        if(!query.isValid())
            qDebug("Query to update values not valid!\n" + query.lastQuery() );
    }

    QSqlQuery qi("SELECT label_no FROM " TABLE_LABEL_DEF " WHERE manufacture='" + producer + "' AND type='" + type + "'" );
    while( qi.next() )
        return qi.value( 0 ).toInt();

    return -1;
}


bool Definition::nodefmsg = true;
bool Definition::showFileError()
{
    if( nodefmsg ) {
        KMessageBox::information( 0,
            i18n("KBarcode is unable to find its label definitions."
                 "Please make sure that the file $KDEDIR/share/apps/kbarcode/labeldefinitions.sql "
                 "does exist. This file is part of the KBarcode distribution. "
                 "You will be prompted now to select the file containing the labeldefinitions."),
                 "", "NoDefinitionsFound" );

        QString f = KFileDialog::getOpenFileName( QString::null, QString::null, 0 );
        if( !f.isEmpty() && QFile::exists( f ) ) {
            KConfig* config = kapp->config();
            config->setGroup( "Definitions" );
            config->writeEntry( "defpath", f );
            config->sync();
        }
        nodefmsg = false;
        return openFile();
    } else
        qDebug("No label definitions found. Please install them.");

    return false;
}

int Definition::getClosest( const QString & producer, const QString & type )
{
    QStringList t = Definition::getTypes(producer); 
    for( unsigned int z = 0; z < t.count(); z++ )  {
        if( t[z] == type ) {
            Definition d( producer, type );
            return d.getId();
        }
    }

    return -1;
}
